/*
 * Copyright 2007-2008 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.chronos.core.schedule;

import java.util.Calendar;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

import org.seasar.chronos.core.TaskScheduleEntry;
import org.seasar.chronos.core.impl.TaskStateType;
import org.seasar.framework.log.Logger;

public class TaskScheduleEntryManager {

    public interface TaskScheduleEntryHanlder {
        public Object processTaskScheduleEntry(TaskScheduleEntry scheduleEntry);
    }

    @SuppressWarnings("unused")
    private static Logger log = Logger
            .getLogger(TaskScheduleEntryManager.class);

    private static TaskScheduleEntryManager instance;

    public static TaskScheduleEntryManager getInstance() {
        if (instance == null) {
            synchronized (TaskScheduleEntryManager.class) {
                if (instance == null) {
                    instance = new TaskScheduleEntryManager();
                }
            }
        }
        return instance;
    }

    private final ConcurrentHashMap<TaskStateType, CopyOnWriteArrayList<TaskScheduleEntry>> taskScheduleEntryMap = new ConcurrentHashMap<TaskStateType, CopyOnWriteArrayList<TaskScheduleEntry>>();

    private final CopyOnWriteArrayList<TaskScheduleEntry> allTaskList = new CopyOnWriteArrayList<TaskScheduleEntry>();

    private final ConcurrentHashMap<Class<?>, TaskScheduleEntry> taskScheduleEntryClassMap = new ConcurrentHashMap<Class<?>, TaskScheduleEntry>();

    private TaskScheduleEntryManager() {

    }

    public boolean addTaskScheduleEntry(final TaskStateType key,
            final TaskScheduleEntry taskScheduleEntry) {
        taskScheduleEntry.setTaskStateType(key);
        boolean result = getScheduleEntryList(key).add(taskScheduleEntry);
        if (key == TaskStateType.SCHEDULED) {
            result = result && allTaskList.addIfAbsent(taskScheduleEntry);
            final Class<?> taskComponentClass = taskScheduleEntry
                    .getTaskClass();
            if (!taskScheduleEntryClassMap.containsKey(taskComponentClass)) {
                taskScheduleEntryClassMap.put(taskComponentClass,
                        taskScheduleEntry);
            } else {
                result = false;
            }
        } else if (key == TaskStateType.UNSCHEDULED) {
            taskScheduleEntry.setUnScheduledDate(Calendar.getInstance()
                    .getTime());
        }
        return result;
    }

    public void allRemove(final TaskStateType key) {
        getScheduleEntryList(key).clear();
    }

    public boolean contains(final Object key) {
        if (key instanceof ScheduleEntry) {
            return allTaskList.contains(key);
        }
        return taskScheduleEntryClassMap.containsKey(key);
    }

    public boolean contains(final TaskStateType key,
            final TaskScheduleEntry taskContena) {
        final CopyOnWriteArrayList<TaskScheduleEntry> result = taskScheduleEntryMap
                .get(key);
        if (result != null) {
            return result.contains(taskContena);
        }
        return false;
    }

    public Object forEach(final TaskScheduleEntryHanlder handler) {
        for (final TaskScheduleEntry tse : allTaskList) {
            final Object result = handler.processTaskScheduleEntry(tse);
            if (result != null) {
                return result;
            }
        }
        return null;
    }

    public Object forEach(final TaskStateType key,
            final TaskScheduleEntryHanlder handler) {
        final CopyOnWriteArrayList<TaskScheduleEntry> scheduleEntryList = getScheduleEntryList(key);
        for (final TaskScheduleEntry tse : scheduleEntryList) {
            final Object result = handler.processTaskScheduleEntry(tse);
            if (result != null) {
                return result;
            }
        }
        return null;
    }

    private CopyOnWriteArrayList<TaskScheduleEntry> getScheduleEntryList(
            final TaskStateType key) {
        CopyOnWriteArrayList<TaskScheduleEntry> result = taskScheduleEntryMap
                .get(key);
        if (result == null) {
            result = new CopyOnWriteArrayList<TaskScheduleEntry>();
            taskScheduleEntryMap.put(key, result);
        }
        return result;
    }

    public TaskScheduleEntry getTaskScheduleEntry(final Object key) {
        return taskScheduleEntryClassMap.get(key);
    }

    public boolean removeTaskScheduleEntry(
            final TaskScheduleEntry taskScheduleEntry) {
        boolean result = allTaskList.remove(taskScheduleEntry);
        if (result) {
            for (final TaskStateType key : taskScheduleEntryMap.keySet()) {
                final CopyOnWriteArrayList<TaskScheduleEntry> taskScheduleEntryList = taskScheduleEntryMap
                        .get(key);
                if (taskScheduleEntryList != null) {
                    result = taskScheduleEntryList.remove(taskScheduleEntry);
                }
            }
            final Class<?> taskComponentClass = taskScheduleEntry
                    .getTaskClass();
            if (taskComponentClass != null) {
                taskScheduleEntryClassMap.remove(taskComponentClass);
            }
        }
        return result;
    }

    public boolean removeTaskScheduleEntry(final TaskStateType key,
            TaskScheduleEntry taskScheduleEntry) {
        boolean result = false;
        result = getScheduleEntryList(key).remove(taskScheduleEntry);
        if (key == TaskStateType.UNSCHEDULED) {
            result = result & allTaskList.remove(taskScheduleEntry);
            result = result
                    & taskScheduleEntryClassMap.remove(taskScheduleEntry
                            .getComponentDef().getComponentClass()) != null;
        }
        taskScheduleEntry = null;
        return result;
    }

    public int size() {
        return allTaskList.size();
    }

    public int size(final TaskStateType key) {
        final List<?> list = taskScheduleEntryMap.get(key);
        if (list == null) {
            return 0;
        }
        return list.size();
    }

}
