import java.awt.Desktop;
import java.awt.Graphics;
import java.io.File;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ResourceBundle;

import javax.swing.JApplet;
import javax.swing.JOptionPane;

public class FileLauncher extends JApplet {

    private static final int REDIRECT_DELAY_TIME = 2000;

    private static final long serialVersionUID = 1L;

    private String message;

    private transient ResourceBundle resources;

    private File file;

    @Override
    public void paint(final Graphics g) {
        g.drawString(message, 0, 20);
    }

    @Override
    public void init() {
        resources = ResourceBundle.getBundle("messages", getLocale());

        try {
            file = getFile();
        } catch (final Exception e) {
            message = e.getLocalizedMessage();
        }
    }

    @Override
    public void start() {
        if (file != null) {
            new Thread(new Runnable() {
                @Override
                public void run() {
                    final int ret = JOptionPane.showConfirmDialog(
                            getContentPane().getParent(),
                            getMsg("dialog.message"), getMsg("dialog.title"),
                            JOptionPane.YES_NO_OPTION,
                            JOptionPane.PLAIN_MESSAGE);
                    if (ret == JOptionPane.YES_OPTION) {
                        launch(file);
                    } else if (ret == JOptionPane.NO_OPTION) {
                        showReferrer(0);
                    }
                }
            }).start();
        }

        if (message == null) {
            message = "";
        }
    }

    private String getMsg(final String key) {
        return resources.getString(key);
    }

    private String getMsg(final String key, final Object... arguments) {
        return MessageFormat.format(resources.getString(key), arguments);
    }

    private File getFile() {

        final String uriParam = getRequestParameter("uri");
        if (isBlank(uriParam)) {
            message = getMsg("msg.no_uri");
            return null;
        }

        File targetFile = null;

        final String path = uriParam.replaceFirst("file:/+", "");
        final int pos1 = path.indexOf(':');
        final int pos2 = path.indexOf('/');
        if (pos1 > 0 && pos2 > 0 && pos1 < pos2) {
            // ex. c:/...
            targetFile = new File(path);
        } else {
            targetFile = new File(uriParam.replace("file:", ""));
        }

        if (!targetFile.exists()) {
            message = getMsg("msg.not_found", targetFile.getAbsolutePath());
            return null;
        }

        message = getMsg("msg.open_file", targetFile.getAbsolutePath());

        return targetFile;
    }

    private void launch(final File file) {
        final Desktop desktop = Desktop.getDesktop();
        try {
            desktop.open(file);
            message = getMsg("msg.opened_file", file.getAbsolutePath());
        } catch (final Exception e) {
            message = e.getMessage();
            repaint();
        }
        try {
            showReferrer(REDIRECT_DELAY_TIME);
        } catch (final Exception e) {
            message = e.getMessage();
            repaint();
        }
    }

    private void showReferrer(final long sleep) {
        try {
            Thread.sleep(sleep);
        } catch (final InterruptedException e) {
            // ignore
        }

        final String referrer = getRequestParameter("referrer");
        if (!isBlank(referrer)) {
            try {
                getAppletContext().showDocument(new URL(referrer), "_self");
            } catch (final Exception e) {
                message = e.getMessage();
                repaint();
            }
        }
    }

    private String getRequestParameter(final String key) {
        final String value = getParameter(key);
        if (isBlank(value)) {
            return null;
        }
        return value;
    }

    private boolean isBlank(final String str) {
        if (str == null || str.length() == 0) {
            return true;
        }
        for (int i = 0; i < str.length(); i++) {
            if (!Character.isWhitespace(str.charAt(i))) {
                return false;
            }
        }
        return true;
    }

}
