/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.service;

import java.beans.PersistenceDelegate;
import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.sql.Timestamp;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.transaction.UserTransaction;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.db.cbean.BrowserTypeCB;
import jp.sf.fess.db.cbean.CrawlingSessionCB;
import jp.sf.fess.db.cbean.CrawlingSessionInfoCB;
import jp.sf.fess.db.cbean.FileConfigToBrowserTypeMappingCB;
import jp.sf.fess.db.cbean.FileCrawlingConfigCB;
import jp.sf.fess.db.cbean.PathMappingCB;
import jp.sf.fess.db.cbean.WebAuthenticationCB;
import jp.sf.fess.db.cbean.WebConfigToBrowserTypeMappingCB;
import jp.sf.fess.db.cbean.WebCrawlingConfigCB;
import jp.sf.fess.db.exbhv.BrowserTypeBhv;
import jp.sf.fess.db.exbhv.CrawlingSessionBhv;
import jp.sf.fess.db.exbhv.CrawlingSessionInfoBhv;
import jp.sf.fess.db.exbhv.FileConfigToBrowserTypeMappingBhv;
import jp.sf.fess.db.exbhv.FileCrawlingConfigBhv;
import jp.sf.fess.db.exbhv.PathMappingBhv;
import jp.sf.fess.db.exbhv.WebAuthenticationBhv;
import jp.sf.fess.db.exbhv.WebConfigToBrowserTypeMappingBhv;
import jp.sf.fess.db.exbhv.WebCrawlingConfigBhv;
import jp.sf.fess.db.exentity.BrowserType;
import jp.sf.fess.db.exentity.CrawlingSession;
import jp.sf.fess.db.exentity.CrawlingSessionInfo;
import jp.sf.fess.db.exentity.FileConfigToBrowserTypeMapping;
import jp.sf.fess.db.exentity.FileCrawlingConfig;
import jp.sf.fess.db.exentity.PathMapping;
import jp.sf.fess.db.exentity.WebAuthentication;
import jp.sf.fess.db.exentity.WebConfigToBrowserTypeMapping;
import jp.sf.fess.db.exentity.WebCrawlingConfig;
import jp.sf.fess.util.FessProperties;

import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatabaseService implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(DatabaseService.class);

    private static int MAJOR_VERSION = 1;

    private static int MINOR_VERSION = 0;

    private static final String VERSION = MAJOR_VERSION + "." + MINOR_VERSION;

    private static final String VERSION_KEY = "version";

    private static final String CRAWLER_PROPERTIES_KEY = "crawlerProperties";

    private static final String BROWSER_TYPE_KEY = "browserType";

    private static final String CRAWLING_SESSION_KEY = "crawlingSession";

    private static final String CRAWLING_SESSION_INFO_KEY = "crawlingSessionInfo";

    private static final String FILE_CRAWLING_CONFIG_KEY = "fileCrawlingConfig";

    private static final String PATH_MAPPING_KEY = "pathMapping";

    private static final String WEB_CRAWLING_CONFIG_KEY = "webCrawlingConfig";

    private static final String FILE_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY = "fileConfigToBrowserTypeMapping";

    private static final String WEB_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY = "webConfigToBrowserTypeMapping";

    private static final String WEB_AUTHENTICATION_KEY = "webAuthentication";

    @Resource
    protected BrowserTypeBhv browserTypeBhv;

    @Resource
    protected CrawlingSessionBhv crawlingSessionBhv;

    @Resource
    protected CrawlingSessionInfoBhv crawlingSessionInfoBhv;

    @Resource
    protected FileCrawlingConfigBhv fileCrawlingConfigBhv;

    @Resource
    protected PathMappingBhv pathMappingBhv;

    @Resource
    protected WebCrawlingConfigBhv webCrawlingConfigBhv;

    @Resource
    protected FileConfigToBrowserTypeMappingBhv fileConfigToBrowserTypeMappingBhv;

    @Resource
    public WebAuthenticationBhv webAuthenticationBhv;

    @Resource
    protected WebConfigToBrowserTypeMappingBhv webConfigToBrowserTypeMappingBhv;

    @Resource
    protected FessProperties crawlerProperties;

    @Resource
    public UserTransaction userTransaction;

    public static class DataSet implements Serializable {
        private static final long serialVersionUID = 1L;

        private Map<String, Object> dataMap = new HashMap<String, Object>();

        public void put(String key, Object value) {
            dataMap.put(key, value);
        }
    }

    public void exportData(OutputStream out) {
        if (out == null) {
            throw new FessSystemException("The output stream is null.");
        }

        Map<String, Object> dataSet = new HashMap<String, Object>();

        // browserType
        dataSet.put(VERSION_KEY, VERSION);

        // browserType
        BrowserTypeCB browserTypeCB = new BrowserTypeCB();
        browserTypeCB.query().setDeletedBy_IsNull();
        dataSet.put(BROWSER_TYPE_KEY + "List", browserTypeBhv
                .selectList(browserTypeCB));
        // crawlingSession
        dataSet.put(CRAWLING_SESSION_KEY + "List", crawlingSessionBhv
                .selectList(new CrawlingSessionCB()));
        // crawlingSessionInfo
        dataSet.put(CRAWLING_SESSION_INFO_KEY + "List", crawlingSessionInfoBhv
                .selectList(new CrawlingSessionInfoCB()));
        // fileCrawlingConfig
        FileCrawlingConfigCB fileCrawlingConfigCB = new FileCrawlingConfigCB();
        fileCrawlingConfigCB.query().setDeletedBy_IsNull();
        dataSet.put(FILE_CRAWLING_CONFIG_KEY + "List", fileCrawlingConfigBhv
                .selectList(fileCrawlingConfigCB));
        // pathMapping
        PathMappingCB pathMappingCB = new PathMappingCB();
        pathMappingCB.query().setDeletedBy_IsNull();
        dataSet.put(PATH_MAPPING_KEY + "List", pathMappingBhv
                .selectList(pathMappingCB));
        // webCrawlingConfig
        WebCrawlingConfigCB webCrawlingConfigCB = new WebCrawlingConfigCB();
        webCrawlingConfigCB.query().setDeletedBy_IsNull();
        dataSet.put(WEB_CRAWLING_CONFIG_KEY + "List", webCrawlingConfigBhv
                .selectList(webCrawlingConfigCB));
        // fileConfigToBrowserTypeMapping
        dataSet.put(FILE_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY + "List",
                fileConfigToBrowserTypeMappingBhv
                        .selectList(new FileConfigToBrowserTypeMappingCB()));
        // webConfigToBrowserTypeMapping
        dataSet.put(WEB_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY + "List",
                webConfigToBrowserTypeMappingBhv
                        .selectList(new WebConfigToBrowserTypeMappingCB()));
        // webAuthentication
        dataSet.put(WEB_AUTHENTICATION_KEY + "List", webAuthenticationBhv
                .selectList(new WebAuthenticationCB()));

        // crawlerProperties
        Map<String, String> crawlerPropertyMap = new HashMap<String, String>();
        crawlerPropertyMap.put(Constants.CRON_EXPRESSION_PROPERTY,
                crawlerProperties.getProperty(
                        Constants.CRON_EXPRESSION_PROPERTY,
                        Constants.DEFAULT_CRON_EXPRESSION));
        crawlerPropertyMap.put(Constants.OPTIMIZE_PROPERTY, crawlerProperties
                .getProperty(Constants.OPTIMIZE_PROPERTY, "true"));
        crawlerPropertyMap.put(Constants.COMMIT_PROPERTY, crawlerProperties
                .getProperty(Constants.COMMIT_PROPERTY, "true"));
        crawlerPropertyMap.put(Constants.SERVER_ROTATION_PROPERTY,
                crawlerProperties.getProperty(
                        Constants.SERVER_ROTATION_PROPERTY, "true"));
        crawlerPropertyMap.put(Constants.DAY_FOR_CLEANUP_PROPERTY,
                crawlerProperties.getProperty(
                        Constants.DAY_FOR_CLEANUP_PROPERTY, "1"));
        crawlerPropertyMap.put(Constants.COMMIT_PER_COUNT_PROPERTY,
                crawlerProperties.getProperty(
                        Constants.COMMIT_PER_COUNT_PROPERTY, Long
                                .toString(Constants.DEFAULT_COMMIT_PER_COUNT)));
        crawlerPropertyMap.put(Constants.CRAWLING_THREAD_COUNT_PROPERTY,
                crawlerProperties.getProperty(
                        Constants.CRAWLING_THREAD_COUNT_PROPERTY, "5"));
        crawlerPropertyMap.put(Constants.MOBILE_TRANSCODER_PROPERTY,
                crawlerProperties.getProperty(
                        Constants.MOBILE_TRANSCODER_PROPERTY,
                        Constants.EMPTY_STRING));
        dataSet.put(CRAWLER_PROPERTIES_KEY, crawlerPropertyMap);

        try {
            XMLEncoder encoder = new XMLEncoder(new BufferedOutputStream(out));
            PersistenceDelegate pd = encoder.getPersistenceDelegate(Date.class);
            encoder.setPersistenceDelegate(Timestamp.class, pd);
            encoder.writeObject(dataSet);
            encoder.close();
        } catch (Exception e) {
            throw new FessSystemException("Could not write a data set.", e);
        }
    }

    @TransactionAttribute(TransactionAttributeType.NEVER)
    public void importData(InputStream in, boolean overwrite) {
        if (in == null) {
            throw new FessSystemException("The input stream is null.");
        }

        Map<String, Object> dataSet;
        try {
            XMLDecoder decoder = new XMLDecoder(new BufferedInputStream(in));
            dataSet = (Map<String, Object>) decoder.readObject();
        } catch (Exception e) {
            throw new FessSystemException("Could not read a data set.", e);
        }

        if (dataSet == null) {
            throw new FessSystemException("The object is null.");
        }

        // TODO check version

        new Thread(new DataImporter(dataSet, overwrite)).start();

    }

    protected class DataImporter implements Runnable {
        protected boolean overwrite;

        protected Map<String, Object> dataSet;

        protected DataImporter(Map<String, Object> dataSet, boolean overwrite) {
            this.dataSet = dataSet;
            this.overwrite = overwrite;
        }

        @Override
        public void run() {
            Map<String, Long> idMap = new HashMap<String, Long>();

            // browserType
            try {
                userTransaction.begin();

                List<BrowserType> browserTypeList = (List<BrowserType>) dataSet
                        .get(BROWSER_TYPE_KEY + "List");
                if (browserTypeList != null) {
                    for (BrowserType browserType : browserTypeList) {
                        Long id = browserType.getId();

                        BrowserTypeCB cb = new BrowserTypeCB();
                        cb.query().setValue_Equal(browserType.getValue());
                        cb.query().setDeletedBy_IsNull();
                        BrowserType entity = browserTypeBhv.selectEntity(cb);
                        browserType.setId(null);
                        if (entity == null) {
                            browserTypeBhv.insert(browserType);
                        } else {
                            // always overwrite
                            // if (overwrite) {
                            browserType.setVersionNo(null);
                            Beans.copy(browserType, entity).excludesNull()
                                    .execute();
                            browserType = entity;
                            browserTypeBhv.update(browserType);
                            // } else {
                            // browserTypeBhv.insert(browserType);
                            // }
                        }
                        idMap.put(BROWSER_TYPE_KEY + ":" + id.toString(),
                                browserType.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: " + BROWSER_TYPE_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // crawlingSession
            try {
                userTransaction.begin();

                List<CrawlingSession> crawlingSessionList = (List<CrawlingSession>) dataSet
                        .get(CRAWLING_SESSION_KEY + "List");
                if (crawlingSessionList != null) {
                    for (CrawlingSession crawlingSession : crawlingSessionList) {
                        Long id = crawlingSession.getId();

                        CrawlingSessionCB cb = new CrawlingSessionCB();
                        cb.query().setSessionId_Equal(
                                crawlingSession.getSessionId());
                        CrawlingSession entity = crawlingSessionBhv
                                .selectEntity(cb);
                        crawlingSession.setId(null);
                        if (entity == null) {
                            crawlingSessionBhv.insert(crawlingSession);
                        } else {
                            Beans.copy(crawlingSession, entity).excludesNull()
                                    .execute();
                            crawlingSession = entity;
                            crawlingSessionBhv.update(crawlingSession);
                            // delete info
                            CrawlingSessionInfoCB cb2 = new CrawlingSessionInfoCB();
                            cb2.query().setCrawlingSessionId_Equal(
                                    crawlingSession.getId());
                            crawlingSessionInfoBhv.queryDelete(cb2);
                        }
                        idMap.put(CRAWLING_SESSION_KEY + ":" + id.toString(),
                                crawlingSession.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: " + CRAWLING_SESSION_KEY,
                        e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // crawlingSessionInfo
            try {
                userTransaction.begin();

                List<CrawlingSessionInfo> crawlingSessionInfoList = (List<CrawlingSessionInfo>) dataSet
                        .get(CRAWLING_SESSION_INFO_KEY + "List");
                if (crawlingSessionInfoList != null) {
                    for (CrawlingSessionInfo crawlingSessionInfo : crawlingSessionInfoList) {
                        Long id = crawlingSessionInfo.getId();
                        // relations
                        crawlingSessionInfo.setCrawlingSessionId(idMap
                                .get(CRAWLING_SESSION_KEY
                                        + ":"
                                        + crawlingSessionInfo
                                                .getCrawlingSessionId()));
                        crawlingSessionInfo.setId(null);
                        crawlingSessionInfoBhv.insert(crawlingSessionInfo);
                        idMap.put(CRAWLING_SESSION_INFO_KEY + ":"
                                + id.toString(), crawlingSessionInfo.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: "
                        + CRAWLING_SESSION_INFO_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // fileCrawlingConfig
            try {
                userTransaction.begin();

                List<FileCrawlingConfig> fileCrawlingConfigList = (List<FileCrawlingConfig>) dataSet
                        .get(FILE_CRAWLING_CONFIG_KEY + "List");
                if (fileCrawlingConfigList != null) {
                    for (FileCrawlingConfig fileCrawlingConfig : fileCrawlingConfigList) {
                        Long id = fileCrawlingConfig.getId();

                        FileCrawlingConfigCB cb = new FileCrawlingConfigCB();
                        cb.query().setName_Equal(fileCrawlingConfig.getName());
                        cb.query().setDeletedBy_IsNull();
                        FileCrawlingConfig entity = fileCrawlingConfigBhv
                                .selectEntity(cb);
                        fileCrawlingConfig.setId(null);
                        if (entity == null) {
                            fileCrawlingConfigBhv.insert(fileCrawlingConfig);
                        } else {
                            if (overwrite) {
                                fileCrawlingConfig.setVersionNo(null);
                                Beans.copy(fileCrawlingConfig, entity)
                                        .excludesNull().execute();
                                fileCrawlingConfig = entity;
                                fileCrawlingConfigBhv
                                        .update(fileCrawlingConfig);
                            } else {
                                fileCrawlingConfigBhv
                                        .insert(fileCrawlingConfig);
                            }
                        }
                        idMap.put(FILE_CRAWLING_CONFIG_KEY + ":"
                                + id.toString(), fileCrawlingConfig.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: "
                        + FILE_CRAWLING_CONFIG_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // pathMapping
            try {
                userTransaction.begin();

                List<PathMapping> pathMappingList = (List<PathMapping>) dataSet
                        .get(PATH_MAPPING_KEY + "List");
                if (pathMappingList != null) {
                    for (PathMapping pathMapping : pathMappingList) {
                        Long id = pathMapping.getId();

                        PathMappingCB cb = new PathMappingCB();
                        cb.query().setRegex_Equal(pathMapping.getRegex());
                        cb.query().setDeletedBy_IsNull();
                        PathMapping entity = pathMappingBhv.selectEntity(cb);
                        pathMapping.setId(null);
                        if (entity == null) {
                            pathMappingBhv.insert(pathMapping);
                        } else {
                            if (overwrite) {
                                pathMapping.setVersionNo(null);
                                Beans.copy(pathMapping, entity).excludesNull()
                                        .execute();
                                pathMapping = entity;
                                pathMappingBhv.update(pathMapping);
                            } else {
                                pathMappingBhv.insert(pathMapping);
                            }
                        }
                        idMap.put(PATH_MAPPING_KEY + ":" + id.toString(),
                                pathMapping.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: " + PATH_MAPPING_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // webCrawlingConfig
            try {
                userTransaction.begin();

                List<WebCrawlingConfig> webCrawlingConfigList = (List<WebCrawlingConfig>) dataSet
                        .get(WEB_CRAWLING_CONFIG_KEY + "List");
                if (webCrawlingConfigList != null) {
                    for (WebCrawlingConfig webCrawlingConfig : webCrawlingConfigList) {
                        Long id = webCrawlingConfig.getId();

                        WebCrawlingConfigCB cb = new WebCrawlingConfigCB();
                        cb.query().setName_Equal(webCrawlingConfig.getName());
                        cb.query().setDeletedBy_IsNull();
                        WebCrawlingConfig entity = webCrawlingConfigBhv
                                .selectEntity(cb);
                        webCrawlingConfig.setId(null);
                        if (entity == null) {
                            webCrawlingConfigBhv.insert(webCrawlingConfig);
                        } else {
                            if (overwrite) {
                                webCrawlingConfig.setVersionNo(null);
                                Beans.copy(webCrawlingConfig, entity)
                                        .excludesNull().execute();
                                webCrawlingConfig = entity;
                                webCrawlingConfigBhv.update(webCrawlingConfig);
                            } else {
                                webCrawlingConfigBhv.insert(webCrawlingConfig);
                            }
                        }
                        idMap.put(
                                WEB_CRAWLING_CONFIG_KEY + ":" + id.toString(),
                                webCrawlingConfig.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: "
                        + WEB_CRAWLING_CONFIG_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // fileConfigToBrowserTypeMapping
            try {
                userTransaction.begin();

                List<FileConfigToBrowserTypeMapping> fileConfigToBrowserTypeMappingList = (List<FileConfigToBrowserTypeMapping>) dataSet
                        .get(FILE_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY + "List");
                if (fileConfigToBrowserTypeMappingList != null) {
                    for (FileConfigToBrowserTypeMapping fileConfigToBrowserTypeMapping : fileConfigToBrowserTypeMappingList) {
                        Long id = fileConfigToBrowserTypeMapping.getId();

                        Long browserTypeId = idMap.get(BROWSER_TYPE_KEY
                                + ":"
                                + fileConfigToBrowserTypeMapping
                                        .getBrowserTypeId());
                        Long fileConfigId = idMap.get(FILE_CRAWLING_CONFIG_KEY
                                + ":"
                                + fileConfigToBrowserTypeMapping
                                        .getFileConfigId());
                        if (browserTypeId == null || fileConfigId == null) {
                            // skip
                            continue;
                        }

                        FileConfigToBrowserTypeMappingCB cb = new FileConfigToBrowserTypeMappingCB();
                        cb.query().setBrowserTypeId_Equal(browserTypeId);
                        cb.query().setFileConfigId_Equal(fileConfigId);
                        FileConfigToBrowserTypeMapping entity = fileConfigToBrowserTypeMappingBhv
                                .selectEntity(cb);
                        if (entity == null) {
                            fileConfigToBrowserTypeMapping = new FileConfigToBrowserTypeMapping();
                            fileConfigToBrowserTypeMapping
                                    .setBrowserTypeId(browserTypeId);
                            fileConfigToBrowserTypeMapping
                                    .setFileConfigId(fileConfigId);
                            fileConfigToBrowserTypeMappingBhv
                                    .insert(fileConfigToBrowserTypeMapping);
                        }
                        idMap.put(FILE_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY + ":"
                                + id.toString(), fileConfigToBrowserTypeMapping
                                .getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: "
                        + FILE_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // webConfigToBrowserTypeMapping
            try {
                userTransaction.begin();

                List<WebConfigToBrowserTypeMapping> webConfigToBrowserTypeMappingList = (List<WebConfigToBrowserTypeMapping>) dataSet
                        .get(WEB_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY + "List");
                if (webConfigToBrowserTypeMappingList != null) {
                    for (WebConfigToBrowserTypeMapping webConfigToBrowserTypeMapping : webConfigToBrowserTypeMappingList) {
                        Long id = webConfigToBrowserTypeMapping.getId();

                        Long browserTypeId = idMap.get(BROWSER_TYPE_KEY
                                + ":"
                                + webConfigToBrowserTypeMapping
                                        .getBrowserTypeId());
                        Long webConfigId = idMap.get(WEB_CRAWLING_CONFIG_KEY
                                + ":"
                                + webConfigToBrowserTypeMapping
                                        .getWebConfigId());
                        if (browserTypeId == null || webConfigId == null) {
                            // skip
                            continue;
                        }

                        WebConfigToBrowserTypeMappingCB cb = new WebConfigToBrowserTypeMappingCB();
                        cb.query().setBrowserTypeId_Equal(browserTypeId);
                        cb.query().setWebConfigId_Equal(webConfigId);
                        WebConfigToBrowserTypeMapping entity = webConfigToBrowserTypeMappingBhv
                                .selectEntity(cb);
                        if (entity == null) {
                            webConfigToBrowserTypeMapping = new WebConfigToBrowserTypeMapping();
                            webConfigToBrowserTypeMapping
                                    .setBrowserTypeId(browserTypeId);
                            webConfigToBrowserTypeMapping
                                    .setWebConfigId(webConfigId);
                            webConfigToBrowserTypeMappingBhv
                                    .insert(webConfigToBrowserTypeMapping);
                        }
                        idMap.put(WEB_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY + ":"
                                + id.toString(), webConfigToBrowserTypeMapping
                                .getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: "
                        + WEB_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }
            // webAuthentication
            try {
                userTransaction.begin();

                List<WebAuthentication> webAuthenticationList = (List<WebAuthentication>) dataSet
                        .get(WEB_AUTHENTICATION_KEY + "List");
                if (webAuthenticationList != null) {
                    for (WebAuthentication webAuthentication : webAuthenticationList) {
                        Long id = webAuthentication.getId();

                        Long webConfigId = idMap.get(WEB_CRAWLING_CONFIG_KEY
                                + ":"
                                + webAuthentication.getWebCrawlingConfigId());
                        if (webConfigId == null) {
                            // skip
                            continue;
                        }

                        WebAuthenticationCB cb = new WebAuthenticationCB();
                        cb.query().setWebCrawlingConfigId_Equal(webConfigId);
                        WebAuthentication entity = webAuthenticationBhv
                                .selectEntity(cb);
                        webAuthentication.setId(null);
                        if (entity == null) {
                            webAuthenticationBhv.insert(webAuthentication);
                        } else {
                            if (overwrite) {
                                webAuthentication.setVersionNo(null);
                                Beans.copy(webAuthentication, entity)
                                        .excludesNull().execute();
                                webAuthentication = entity;
                                webAuthenticationBhv.update(webAuthentication);
                            } else {
                                webAuthenticationBhv.insert(webAuthentication);
                            }
                        }
                        idMap.put(WEB_AUTHENTICATION_KEY + ":" + id.toString(),
                                webAuthentication.getId());
                    }
                }
                userTransaction.commit();
            } catch (Exception e) {
                logger.warn("Failed to restore data: "
                        + WEB_CONFIG_TO_BROWSER_TYPE_MAPPING_KEY, e);
                try {
                    userTransaction.rollback();
                } catch (Exception e1) {
                    logger.warn("Failed to rollback data.", e1);
                }
            }

            // crawlerProperties
            try {
                Map<String, String> crawlerPropertyMap = (Map<String, String>) dataSet
                        .get(CRAWLER_PROPERTIES_KEY);
                String value;
                value = crawlerPropertyMap
                        .get(Constants.CRON_EXPRESSION_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(
                            Constants.CRON_EXPRESSION_PROPERTY, value);
                }
                value = crawlerPropertyMap.get(Constants.OPTIMIZE_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(Constants.OPTIMIZE_PROPERTY,
                            value);
                }
                value = crawlerPropertyMap.get(Constants.COMMIT_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(Constants.COMMIT_PROPERTY,
                            value);
                }
                value = crawlerPropertyMap
                        .get(Constants.SERVER_ROTATION_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(
                            Constants.SERVER_ROTATION_PROPERTY, value);
                }
                value = crawlerPropertyMap
                        .get(Constants.DAY_FOR_CLEANUP_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(
                            Constants.DAY_FOR_CLEANUP_PROPERTY, value);
                }
                value = crawlerPropertyMap
                        .get(Constants.COMMIT_PER_COUNT_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(
                            Constants.COMMIT_PER_COUNT_PROPERTY, value);
                }
                value = crawlerPropertyMap
                        .get(Constants.CRAWLING_THREAD_COUNT_PROPERTY);
                if (StringUtil.isNotBlank(value)) {
                    crawlerProperties.setProperty(
                            Constants.CRAWLING_THREAD_COUNT_PROPERTY, value);
                }
                value = crawlerPropertyMap
                        .get(Constants.MOBILE_TRANSCODER_PROPERTY);
                if (value != null) {
                    crawlerProperties.setProperty(
                            Constants.MOBILE_TRANSCODER_PROPERTY, value);
                }
                crawlerProperties.store();
            } catch (Exception e) {
                logger.warn(
                        "Failed to restore data: " + CRAWLER_PROPERTIES_KEY, e);
            }

        }
    }
}
