/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper;

import java.io.File;
import java.io.FilenameFilter;
import java.io.Serializable;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.util.RequestUtil;

public class SystemHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    private AtomicBoolean crawlProcessStatus = new AtomicBoolean();

    private AtomicBoolean replicationProcessStatus = new AtomicBoolean();

    private String sessionId;

    private boolean forceStop = false;

    private String logFilePath = System.getProperty("fess.log.file");

    private String adminRole = "fess";

    private List<String> authenticatedRoleList;

    public String getUsername() {
        String username = RequestUtil.getRequest().getRemoteUser();
        if (StringUtil.isBlank(username)) {
            username = "guest";
        }
        return username;
    }

    public Timestamp getCurrentTimestamp() {
        return new Timestamp(new Date().getTime());
    }

    public boolean readyCrawlProcess() {
        return crawlProcessStatus.compareAndSet(false, true);
    }

    public boolean isCrawlProcessRunning() {
        return crawlProcessStatus.get();
    }

    public void finishCrawlProcess() {
        crawlProcessStatus.set(false);
        sessionId = null;
    }

    public boolean readyReplicationProcess() {
        return replicationProcessStatus.compareAndSet(false, true);
    }

    public boolean isReplicationProcessRunning() {
        return replicationProcessStatus.get();
    }

    public void finishReplicationProcess() {
        replicationProcessStatus.set(false);
    }

    public String getSessionId() {
        return sessionId;
    }

    public void setSessionId(String sessionId) {
        this.sessionId = sessionId;
    }

    public File getSnapshotDir(String path) {
        File file = new File(path);
        if (!file.getName().contains("*")) {
            return file;
        }
        File targetDir = null;
        final String dirName = file.getName().replaceAll("\\.", "\\\\.")
                .replaceAll("\\*", ".*");
        for (File f : file.getParentFile().listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return name.matches(dirName);
            }
        })) {
            if (targetDir == null
                    || targetDir.lastModified() < f.lastModified()) {
                targetDir = f;
            }
        }
        if (targetDir != null) {
            return targetDir;
        }
        return file;
    }

    public boolean isForceStop() {
        return forceStop;
    }

    public void setForceStop(boolean forceStop) {
        this.forceStop = forceStop;
        if (forceStop) {
            SingletonS2Container.getComponent(WebIndexHelper.class)
                    .stopProcesses();
            SingletonS2Container.getComponent(FileSystemIndexHelper.class)
                    .stopProcesses();
        }
    }

    public String getLogFilePath() {
        return logFilePath;
    }

    public void setLogFilePath(String logFilePath) {
        this.logFilePath = logFilePath;
    }

    public String getAdminRole() {
        return adminRole;
    }

    public void setAdminRole(String adminRole) {
        this.adminRole = adminRole;
    }

    public List<String> getAuthenticatedRoleList() {
        return authenticatedRoleList;
    }

    public void setAuthenticatedRoleList(List<String> authenticatedRoleList) {
        this.authenticatedRoleList = authenticatedRoleList;
    }

    public void setAuthenticatedRoles(String roles) {
        if (StringUtil.isNotBlank(roles)) {
            String[] values = roles.split(",");
            authenticatedRoleList = new ArrayList<String>();
            for (String value : values) {
                if (StringUtil.isNotBlank(value)) {
                    authenticatedRoleList.add(value.trim());
                }
            }
        }
    }
}
