/* 
 * Copyright 2008 the Seasar Foundation and the Others.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not 
 * use this file except in compliance with the License. You may obtain a copy 
 * of the License at 
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0 
 *   
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT 
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the 
 * License for the specific language governing permissions and limitations 
 * under the License.
 * 
 */

package org.seasar.chronos.core.task.strategy.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.seasar.chronos.core.Scheduler;
import org.seasar.chronos.core.TaskScheduleEntry;
import org.seasar.chronos.core.TaskThreadPool;
import org.seasar.chronos.core.TaskTrigger;
import org.seasar.chronos.core.ThreadPoolType;
import org.seasar.chronos.core.annotation.task.Task;
import org.seasar.chronos.core.delegate.AsyncResult;
import org.seasar.chronos.core.delegate.MethodInvoker;
import org.seasar.chronos.core.executor.ExecutorServiceFactory;
import org.seasar.chronos.core.schedule.TaskScheduleEntryManager;
import org.seasar.chronos.core.task.TaskPropertyReader;
import org.seasar.chronos.core.task.TaskPropertyWriter;
import org.seasar.chronos.core.task.TaskType;
import org.seasar.chronos.core.task.Transition;
import org.seasar.chronos.core.task.handler.TaskExecuteHandler;
import org.seasar.chronos.core.task.handler.impl.execute.TaskGroupMethodExecuteHandlerImpl;
import org.seasar.chronos.core.task.handler.impl.execute.TaskMethodExecuteHandlerImpl;
import org.seasar.chronos.core.task.impl.TaskMethodManager;
import org.seasar.chronos.core.task.impl.TaskMethodMetaData;
import org.seasar.chronos.core.task.strategy.TaskExecuteStrategy;
import org.seasar.chronos.core.util.ObjectUtil;
import org.seasar.framework.beans.BeanDesc;
import org.seasar.framework.beans.PropertyDesc;
import org.seasar.framework.beans.factory.BeanDescFactory;
import org.seasar.framework.container.ComponentDef;
import org.seasar.framework.container.S2Container;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.framework.container.hotdeploy.HotdeployUtil;
import org.seasar.framework.exception.IORuntimeException;
import org.seasar.framework.log.Logger;
import org.seasar.framework.util.SerializeUtil;
import org.seasar.framework.util.tiger.CollectionsUtil;
import org.seasar.framework.util.tiger.ReflectionUtil;

public class TaskExecuteStrategyImpl implements TaskExecuteStrategy {

    @SuppressWarnings("unused")
    private static Logger log = Logger.getLogger(TaskExecuteStrategyImpl.class);

    private static final String METHOD_PREFIX_NAME_DO = "do";

    private static final String METHOD_NAME_INITIALIZE = "initialize";

    private static final String METHOD_NAME_DESTROY = "destroy";

    private static final String[] METHOD_NAME_START = { "start", "begin" };

    private static final String[] METHOD_NAME_END = { "end", "finish" };

    private static final String METHOD_NAME_DEFAULT_TASK_NAME = "execute";

    private static final String METHOD_NAME_DEFAULT_TASK_METHOD_NAME = "doExecute";

    private static final ThreadPoolType DEFAULT_THREADPOOL_TYPE = ThreadPoolType.CACHED;

    private static final int DEFAULT_THREAD_POOLSIZE = 1;

    private static ConcurrentHashMap<TaskThreadPool, ExecutorService> threadPoolExecutorServiceMap = CollectionsUtil
            .newConcurrentHashMap();

    private S2Container s2Container;

    private Object task;

    private Class<?> taskClass;

    private BeanDesc beanDesc;

    private MethodInvoker taskMethodInvoker;

    private TaskMethodManager taskMethodManager;

    private TaskExecuteHandler taskMethodExecuteHandler;

    private TaskExecuteHandler taskGroupMethodExecuteHandler;

    private TaskPropertyReader taskPropertyReader;

    private TaskPropertyWriter taskPropertyWriter;

    private Object getterSignal;

    private Scheduler scheduler;

    private long taskId = 0;

    private boolean prepared = false;

    private ExecutorServiceFactory executorServiceFactory;

    private ComponentDef componentDef;

    private boolean hotdeployStart;

    private boolean hotdeployDisable;

    public TaskExecuteStrategyImpl() {

    }

    public boolean await(final long time, final TimeUnit timeUnit)
            throws InterruptedException {
        return taskMethodInvoker.awaitInvokes(time, timeUnit);
    }

    public boolean cancel() {
        taskMethodInvoker.cancelInvokes();
        return true;
    }

    public boolean checkMoveAnotherTask(final String nextTaskName) {
        final TaskScheduleEntryManager tcsm = TaskScheduleEntryManager
                .getInstance();
        final Object result = tcsm
                .forEach(new TaskScheduleEntryManager.TaskScheduleEntryHanlder() {
                    public Object processTaskScheduleEntry(
                            final TaskScheduleEntry taskScheduleEntry) {
                        final Class<?> clazz = taskScheduleEntry.getTaskClass();
                        final Task task = clazz.getAnnotation(Task.class);
                        if (task == null) {
                            return null;
                        }
                        final String taskName = task.name();
                        if (nextTaskName.equals(taskName)) {
                            return new Object();
                        }
                        return null;
                    }
                });
        if (result != null) {
            return true;
        }
        return false;
    }

    public void initialize() throws InterruptedException {
        if (taskMethodInvoker.hasMethod(METHOD_NAME_INITIALIZE)) {
            taskMethodInvoker.beginInvoke(METHOD_NAME_INITIALIZE);
        }
    }

    public void destroy() throws InterruptedException {
        if (taskMethodInvoker.hasMethod(METHOD_NAME_DESTROY)) {
            final AsyncResult ar = taskMethodInvoker
                    .beginInvoke(METHOD_NAME_DESTROY);
            taskMethodInvoker.endInvoke(ar);
        }
    }

    public String end() throws InterruptedException {
        String nextTask = null;
        for (final String methodName : METHOD_NAME_END) {
            if (taskMethodInvoker.hasMethod(methodName)) {
                final AsyncResult ar = taskMethodInvoker
                        .beginInvoke(methodName);
                taskMethodInvoker.endInvoke(ar);
                final TaskMethodMetaData md = new TaskMethodMetaData(beanDesc,
                        methodName);
                nextTask = md.getNextTask();
                break;
            }
        }
        setExecuted(false);
        notifyGetterSignal();
        return nextTask;
    }

    public void execute(final String startTaskName) throws InterruptedException {
        TaskType type = isGroupMethod(startTaskName) ? TaskType.JOBGROUP
                : TaskType.JOB;
        String nextTaskName = startTaskName;
        while (true) {
            final TaskExecuteHandler teh = getTaskExecuteHandler(type);
            final Transition transition = handleRequest(teh, nextTaskName);
            notifyGetterSignal();
            if (transition == null || transition.isProcessResult()) {
                break;
            }
            type = type == TaskType.JOB ? TaskType.JOBGROUP : TaskType.JOB;
            nextTaskName = transition.getNextTaskName();
        }
    }

    public String getDescription() {
        return taskPropertyReader.getDescription(null);
    }

    public Scheduler getScheduler() {
        return scheduler;
    }

    public Object getTask() {
        return task;
    }

    public Class<?> getTaskClass() {
        return taskClass;
    }

    public long getTaskId() {
        taskId = taskPropertyReader.getTaskId(0L);
        if (taskId == 0) {
            taskId = ObjectUtil.generateObjectId();
        }
        return taskId;
    }

    public String getTaskName() {
        return taskPropertyReader.getTaskName(null);
    }

    public TaskPropertyReader getTaskPropertyReader() {
        return taskPropertyReader;
    }

    public TaskPropertyWriter getTaskPropertyWriter() {
        return taskPropertyWriter;
    }

    public TaskThreadPool getThreadPool() {
        return taskPropertyReader.getThreadPool(null);
    }

    public int getThreadPoolSize() {
        return this.getThreadPoolSize(task);
    }

    public ThreadPoolType getThreadPoolType() {
        return this.getThreadPoolType(task);
    }

    public TaskTrigger getTrigger() {
        return taskPropertyReader.getTrigger(null);
    }

    public synchronized void hotdeployStart() {
        if (!hotdeployDisable && HotdeployUtil.isHotdeploy()) {
            final String name = componentDef.getComponentClass().getName();
            ReflectionUtil.forNameNoException(name);
            HotdeployUtil.start();
            hotdeployStart = true;
        }
    }

    public synchronized void hotdeployStop() {
        if (!hotdeployDisable && HotdeployUtil.isHotdeploy()) {
            if (hotdeployStart) {
                HotdeployUtil.stop();
                hotdeployStart = false;
            }
        }
    }

    public String start() throws InterruptedException {
        setExecuted(true);
        for (final String methodName : METHOD_NAME_START) {
            if (taskMethodInvoker.hasMethod(methodName)) {
                final AsyncResult ar = taskMethodInvoker
                        .beginInvoke(methodName);
                taskMethodInvoker.endInvoke(ar);
                final TaskMethodMetaData md = new TaskMethodMetaData(beanDesc,
                        methodName);
                notifyGetterSignal();
                final String nextTaskName = md.getNextTask();
                if (nextTaskName == null
                        && taskMethodInvoker
                                .hasMethod(METHOD_NAME_DEFAULT_TASK_METHOD_NAME)) {
                    return METHOD_NAME_DEFAULT_TASK_NAME;
                }
                return nextTaskName;
            }
        }
        if (taskMethodInvoker.hasMethod(METHOD_NAME_DEFAULT_TASK_METHOD_NAME)) {
            return METHOD_NAME_DEFAULT_TASK_NAME;
        }
        return null;
    }

    public boolean isEndTask() {
        return taskPropertyReader.isEndTask(false);
    }

    public boolean isExecuted() {
        return taskPropertyReader.isExecuted(false);
    }

    public boolean isPrepared() {
        return prepared;
    }

    public boolean isReScheduleTask() {
        return taskPropertyReader.isReScheduleTask(false);
    }

    public boolean isShutdownTask() {
        return taskPropertyReader.isShutdownTask(false);
    }

    public boolean isStartTask() {
        return taskPropertyReader.isStartTask(false);
    }

    public void load() {
        FileInputStream fis = null;
        try {
            final File targetFile = new File("C:\\temp\\", getTaskClass()
                    .getCanonicalName());
            if (targetFile.exists()) {
                fis = new FileInputStream(targetFile);
                final int size = fis.read();
                final byte[] objectArray = new byte[size];
                fis.read(objectArray);
                task = SerializeUtil.fromBinaryToObject(objectArray);
            }
        } catch (final IOException e) {
            throw new IORuntimeException(e);
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (final IOException e) {
                    throw new IORuntimeException(e);
                }
            }
        }

        final Map<String, Object> taskProperties = new HashMap<String, Object>();
        final int size = beanDesc.getPropertyDescSize();
        for (int i = 0; i < size; i++) {
            final PropertyDesc pd = beanDesc.getPropertyDesc(i);
            final Object value = pd.getValue(task);
            taskProperties.put(pd.getPropertyName(), value);
        }
    }

    public void prepare() {

        task = componentDef.getComponent();
        taskClass = componentDef.getComponentClass();

        taskMethodExecuteHandler = createTaskMethodExecuteHandler();
        taskGroupMethodExecuteHandler = createTaskGroupMethodExecuteHandler(taskMethodExecuteHandler);

        taskMethodExecuteHandler.setTaskExecuteStrategy(this);
        taskGroupMethodExecuteHandler.setTaskExecuteStrategy(this);

        beanDesc = BeanDescFactory.getBeanDesc(taskClass);

        taskPropertyReader = (TaskPropertyReader) s2Container
                .getComponent(TaskPropertyReader.class);
        taskPropertyWriter = (TaskPropertyWriter) s2Container
                .getComponent(TaskPropertyWriter.class);

        taskPropertyReader.setup(task, beanDesc);
        taskPropertyWriter.setup(task, beanDesc);

        taskMethodManager = new TaskMethodManager(taskClass,
                METHOD_PREFIX_NAME_DO);

        if (taskMethodInvoker == null) {
            final ExecutorService jobMethodExecutorService = getExecutorService();
            taskMethodInvoker = new MethodInvoker(jobMethodExecutorService,
                    task, beanDesc);
            taskMethodInvoker.setExecutorServiceFactory(executorServiceFactory);
        }
        prepared = true;
        save();
        load();
    }

    public void save() {

    }

    public void setComponentDef(final ComponentDef componentDef) {
        this.componentDef = componentDef;
    }

    public void setEndTask(final boolean endTask) {
        taskPropertyWriter.setEndTask(endTask);
    }

    public void setExecuted(final boolean executed) {
        taskPropertyWriter.setExecuted(executed);
    }

    public void setExecutorServiceFactory(
            final ExecutorServiceFactory executorServiceFactory) {
        this.executorServiceFactory = executorServiceFactory;
    }

    public void setGetterSignal(final Object getterSignal) {
        this.getterSignal = getterSignal;
    }

    public void setS2Container(final S2Container container) {
        s2Container = container;
    }

    public void setScheduler(final Scheduler scheduler) {
        this.scheduler = scheduler;
    }

    public void setShutdownTask(final boolean shutdownTask) {
        taskPropertyWriter.setShutdownTask(shutdownTask);
    }

    public void setStartTask(final boolean startTask) {
        taskPropertyWriter.setStartTask(startTask);
    }

    public void setTask(final Object task) {
        this.task = task;
    }

    public void setTaskClass(final Class<?> taskClass) {
        this.taskClass = taskClass;
    }

    public void setTaskId(final long taskId) {
        if (taskPropertyWriter.hasTaskId()) {
            taskPropertyWriter.setTaskId(taskId);
            return;
        }
        this.taskId = taskId;
    }

    @Binding(bindingType = BindingType.NONE)
    public void setThreadPool(final TaskThreadPool taskThreadPool) {
        taskPropertyWriter.setThreadPool(taskThreadPool);
    }

    @Binding(bindingType = BindingType.NONE)
    public void setTrigger(final TaskTrigger taskTrigger) {
        taskPropertyWriter.setTrigger(taskTrigger);
    }

    public void unprepare() {
        prepared = false;
        taskGroupMethodExecuteHandler = null;
    }

    public void waitOne() throws InterruptedException {
        taskMethodInvoker.waitInvokes();
    }

    private ExecutorService createJobMethodExecutorService(final Object target) {
        final ExecutorService result = executorServiceFactory.create(
                this.getThreadPoolType(target), this.getThreadPoolSize(target));
        return result;
    }

    /**
     * TaskThreadPoolに対応するExecutorServiceを返します．
     * 
     * @param taskThreadPool
     *            スレッドプール
     * @return
     */
    private ExecutorService getCacheExecutorsService(
            final TaskThreadPool taskThreadPool) {
        ExecutorService executorService = threadPoolExecutorServiceMap
                .get(taskThreadPool);
        if (executorService == null) {
            executorService = createJobMethodExecutorService(taskThreadPool);
            threadPoolExecutorServiceMap.put(taskThreadPool, executorService);
        }
        return executorService;
    }

    /**
     * ExecutorServiceを返します．
     * <p>
     * TaskにTaskThreadPoolが存在する場合はキャッシュから対応するExecutorServiceを返します．<br>
     * 存在しない場合はTaskのThreadTypeとThreadPoolSizeから作成して返します．<br>
     * 複数のタスク間でスレッドプールを共有したければTaskThreadPoolを利用すること．
     * </p>
     * 
     * @return
     */
    private ExecutorService getExecutorService() {
        final TaskThreadPool taskThreadPool = getThreadPool();
        ExecutorService jobMethodExecutorService = null;
        if (taskThreadPool == null) {
            jobMethodExecutorService = createJobMethodExecutorService(task);
        } else {
            jobMethodExecutorService = getCacheExecutorsService(taskThreadPool);
        }

        return jobMethodExecutorService;
    }

    private TaskExecuteHandler getTaskExecuteHandler(final TaskType type) {
        return type == TaskType.JOB ? taskMethodExecuteHandler
                : taskGroupMethodExecuteHandler;
    }

    private int getThreadPoolSize(final Object target) {
        return taskPropertyReader.getThreadPoolSize(DEFAULT_THREAD_POOLSIZE);
    }

    private ThreadPoolType getThreadPoolType(final Object target) {
        return taskPropertyReader.getThreadPoolType(DEFAULT_THREADPOOL_TYPE);
    }

    private Transition handleRequest(
            final TaskExecuteHandler taskExecuteHandler,
            final String startTaskName) throws InterruptedException {
        taskExecuteHandler.setTaskExecuteStrategy(this);
        taskExecuteHandler.setMethodInvoker(taskMethodInvoker);
        taskExecuteHandler.setMethodGroupMap(taskMethodManager);
        return taskExecuteHandler.handleRequest(startTaskName);
    }

    private boolean isGroupMethod(final String groupName) {
        return taskMethodManager.existGroup(groupName);
    }

    private void notifyGetterSignal() {
        synchronized (getterSignal) {
            getterSignal.notify();
        }
    }

    protected TaskExecuteHandler createTaskGroupMethodExecuteHandler(
            final TaskExecuteHandler taskMethdoExecuteHandler) {
        final TaskGroupMethodExecuteHandlerImpl result = new TaskGroupMethodExecuteHandlerImpl();
        result.setTaskMethodExecuteHandler(taskMethodExecuteHandler);
        return result;
    }

    protected TaskExecuteHandler createTaskMethodExecuteHandler() {
        return new TaskMethodExecuteHandlerImpl();
    }

    public Exception getException() {
        return taskPropertyReader.getException(null);
    }

    public void setException(final Exception exception) {
        taskPropertyWriter.setException(exception);
    }

    public boolean isHotdeployDisable() {
        return hotdeployDisable;
    }

    public void setHotdeployDisable(final boolean hotdeployDisable) {
        this.hotdeployDisable = hotdeployDisable;
    }

    public void catchException(final Exception exception) {
        if (beanDesc.hasMethod("catchException")) {
            beanDesc.invoke(task, "catchException", new Object[] { exception });
        }
    }

}
