/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Expanded the file access and supported two GestureStore. 
 * The function of the preference access deleted it.
 */

package ogakisoft.android.gesture.reform;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import ogakisoft.android.gestureime.NamedGesture;
import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;
import android.os.Environment;

/**
 * File Access Utility for gesture-data
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.1
 */
public class GestureLibrary {
	public static final File DataDir = new File(Environment
			.getExternalStorageDirectory().getAbsoluteFile(), "gestureime");
	public static final File[] DataFile = {
			new File(DataDir, "gesture_data.0"),
			new File(DataDir, "gesture_data.1"),
			new File(DataDir, "gesture_data.2"),
			new File(DataDir, "gesture_data.3"),
			new File(DataDir, "gesture_data.4"),
			new File(DataDir, "gesture_data.5") };
	public static final File[] OldDataFile = { new File(DataDir, "gesture_data"), };
	private static final int MAX_NUM_OF_PREDICT = 5;
	private static GestureLibrary Me;
	private static GestureStore[] mStore;
	public static final int MAX_NUM_OF_STORE = 6;
	private static final String TAG = "GestureLibrary";
	private static final int IO_BUFFER_SIZE = 32 * 1024; // 32K
	/** classified gesture-data, input for CreateGestureActivity */
	private static final File GESTURE_TEST_FILE = new File(
			GestureLibrary.DataDir, "gesture_test");
	public static final int GESTURE_WIDTH = 200;
	public static final int GESTURE_HEIGHT = 200;

	public static GestureLibrary getInstance() {
		if (null == Me) {
			Me = new GestureLibrary();
			mStore = new GestureStore[MAX_NUM_OF_STORE];
			for (int i = 0; i < MAX_NUM_OF_STORE; i++) {
				mStore[i] = new GestureStore(i);
			}
			if (!DataDir.exists() || !DataDir.isDirectory()) {
				if (!DataDir.mkdir()) {
					final File f = Environment.getExternalStorageDirectory();
					if (!f.canRead() || !f.canWrite()) {
						LOG.e(TAG, "getInstance: sdcard not ready");
					} else {
						LOG.e(TAG, "getInstance: can not mkdir {0}",
								DataDir.getAbsolutePath());
					}
				}
			} else {
				// if exist old data file, copy to new one.
				if (OldDataFile[0].exists()) {
					if (!DataFile[0].exists()
							|| DataFile[0].lastModified() < OldDataFile[0]
									.lastModified()) {
						try {
							Utils.copyFile(OldDataFile[0].getAbsolutePath(),
									DataFile[0].getAbsolutePath());
						} catch (IOException e) {
							LOG.e(TAG,
									"getInstance: can not copy from {0} to {1}",
									OldDataFile[0].getAbsolutePath().toString(),
									DataFile[0].getAbsolutePath().toString());
						}
					}
				}
			}
		}
		return Me;
	}

	private GestureLibrary() {
	}

	public void addGesture(String label, Gesture gesture) {
		mStore[0].addGesture(label, gesture);
	}

	public int existGesture(long id) {
		int result = -1;
		for (int i = 0; i < MAX_NUM_OF_STORE; i++) {
			final Gesture g = mStore[i].getGesture(id);
			if (null != g) {
				result = i;
				break;
			}
		}
		return result;
	}

	public NamedGesture getNamedGesture(long id) {
		NamedGesture result = null;
		for (int num = 0; num < MAX_NUM_OF_STORE; num++) {
			result = mStore[num].getNamedGesture(id);
			if (null != result) {
				break;
			}
		}
		LOG.d(TAG, "getNamedGesture: id={0,number,#}", id);
		if (null != result) {
			LOG.d(TAG, "getNamedGesture: label={0}", result.getName());
		}
		return result;
	}

	public Set<String> getGestureEntries(int num) {
		if (num < 0 || MAX_NUM_OF_STORE <= num) {
			return null;
		}
		Set<String> result = new HashSet<String>();
		Set<String> set;
		set = mStore[num].getGestureEntries();
		if (null != set) {
			for (String str : set) {
				result.add(str);
			}
		}
		return result;
	}

	public List<Gesture> getGestures(int num, String label) {
		if (num < 0 || MAX_NUM_OF_STORE <= num) {
			return null;
		}
		return mStore[num].getGestures(label);
	}

	public Learner getLearner(int num) {
		if (0 <= num && num < MAX_NUM_OF_STORE) {
			return mStore[num].getLearner();
		} else {
			return null;
		}
	}

	public boolean hasLoaded() {
		boolean result = true;
		for (int i = 0; i < MAX_NUM_OF_STORE; i++) {
			if (!mStore[i].hasLoaded() && !isEmpty(DataFile[i])) {
				result = false;
				break;
			}
		}
		return result;
	}

	private boolean isEmpty(File file) {
		boolean result = false;
		if (!file.exists()) {
			return true;
		}
		FileInputStream fi = null;
		try {
			fi = new FileInputStream(file);
			try {
				if (fi.read() == -1) {
					result = true;
				}
			} catch (IOException e) {
			}
		} catch (FileNotFoundException e) {
		} finally {
			if (null != fi) {
				try {
					fi.close();
				} catch (IOException e) {
				}
			}
		}
		// LOG.d(TAG, "isEmpty: result={0}, file={1}", result,
		// file.getAbsoluteFile());
		return result;
	}

	private void load(File file, GestureStore store) throws IOException {
		if (file.exists() && file.canRead()) {
			FileInputStream fis = null;
			try {
				fis = new FileInputStream(file);
				store.load(fis);
			} catch (FileNotFoundException e) {
				LOG.d(TAG, "load: File not found: {0}", file.getAbsoluteFile()
						.toString());
			} finally {
				if (null != fis) {
					fis.close();
				}
			}
		} else {
			LOG.e(TAG, "load: not exist or can not read: {0}",
					file.getAbsolutePath());
		}
	}

	public void loadGestures() {
		try {
			for (int i = 0; i < MAX_NUM_OF_STORE; i++) {
				load(DataFile[i], mStore[i]);
			}
		} catch (IOException e) {
			LOG.e(TAG, "loadGestures: {0}", e.getMessage());
		}
	}

	public List<String> predict(Gesture gesture, int width, int height) {
		if (null == gesture) {
			LOG.d(TAG, "predict: query-gesture is null");
			return new ArrayList<String>(0);
		}
		List<Prediction> allList = new ArrayList<Prediction>();
		List<Prediction> list;
		int count;
		List<GestureStroke> strokes = GestureUtilities.scaling(gesture, width,
				height, GestureLibrary.GESTURE_WIDTH,
				GestureLibrary.GESTURE_HEIGHT);
		Gesture query = new Gesture();
		query.setId(gesture.getId());
		query.setStrokes(strokes);
		for (int i = 0; i < MAX_NUM_OF_STORE; i++) {
			list = mStore[i].recognize(query);
			count = list.size();
			LOG.d(TAG, "result fileno={0,number,#}, count={1,number,#}", i,
					count);
			for (int j = 0; j < count; j++) {
				allList.add(list.get(j));
			}
		}
		// Sort by score
		Collections.sort(allList, new Comparator<Prediction>() {
			public int compare(Prediction object1, Prediction object2) {
				final double score1 = object1.score;
				final double score2 = object2.score;
				if (score1 > score2) {
					return -1;
				} else if (score1 < score2) {
					return 1;
				} else {
					return 0;
				}
			}
		});
		// omit duplicate
		// and get top 5(=MAX_NUM_OF_PREDICT) to result
		String label = null;
		String last_label = null;
		count = allList.size();
		int result_count = 0;
		List<String> result = new ArrayList<String>();
		for (int i = 0; i < count && result_count < MAX_NUM_OF_PREDICT; i++) {
			label = allList.get(i).label;
			if (null != last_label && last_label.equals(label)) {
				// omit
			} else {
				result.add(label);
				result_count++;
				LOG.d(TAG, "predict: {0}", label);
			}
			last_label = label;
		}
		return result;
	}

	public void removeGesture(String label, Gesture gesture) {
		final long id = gesture.getId();
		int num = 0;
		if ((num = existGesture(id)) >= 0) {
			mStore[num].removeGesture(label, gesture);
		}
	}

	private void save(File file, GestureStore store) throws IOException {
		final File parentFile = file.getParentFile();
		FileOutputStream fos = null;

		if (!store.hasChanged()) {
			LOG.d(TAG, "save: has not changed");
			return;
		}
		if (null != parentFile) {
			if (!parentFile.exists()) {
				if (!parentFile.mkdirs()) {
					LOG.e(TAG, "save: can't create parent dir: {0}",
							parentFile.getAbsolutePath());
					return;
				}
			}
		}
		try {
			// no inspection ResultOfMethodCallIgnored
			file.createNewFile();
			fos = new FileOutputStream(file);
			store.save(fos);
		} catch (FileNotFoundException e) {
			LOG.d(TAG, "save: file not found:{0}", file.getAbsoluteFile()
					.toString());
		} finally {
			if (null != fos) {
				fos.close();
			}
		}
	}

	public boolean saveGestures() {
		boolean result = false;
		try {
			for (int i = 0; i < MAX_NUM_OF_STORE; i++) {
				save(DataFile[i], mStore[i]);
			}
			result = true;
		} catch (IOException e) {
			LOG.e(TAG, "saveGestures: {0}", e.getMessage());
		}
		return result;
	}

	public boolean hasChanged(int num) {
		if (num < 0 || num <= MAX_NUM_OF_STORE) {
			return false;
		}
		return mStore[num].hasChanged();
	}

	/**
	 * Load one gesture data from a file, can CreateGestureActivity at any time
	 * 
	 * @return Gesture
	 */
	public static Gesture loadGesture() throws IOException {
		DataInputStream in = null;
		Gesture gesture = null;
		final int w;
		final int h;
		float x;
		float y;
		long timeStamp;
		int countP;
		List<GesturePoint> points;
		final int countS;
		final long id;
		try {
			in = new DataInputStream(new BufferedInputStream(
					new FileInputStream(GESTURE_TEST_FILE), IO_BUFFER_SIZE));
			gesture = new Gesture();
			// Read width
			w = in.readInt();
			// Read height
			h = in.readInt();
			// Read gesture-id
			id = in.readLong();
			gesture.setId(id);
			// Read number of strokes
			countS = in.readInt();
			// Read strokes
			for (int i = 0; i < countS; i++) {
				countP = in.readInt();
				points = new ArrayList<GesturePoint>(countP);
				for (int j = 0; j < countP; j++) {
					x = in.readFloat();
					y = in.readFloat();
					timeStamp = in.readLong();
					points.add(new GesturePoint(x, y, timeStamp));
				}
				gesture.addStroke(new GestureStroke(points));
			}
		} finally {
			if (null != in) {
				in.close();
			}
		}
		LOG.d(TAG, "loadGesture: {0,number,#},{1,number,#}", w, h);
		// LOG.d(TAG,
		// "loadGesture: {0,number,#},{1,number,#} -> {2,number,#},{3,number,#}",
		// w, h, width, height);
		// List<GestureStroke> strokes = GestureUtilities.scaling(gesture, w, h,
		// width, height);
		// gesture.setStrokes(strokes);
		return gesture;
	}

	/**
	 * Save one gesture data in a file, can CreateGestureActivity at any time
	 * 
	 * @param gesture
	 *            Gesture query-data (in GestureStore.recognize())
	 */
	public static void saveGesture(Gesture gesture, int width, int height)
			throws IOException {
		DataOutputStream out = null;
		float[] pts;
		long[] times;
		int countP;
		final List<GestureStroke> strokes;
		final int countS;
		if (null == gesture) {
			LOG.d(TAG, "saveGesture: gesture is null");
			return;
		}
		try {
			out = new DataOutputStream(new BufferedOutputStream(
					new FileOutputStream(GESTURE_TEST_FILE), IO_BUFFER_SIZE));

			strokes = GestureUtilities.scaling(gesture, width, height,
					GESTURE_WIDTH, GESTURE_HEIGHT);
			LOG.d(TAG,
					"saveGesture: {0,number,#},{1,number,#} -> {2,number,#},{3,number,#}",
					width, height, GESTURE_WIDTH, GESTURE_HEIGHT);
			countS = strokes.size();

			// Write width
			out.writeInt(GESTURE_WIDTH);
			// Write height
			out.writeInt(GESTURE_HEIGHT);
			// Write gesture ID
			out.writeLong(gesture.getId());
			// Write number of strokes
			out.writeInt(countS);
			// Write strokes
			for (int i = 0; i < countS; i++) {
				pts = strokes.get(i).points;
				times = strokes.get(i).timestamps;
				countP = strokes.get(i).points.length;
				// Write number of points
				out.writeInt(countP / 2);
				for (int j = 0; j < countP; j += 2) {
					// Write X
					out.writeFloat(pts[j]);
					// Write Y
					out.writeFloat(pts[j + 1]);
					// Write time-stamp
					out.writeLong(times[j / 2]);
				}
			}
			out.flush();
		} finally {
			if (null != out) {
				out.close();
			}
		}
	}
}
