/*
 * Copyright (C) 2011 OgakiSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ogakisoft.android.gestureime;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import ogakisoft.android.gesture.reform.GestureLibrary;
import ogakisoft.android.svm.SvmWrapper;
import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.DefaultHttpClient;

import android.os.AsyncTask;
import android.os.Handler;
import android.os.Message;

/**
 * Download and extracts a compressed file including the gesture data.
 * 
 * @version 1.0
 * @author noritoshi ogaki
 */
public class DownloadTask extends AsyncTask<String, HttpResponse, Integer> {
	/** LOG tag */
	private static final String TAG = "DownloadTask";
	/** zip-file filename */
	private static final String ZIP_FILE = "gestureime_data.zip";
	/** The state of the task result; does not yet */
	public static final int STATUS_NONE = 0;
	/** The state of the task result; finished normally */
	public static final int STATUS_SUCCESS = 1;
	/** The state of the task result; canceled */
	public static final int STATUS_CANCELLED = 2;
	/** The state of the task result; an error occurred */
	public static final int STATUS_ERROR = 3;
	/** The state of the task result */
	private int mStatus;
	/** for call-back at the end of task */
	private final Handler mHandler;
	/** file buffer size */
	private static final int IO_BUFFER_SIZE = 32 * 1024; // 32K

	/**
	 * Default Constructor for DownloadTask.
	 */
	public DownloadTask() {
		super();
		mHandler = null;
	}

	/**
	 * Constructor for DownloadTask.
	 * 
	 * @param handler
	 *            Handler
	 */
	public DownloadTask(Handler handler) {
		mHandler = handler;
	}

	/**
	 * Method doInBackground.
	 * 
	 * @param params
	 *            String[]
	 * @return Integer
	 */
	@Override
	protected Integer doInBackground(String... params) {
		final HttpGet request = new HttpGet(params[0]);
		final DefaultHttpClient httpClient = new DefaultHttpClient();
		HttpResponse response = null;
		if (isCancelled()) {
			return STATUS_CANCELLED;
		}
		mStatus = STATUS_NONE;
		try {
			if (null != request) {
				LOG.d(TAG, "doInBackground: URI={0}", params[0]);
				synchronized (httpClient) {
					response = httpClient.execute(request);
				}
				if (null != response
						&& response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
					publishProgress(response);
				} else {
					LOG.e(TAG, "doInBackground: status={0}", response
							.getStatusLine().getStatusCode());
					LOG.e(TAG, "doInBackground: reason={0}", response
							.getStatusLine().getReasonPhrase());
					mStatus = STATUS_ERROR;
					mHandler.sendMessageAtFrontOfQueue(Message.obtain(mHandler,
							mStatus));
				}
			}
		} catch (ClientProtocolException e) {
			e.printStackTrace();
			LOG.e(TAG, "doInBackground: {0}", e.getMessage());
			mStatus = STATUS_ERROR;
			mHandler.sendMessageAtFrontOfQueue(Message
					.obtain(mHandler, mStatus));
		} catch (IOException e) {
			e.printStackTrace();
			LOG.e(TAG, "doInBackground: {0}", e.getMessage());
			mStatus = STATUS_ERROR;
			mHandler.sendMessageAtFrontOfQueue(Message
					.obtain(mHandler, mStatus));
		}
		return mStatus;
	}

	/**
	 * Method onProgressUpdate.
	 * 
	 * @param values
	 *            HttpResponse[]
	 */
	@Override
	protected void onProgressUpdate(HttpResponse... values) {
		try {
			final HttpResponse response = values[0];
			final InputStream in = response.getEntity().getContent();
			saveZipFile(in);
		} catch (IllegalStateException e) {
			LOG.e(TAG, "onProgressUpdate: {0}", e.getMessage());
			mStatus = STATUS_ERROR;
		} catch (IOException e) {
			LOG.e(TAG, "onProgressUpdate: {0}", e.getMessage());
			mStatus = STATUS_ERROR;
		} finally {
			mHandler.sendMessageAtFrontOfQueue(Message
					.obtain(mHandler, mStatus));
		}
	}

	/**
	 * Method saveZipFile.
	 * 
	 * @param is
	 *            InputStream
	 */
	private void saveZipFile(InputStream is) {
		ByteArrayOutputStream ba = null;
		BufferedOutputStream out = null;
		final File dir = GestureLibrary.DataDir;
		final File file = new File(dir, ZIP_FILE);
		final byte[] buf = new byte[IO_BUFFER_SIZE];
		int count = 0;
		ZipInputStream zis = null;
		File parent = null;
		try {
			parent = file.getParentFile();
			if (null != parent && !parent.exists()) {
				if (!parent.mkdirs()) {
					mStatus = STATUS_ERROR;
					LOG.e(TAG, "saveZipFile: can't create parent dir: {0}",
							parent.getAbsolutePath());
					return;
				}
			}
			if (file.exists()) {
				if (!file.delete()) {
					mStatus = STATUS_ERROR;
					LOG.e(TAG, "saveZipFile: can't delete file: {0}",
							file.getAbsolutePath());
					return;
				}
			}
			out = new BufferedOutputStream(new FileOutputStream(file),
					IO_BUFFER_SIZE);
			ba = new ByteArrayOutputStream();
			for (count = is.read(buf); count != -1; count = is.read(buf)) {
				ba.write(buf, 0, count);
			}
			out.write(ba.toByteArray());

		} catch (IOException e) {
			mStatus = STATUS_ERROR;
			LOG.e(TAG, "saveZipFile: {0}", e.getMessage());
		} finally {
			if (null != ba) {
				try {
					ba.close();
				} catch (IOException e) {
					mStatus = STATUS_ERROR;
					LOG.e(TAG, "saveZipFile: {0}", e.getMessage());
				}
			}
			if (null != out) {
				try {
					out.close();
				} catch (IOException e) {
					mStatus = STATUS_ERROR;
					LOG.e(TAG, "saveZipFile: {0}", e.getMessage());
				}
			}
		}
		/* backup old file */
		count = GestureLibrary.MAX_NUM_OF_STORE;
		for (int num = 0; num < count; num++) {
			if (SvmWrapper.LibsvmData[num].exists()) {
				final File datafile = SvmWrapper.LibsvmData[num];
				final String name = Utils.concat(datafile.getAbsolutePath(),
						".backup");
				final File backup = new File(name);
				if (datafile.renameTo(backup)) {
					LOG.d(TAG, "extractZipFile: rename {0}",
							backup.getAbsoluteFile());
				} else {
					LOG.e(TAG, "extractZipFile: cannot rename {0}", name);
				}
			}
		}
		/* Extracts zip file */
		try {
			zis = new ZipInputStream(new FileInputStream(file));
			for (ZipEntry ze = zis.getNextEntry(); null != ze; ze = zis
					.getNextEntry()) {
				LOG.d(TAG, "extractZipFile: ze.getName()={0}", ze.getName());
				if ('.' == ze.getName().charAt(0)) {
					continue;
				}
				if (ze.isDirectory()) {
					parent = new File(dir, ze.getName());
					if (parent.exists()) {
						continue;
					}
					if (!parent.mkdirs()) {
						mStatus = STATUS_ERROR;
						LOG.e(TAG, "saveZipFile: can't create dir: {0}",
								parent.getAbsolutePath());
						break;
					}
					continue;
				}
				try {
					out = new BufferedOutputStream(new FileOutputStream(
							new File(dir, ze.getName())), IO_BUFFER_SIZE);
					for (count = zis.read(buf); count != -1; count = zis
							.read(buf)) {
						out.write(buf, 0, count);
					}
				} finally {
					mStatus = STATUS_SUCCESS;
					if (null != out) {
						try {
							out.close();
						} catch (IOException e) {
							mStatus = STATUS_ERROR;
							LOG.e(TAG, "extractZipFile: {0}", e.getMessage());
						}
					}
				}
			}
		} catch (IOException e) {
			mStatus = STATUS_ERROR;
			LOG.e(TAG, "extractZipFile: {0}", e.getMessage());
		} finally {
			if (null != zis) {
				try {
					zis.close();
				} catch (IOException e) {
					mStatus = STATUS_ERROR;
					LOG.e(TAG, "extractZipFile: {0}", e.getMessage());
				}
			}
		}
	}

}