///<reference path="all.ts"/>
module jg  {
	/**
	 * 継承前提で作られている基本のウィンドウクラス。
	 * jgame.jsはUIライブラリではなくシンプルにエンジンのみの提供を目指しているため、本クラスを継承したUIクラスはMessageeWindowクラス以外用意されていないｋ。
	 */
	export class UIWindow extends E {
		/** 余白のサイズ */
		padding:IRectangle;
		/** 背景。変更する場合はsetBgを利用するべき */
		bg:E;

		/**
		 * コンストラクタ
		 * @param width 横幅
		 * @param height 縦幅
		 * @param bgImage 背景画像。省略時はdefaultSkinメソッドによるUIWindowが生成される
		 * @param padding 余白
		 */
		constructor(width:number, height:number, bgImage?:any, padding?:IRectangle) {
			super();
			this.width = width;
			this.height = height;

			this.padding = padding ? padding : new Rectangle(8, 8, 8, 8);
			this.entities = [];
			if (bgImage) {
				var bg = (bgImage instanceof E) ? bgImage : new Sprite(bgImage);
				this.setBg(this.createBgImage(bg));
			} else {
				this.defaultSkin();
			}
		}

		/**
		 * 背景要素を返す。
		 * って実態はthis.entities[0]を返しているけどいいのかこれ？
		 */
		getBg():E {
			return this.entities[0];
		}

		/**
		 * 背景要素を指定する。背景は常に最背面に配置される点に注意。
		 * @param bg 指定する背景。通常、createBgImageで生成した画像を指定する
		 */
		setBg(bg:E) {
			if (this.bg)
				this.bg.remove();
			this.bg = bg;
			this.insert(bg, 0);
		}

		/**
		 * 描画要素を基に背景画像を生成する
		 * @param e 背景の基となる描画要素。余白に合わせた引き伸ばし処理が行われる
		 * @param srcPadding 余白。省略時はthis.paddingが用いられる
		 * @param buf 生成に利用するバッファ。省略時は自動生成。フクスウUIを生成する場合には、指定すると若干高速になる
		 */
		createBgImage(e:E, srcPadding?:IRectangle, buf?:BufferedRenderer) {
			if (! buf)
				var buf = new BufferedRenderer(e);
			if (srcPadding === undefined)
				srcPadding = this.padding;
			buf.renderUnit(e);
			if (buf.filter)
				buf.applyFilter(buf.c, buf.size);
			var bg = new E();
			bg.width = this.width;
			bg.height = this.height;
			bg.entities = [];
			//四隅
			var s4_1 = buf.createSprite({x: 0, y: 0, width: srcPadding.left, height:srcPadding.top});
			s4_1.moveTo(0, 0);
			bg.append(s4_1);
			var s4_2 = buf.createSprite({x: e.width-srcPadding.right, y: 0, width: srcPadding.right, height:srcPadding.top});
			s4_2.moveTo(this.width-srcPadding.right, 0);
			bg.append(s4_2);
			var s4_3 = buf.createSprite({x: e.width-srcPadding.right, y: e.height-srcPadding.bottom, width: srcPadding.right, height:srcPadding.bottom});
			s4_3.moveTo(this.width - srcPadding.right, this.height - srcPadding.bottom);
			bg.append(s4_3);
			var s4_4 = buf.createSprite({x: 0, y: e.height-srcPadding.bottom, width: srcPadding.left, height:srcPadding.bottom});
			s4_4.moveTo(0, this.height - srcPadding.bottom);
			bg.append(s4_4);

			//四片
			var h4_1 =buf.createSprite({
				x: srcPadding.left,
				y: 0,
				width: e.width-srcPadding.right - srcPadding.left,
				height:srcPadding.top
			});
			h4_1.moveTo(srcPadding.left, 0);
			h4_1.width = this.width - srcPadding.right - srcPadding.left;
			bg.append(h4_1);
			var h4_2 = buf.createSprite({
				x: e.width - srcPadding.right,
				y: srcPadding.top,
				width: srcPadding.right,
				height: e.height - srcPadding.top - srcPadding.bottom
			});
			h4_2.moveTo(this.width - srcPadding.right, srcPadding.top);
			h4_2.height = this.height - srcPadding.top - srcPadding.bottom;
			bg.append(h4_2);
			var h4_3 = buf.createSprite({
				x: srcPadding.left,
				y: e.height - srcPadding.bottom,
				width: e.width - srcPadding.right - srcPadding.left,
				height: srcPadding.bottom,
			});
			h4_3.moveTo(srcPadding.left, this.height - srcPadding.bottom);
			h4_3.width = this.width - srcPadding.right - srcPadding.left;
			bg.append(h4_3);
			var h4_4 = buf.createSprite({
				x: 0,
				y: srcPadding.top,
				width: srcPadding.left,
				height: e.height - srcPadding.top - srcPadding.bottom
			});
			h4_4.moveTo(0, srcPadding.top);
			h4_4.height = this.height - srcPadding.top - srcPadding.bottom;
			bg.append(h4_4);

			//中央
			var center = buf.createSprite({
				x: srcPadding.left,
				y: srcPadding.top,
				width: e.width - srcPadding.left - srcPadding.right,
				height: e.height - srcPadding.top - srcPadding.bottom
			});
			center.moveTo(srcPadding.left, srcPadding.top);
			center.width = this.width - srcPadding.left - srcPadding.right;
			center.height = this.height - srcPadding.top - srcPadding.bottom;
			bg.append(center);

			return bg.createSprite();
		}

		/**
		 * 簡素な見た目のデフォルトのスキンを生成する。
		 */
		defaultSkin() {
			if (this.bg)
				this.bg.remove();
			var bgColor = JGUtil.createLinearGradient(
				new Rectangle(0, 0, this.width, this.height),
				["rgba(138,193,255,0.5)", "rgba(222, 235, 250, 0.5)"]
			);
			var dummy = new E();
			dummy.width = this.width;
			dummy.height = this.height;
			dummy.entities = [];
			var bg:Shape = new Shape(
				this.width,
				this.height,
				ShapeStyle.Fill,
				bgColor
			);
			dummy.append(bg);

			var textBg:Shape = new Shape(
				this.width-(this.padding.left+this.padding.right)+4,
				this.height-(this.padding.top+this.padding.bottom)+4,
				ShapeStyle.Fill,
				"rgba(45,73,136,0.2)"
			);
			textBg.moveTo(this.padding.left-2, this.padding.top-2);
			dummy.append(textBg);
			this.setBg(dummy.createSprite());
		}
	}
}