#include "jwk.h"

#define LINE_TEXT_SIZE 256

typedef struct LogTextBuffer LogTextBuffer;
struct LogTextBuffer{
	char text[LINE_TEXT_SIZE];
	JtkColor color;
}; 

typedef struct LogBox LogBox;
struct LogBox
{
	JwkWindow *jw;
	JtkGC *gc;

	int buffer_num;
	LogTextBuffer *buffer;
	JtkColor text_color;
};

static void DestroyLogBoxBuffer(LogBox *lb)
{
	if(lb->buffer == NULL)
		return;

	j_free(lb->buffer);
	lb->buffer = NULL;
	lb->buffer_num = 0;
}

static void CreateLogBoxBuffer(LogBox *lb, int line_num)
{
	if(lb->buffer != NULL)
		DestroyLogBoxBuffer(lb);
	
	if(line_num < 1)
		line_num = 1;
	
	lb->buffer = j_malloc(sizeof(LogTextBuffer) * line_num);
	if(lb->buffer == NULL)
		return;
	j_zero(lb->buffer, sizeof(LogTextBuffer) * line_num);
	lb->buffer_num = line_num;
}

static void logbox_callback(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	LogBox *lb = widget->widget;

	switch(event->type){
	case JWK_EVENT_EXPOSE:
		{
			JtkSize window_size;
			JtkSize text_size;
			char *test_text = "abcあいう12３４*!＃＄";
			int i;
			int py;
			JtkEvent je;
			
			je.type = JTK_EVENT_EXPOSE;
			je.expose.widget = widget;
			je.expose.data = widget->data;
			je.expose.x = event->expose.x;
			je.expose.y = event->expose.y;
			je.expose.width = event->expose.width;
			je.expose.height = event->expose.height;
			je.expose.count = event->expose.count;
			if(widget->callback != NULL)
				widget->callback(&je);
			
			jwkGetWindowSize(lb->jw, &window_size);
			jtkGetTextSize(lb->gc, test_text, j_strlen(test_text), &text_size);
			for(i=0; i<lb->buffer_num; i++){
				py = window_size.height - (i + 1)*(text_size.height + 2);
				if(py < -text_size.height)
					break;
				jtkSetColor(lb->gc, lb->buffer[i].color);
				jtkDrawText(lb->gc, 5, py,
								lb->buffer[i].text,
								j_strlen(lb->buffer[i].text));
			}
		}
		break;
	default:
		break;
	}
}

static JtkGC* create_logbox_gc(JtkWidget *widget, JtkGCMode mode, JtkGCType type)
{
	LogBox *lb = widget->widget;
	
	return jwkCreateGC(lb->jw, mode, type);
}

static void destroy_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;

	jwkDestroyWindow(lb->jw);
	DestroyLogBoxBuffer(lb);
	j_free(lb);
}

static void map_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;

	jwkMapWindow(lb->jw);
}

static void unmap_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;

	jwkUnmapWindow(lb->jw);
}

static void clear_logbox(JtkWidget *widget,
	int px, int py, int width, int height, JtkBool exposures)
{
	LogBox *lb = widget->widget;
	
	jwkClearWindow(lb->jw,
		px, py, width, height, exposures);
}

static void raise_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;
	
	jwkRaiseWindow(lb->jw);
}

static void lower_logbox(JtkWidget *widget)
{
	LogBox *lb = widget->widget;
	
	jwkLowerWindow(lb->jw);
}

static void set_logbox_pos(JtkWidget *widget, int px, int py)
{
	LogBox *lb = widget->widget;

	jwkSetWindowPos(lb->jw, px, py);
}

static void set_logbox_size(JtkWidget *widget, int width, int height)
{
	LogBox *lb = widget->widget;

	jwkSetWindowSize(lb->jw, width, height);
}

static void get_logbox_pos(JtkWidget *widget, JtkPoint *point)
{
	LogBox *lb = widget->widget;
	
	jwkGetWindowPos(lb->jw, point);
}

static void get_logbox_size(JtkWidget *widget, JtkSize *size)
{
	LogBox *lb = widget->widget;
	
	jwkGetWindowSize(lb->jw, size);
}

static void set_logbox_color(JtkWidget *widget, JtkColor color)
{
	LogBox *lb = widget->widget;
	
	jwkSetWindowColor(lb->jw, color);
}

static void set_logbox_border_size(JtkWidget *widget, int size)
{
	LogBox *lb = widget->widget;
	
	jwkSetWindowBorderSize(lb->jw, size);
}

static void set_logbox_border_color(JtkWidget *widget, JtkColor color)
{
	LogBox *lb = widget->widget;
	
	jwkSetWindowBorderColor(lb->jw, color);
}

static void set_logbox_text_color(JtkWidget *widget, JtkColor color)
{
	LogBox *lb = widget->widget;
	
	lb->text_color = color;
}

static void set_logbox_text_lines(JtkWidget *widget, int lines)
{
	LogBox *lb = widget->widget;
	
	CreateLogBoxBuffer(lb, lines);
}

static void set_logbox_text(JtkWidget *widget, char *text)
{
	int i;
	LogBox *lb = widget->widget;
	
	for(i=lb->buffer_num - 2; i>=0; i--){
		j_strcpy(lb->buffer[i+1].text, lb->buffer[i].text);
		lb->buffer[i+1].color = lb->buffer[i].color;
	}
	j_strcpy(lb->buffer[0].text, text);
	lb->buffer[0].color = lb->text_color;
	
	jwkClearWindow(lb->jw, 0, 0, 0, 0, JTK_TRUE);
}

static void clear_logbox_text(JtkWidget *widget)
{
	LogBox *lb = widget->widget;
	
	j_zero(lb->buffer, sizeof(LogTextBuffer) * lb->buffer_num);
	jwkClearWindow(lb->jw, 0, 0, 0, 0, JTK_TRUE);
}

static JwkWidgetFunc logbox_func =
{
	.create_gc = create_logbox_gc,
	.destroy = destroy_logbox,
	.map = map_logbox,
	.unmap = unmap_logbox,
	.clear = clear_logbox,
	.raise = raise_logbox,
	.lower = lower_logbox,
	.set_pos = set_logbox_pos,
	.set_size = set_logbox_size,
	.get_pos = get_logbox_pos,
	.get_size = get_logbox_size,
	.set_color = set_logbox_color,
	.set_border_size = set_logbox_border_size,
	.set_border_color = set_logbox_border_color,
	.set_text_color = set_logbox_text_color,
	.set_text_lines = set_logbox_text_lines,
	.set_text = set_logbox_text,
	.clear_text = clear_logbox_text,
};

JtkWidget* jtkCreateLogbox(JtkWidget *parent)
{
	JtkWidget *widget;
	LogBox *lb;

	if(parent == NULL)
		return NULL;

	widget = j_malloc(sizeof(JtkWidget));
	if(widget == NULL)
		return NULL;
	
	lb = j_malloc(sizeof(LogBox));
	if(lb == NULL){
		j_free(widget);
		return NULL;
	}

	j_zero(widget, sizeof(JtkWidget));
	j_zero(lb, sizeof(LogBox));
	
	widget->widget = lb;
	widget->func = &logbox_func;
	
	lb->jw = jwkCreateWindow(jwkGetContainer(parent));
	lb->gc = jwkCreateGC(lb->jw, JTK_GCMODE_SCREEN, JTK_GCTYPE_NORMAL);
	lb->text_color = jtkRGB(255,255,255);
	CreateLogBoxBuffer(lb, 10);

	jwkSetWindowColor(lb->jw, jtkRGB(64,64,64));
	jwkSetWindowBorderSize(lb->jw, 1);
	jwkSetWindowBorderColor(lb->jw, jtkRGB(128,128,128));
	jwkSetWidget(lb->jw, logbox_callback, widget);
	jwkSetWindowEventMask(lb->jw, JWK_EVENTMASK_EXPOSE);
	
	return widget;
}
