/*
	JTK Window Widget
*/

#include "jwk.h"

typedef struct Window Window;
struct Window
{
	JwkWindow *frame;
	JwkWindow *container;
};

static void window_frame_callback(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	Window *window = widget->widget;
	JtkEvent je;
	
	switch(event->type){
		case JWK_EVENT_DESTROY:
		je.type = JTK_EVENT_DESTROY;
		je.destroy.widget = widget;
		je.destroy.data = widget->data;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_MAP:
		je.type = JTK_EVENT_MAP;
		je.map.widget = widget;
		je.map.data = widget->data;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_UNMAP:
		je.type = JTK_EVENT_UNMAP;
		je.unmap.widget = widget;
		je.unmap.data = widget->data;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_MOVE:
		je.type = JTK_EVENT_MOVE;
		je.move.widget = widget;
		je.move.data = widget->data;
		je.move.px = event->move.px;
		je.move.py = event->move.py;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_RESIZE:
		je.type = JTK_EVENT_RESIZE;
		je.resize.widget = widget;
		je.resize.data = widget->data;
		je.resize.width = event->resize.width;
		je.resize.height = event->resize.height;
		if(event->resize.width >= 1 &&
			event->resize.height >= 1){
			jwkSetWindowPos(window->container, 0, 0);
			jwkSetWindowSize(window->container,
				event->resize.width, event->resize.height);
		}
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_CLOSE:
		je.type = JTK_EVENT_CLOSE;
		je.close.widget = widget;
		je.close.data = widget->data;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	default:
		break;
	}
}

static void window_container_callback(JwkEvent *event)
{
	JtkWidget *widget = event->any.widget;
	JtkEvent je;
	
	switch(event->type){
	case JWK_EVENT_KEYDOWN:
		je.type = JTK_EVENT_KEYDOWN;
		je.keydown.widget = widget;
		je.keydown.data = widget->data;
		je.keydown.px = event->keydown.px;
		je.keydown.py = event->keydown.py;
		je.keydown.screen_px = event->keydown.screen_px;
		je.keydown.screen_py = event->keydown.screen_py;
		je.keydown.keymask = event->keydown.keymask;
		je.keydown.keysym = event->keydown.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_KEYUP:
		je.type = JTK_EVENT_KEYUP;
		je.keyup.widget = widget;
		je.keyup.data = widget->data;
		je.keyup.px = event->keyup.px;
		je.keyup.py = event->keyup.py;
		je.keyup.screen_px = event->keyup.screen_px;
		je.keyup.screen_py = event->keyup.screen_py;
		je.keyup.keymask = event->keyup.keymask;
		je.keyup.keysym = event->keyup.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_BUTTONDOWN:
		je.type = JTK_EVENT_BUTTONDOWN;
		je.buttondown.widget = widget;
		je.buttondown.data = widget->data;
		je.buttondown.px = event->buttondown.px;
		je.buttondown.py = event->buttondown.py;
		je.buttondown.screen_px = event->buttondown.screen_px;
		je.buttondown.screen_py = event->buttondown.screen_py;
		je.buttondown.button = event->buttondown.button;
		je.buttondown.keymask = event->buttondown.keymask;
		je.buttondown.keysym = event->buttondown.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_BUTTONUP:
		je.type = JTK_EVENT_BUTTONUP;
		je.buttonup.widget = widget;
		je.buttonup.data = widget->data;
		je.buttonup.px = event->buttonup.px;
		je.buttonup.py = event->buttonup.py;
		je.buttonup.screen_px = event->buttonup.screen_px;
		je.buttonup.screen_py = event->buttonup.screen_py;
		je.buttonup.button = event->buttonup.button;
		je.buttonup.keymask = event->buttonup.keymask;
		je.buttonup.keysym = event->buttonup.keysym;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_MOTION:
		je.type = JTK_EVENT_MOTION;
		je.motion.widget = widget;
		je.motion.data = widget->data;
		je.motion.px = event->motion.px;
		je.motion.py = event->motion.py;
		je.motion.screen_px = event->motion.screen_px;
		je.motion.screen_py = event->motion.screen_py;
		je.motion.keymask = event->motion.keymask;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_ENTER:
		je.type = JTK_EVENT_ENTER;
		je.enter.widget = widget;
		je.enter.data = widget->data;
		je.enter.px = event->enter.px;
		je.enter.py = event->enter.py;
		je.enter.screen_px = event->enter.screen_px;
		je.enter.screen_py = event->enter.screen_py;
		je.enter.keymask = event->enter.keymask;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_LEAVE:
		je.type = JTK_EVENT_LEAVE;
		je.leave.widget = widget;
		je.leave.data = widget->data;
		je.leave.px = event->leave.px;
		je.leave.py = event->leave.py;
		je.leave.screen_px = event->leave.screen_px;
		je.leave.screen_py = event->leave.screen_py;
		je.leave.keymask = event->leave.keymask;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	case JWK_EVENT_EXPOSE:
		je.type = JTK_EVENT_EXPOSE;
		je.expose.widget = widget;
		je.expose.data = widget->data;
		je.expose.x = event->expose.x;
		je.expose.y = event->expose.y;
		je.expose.width = event->expose.width;
		je.expose.height = event->expose.height;
		je.expose.count = event->expose.count;
		if(widget->callback != NULL)
			widget->callback(&je);
		break;
	default:
		break;
	}
}

static JwkWindow* window_get_container(JtkWidget *widget)
{
	Window *window = widget->widget;
	return window->container;
}

static JwkWindow* window_get_frame(JtkWidget *widget)
{
	Window *window = widget->widget;
	return window->frame;
}

static JtkGC* window_create_gc(JtkWidget *widget, JtkGCMode mode, JtkGCType type)
{
	Window *window = widget->widget;
	
	return jwkCreateGC(window->container, mode, type);
}

static void destroy_window(JtkWidget *widget)
{
	Window *window = widget->widget;
	
	jwkDestroyWindow(window->frame);
	j_free(window);
}

static void map_window(JtkWidget *widget)
{
	Window *window = widget->widget;
	
	jwkMapWindow(window->frame);
}

static void unmap_window(JtkWidget *widget)
{
	Window *window = widget->widget;
	
	jwkUnmapWindow(window->frame);
}

static void clear_window(JtkWidget *widget,
	int px, int py, int width, int height, JtkBool exposures)
{
	Window *window = widget->widget;
	
	jwkClearWindow(window->container,
		px, py, width, height, exposures);
}

static void raise_window(JtkWidget *widget)
{
	Window *window = widget->widget;
	
	jwkRaiseWindow(window->frame);
}

static void lower_window(JtkWidget *widget)
{
	Window *window = widget->widget;
	
	jwkLowerWindow(window->frame);
}

static void set_window_pos(JtkWidget *widget, int px, int py)
{
	Window *window = widget->widget;
	
	jwkSetWindowPos(window->frame, px, py);
}

static void set_window_size(JtkWidget *widget, int width, int height)
{
	Window *window = widget->widget;
	
	jwkSetWindowSize(window->frame, width, height);
	jwkSetWindowSize(window->container, width, height);
}

static void get_window_pos(JtkWidget *widget, JtkPoint *point)
{
	Window *window = widget->widget;
	
	jwkGetWindowPos(window->frame, point);
}

static void get_window_size(JtkWidget *widget, JtkSize *size)
{
	Window *window = widget->widget;
	
	jwkGetWindowSize(window->frame, size);
}

static void set_window_color(JtkWidget *widget, JtkColor color)
{
	Window *window = widget->widget;
	
	jwkSetWindowColor(window->container, color);
}

static void set_window_border_size(JtkWidget *widget, int size)
{
	Window *window = widget->widget;
	
	jwkSetWindowBorderSize(window->frame, size);
}

static void set_window_border_color(JtkWidget *widget, JtkColor color)
{
	Window *window = widget->widget;
	
	jwkSetWindowBorderColor(window->frame, color);
}

static void set_window_text(JtkWidget *widget, char *text)
{
	Window *window = widget->widget;
	
	jwkSetWindowText(window->frame, text);
}

static void clear_window_text(JtkWidget *widget)
{
	set_window_text(widget, NULL);
}

static void set_window_focus(JtkWidget *widget)
{
	Window *window = widget->widget;
	
	jwkSetFocus(window->container);
}

static JwkWidgetFunc window_func =
{
	.get_container = window_get_container,
	.get_frame = window_get_frame,
	.create_gc = window_create_gc,
	.destroy = destroy_window,
	.map = map_window,
	.unmap = unmap_window,
	.clear = clear_window,
	.raise = raise_window,
	.lower = lower_window,
	.set_pos = set_window_pos,
	.set_size = set_window_size,
	.get_pos = get_window_pos,
	.get_size = get_window_size,
	.set_color = set_window_color,
	.set_border_size = set_window_border_size,
	.set_border_color = set_window_border_color,
	.set_text = set_window_text,
	.clear_text = clear_window_text,
	.set_focus = set_window_focus,
};

JtkWidget* jtkCreateWindow(JtkWidget *parent)
{
	JtkWidget *widget;
	Window *window;
	
	widget = j_malloc(sizeof(JtkWidget));
	if(widget == NULL)
		return NULL;
	
	window = j_malloc(sizeof(Window));
	if(window == NULL){
		j_free(widget);
		return NULL;
	}
	
	j_zero(widget, sizeof(JtkWidget));
	j_zero(window, sizeof(Window));
	
	widget->widget = window;
	widget->func = &window_func;
	
	window->frame = jwkCreateWindow(jwkGetContainer(parent));
	jwkSetWidget(window->frame, window_frame_callback, widget);
	jwkSetWindowEventMask(window->frame,
		JWK_EVENTMASK_KEYDOWN | JWK_EVENTMASK_KEYUP |
		JWK_EVENTMASK_DESTROY |
		JWK_EVENTMASK_MAP |
		JWK_EVENTMASK_UNMAP |
		JWK_EVENTMASK_MOVE |
		JWK_EVENTMASK_RESIZE);
	
	window->container = jwkCreateWindow(window->frame);
	jwkSetWidget(window->container, window_container_callback, widget);
	jwkSetWindowEventMask(window->container,
	//	JWK_EVENTMASK_KEYDOWN | JWK_EVENTMASK_KEYUP |
		JWK_EVENTMASK_BUTTONDOWN | JWK_EVENTMASK_BUTTONUP |
		JWK_EVENTMASK_MOTION | JWK_EVENTMASK_ENTER | JWK_EVENTMASK_LEAVE |
		JWK_EVENTMASK_EXPOSE);
	jwkSetWindowPos(window->container, 0, 0);
	jwkMapWindow(window->container);
	
	return widget;
}
