#include "x11-core.h"

static XIMStyle SelectInputStyle(XIM im)
{
	static XIMStyle preedit[] = {
		XIMPreeditPosition,
		XIMPreeditArea,
		XIMPreeditNothing,
		0};
	static XIMStyle status[] = {
		XIMStatusArea,
		XIMStatusNothing,
		0};
	XIMStyles *im_styles;
	int i, j, k;
	
	XGetIMValues(im, XNQueryInputStyle, &im_styles, NULL);
	
	i = 0;
	while(preedit[i] != 0){
		j = 0;
		while(status[j] != 0){
			for(k = 0; k < im_styles->count_styles; k++){
				if((preedit[i] & im_styles->supported_styles[k]) &&
					(status[j] & im_styles->supported_styles[k]))
					return im_styles->supported_styles[k];
			}
			j++;
		}
		i++;
	}
	
	fprintf(stderr, "Input Style does not match.\n");
	return 0;
}

static XIC CreateInputContext(XIM im, XIMStyle style, XFontSet fs, Window p)
{
	XIC ic;
	XVaNestedList list;
	XPoint spot;
	
	spot.x = 10;
	spot.y = 30;
	list = XVaCreateNestedList(0,
		XNFontSet, fs,
		XNSpotLocation, &spot,
		NULL);
	ic = XCreateIC(im,
		XNInputStyle, style,
		XNClientWindow, p,
		XNPreeditAttributes, list,
		XNStatusAttributes, list,
		NULL);
	
	if(ic == NULL){
		fprintf(stderr, "Cannot create input context.\n");
		exit(1);
	}
	XFree(list);
	return ic;
}

/* Window */
JwkWindow* jwkCreateWindow(JwkWindow *parent)
{
	JwkWindow *jw = NULL;
	Window p;
	XSizeHints hints;
	
	XFontSet fs;
	char **miss, *def;
	int n_miss;
	XIMStyle style;
	
	if(parent == NULL)
		p = RootWindow(display, 0);
	else
		p = parent->window_id;
	
	jw = j_malloc(sizeof(JwkWindow));
	if(jw == NULL)
		return NULL;
	j_zero(jw, sizeof(JwkWindow));
	
	jw->window_id = XCreateSimpleWindow(display, p, 0, 0, 32, 32, 0,
										BlackPixel(display, 0),
										WhitePixel(display, 0));
	x11_wlist_add_window(jw);
	
	if(parent == NULL){
		jw->top_level = 1;
		XSetWMProtocols(display, jw->window_id, &WM_DELETE_WINDOW, 1);
		
		memset(&hints, 0, sizeof(XSizeHints));
		hints.flags = USPosition | USSize;
		XSetWMNormalHints(display, jw->window_id, &hints);
		
		/* IM */
		fs = XCreateFontSet(display, "-*-*-medium-r-normal--14-*",
			&miss, &n_miss, &def);
		style = SelectInputStyle(xim);
		jw->xic = CreateInputContext(xim, style, fs, jw->window_id);
		
		XSetICValues(jw->xic, XNFocusWindow, jw->window_id, NULL);
		XSetICFocus(jw->xic);
	}
	else{
		jw->top_level = 0;
		jw->xic = parent->xic;
	}
	
	return jw;
}

void jwkDestroyWindow(JwkWindow *jw)
{
	JwkEvent je;

	x11_wlist_remove_window(jw);

	je.type = JWK_EVENT_DESTROY;
	je.any.window = jw;
	je.any.widget = jw->widget;
	if((jw->event_mask & JWK_EVENTMASK_DESTROY) &&
		(jw->callback != NULL))
		jw->callback(&je);
	XDestroyWindow(display, jw->window_id);
}

void jwkMapWindow(JwkWindow *jw)
{
	XMapWindow(display, jw->window_id);
	XFlush(display);
}

void jwkUnmapWindow(JwkWindow *jw)
{
	XUnmapWindow(display, jw->window_id);
	XFlush(display);
}

void jwkClearWindow(JwkWindow *jw, int x, int y,
						int width, int height,
						JtkBool exposures)
{
	if(exposures == JTK_TRUE)
		XClearArea(display, jw->window_id, x, y, width, height, True);
	else
		XClearArea(display, jw->window_id, x, y, width, height, False);
	XSync(display, False);
}

void jwkRaiseWindow(JwkWindow *jw)
{
	XRaiseWindow(display, jw->window_id);
	XFlush(display);
}

void jwkLowerWindow(JwkWindow *jw)
{
	XLowerWindow(display, jw->window_id);
	XFlush(display);
}

void jwkSetWindowPos(JwkWindow *jw, int px, int py)
{
	XMoveWindow(display, jw->window_id, px, py);
	XFlush(display);
}

void jwkSetWindowSize(JwkWindow *jw, int width, int height)
{
	XResizeWindow(display, jw->window_id, width, height);
	XFlush(display);
}

void jwkSetWindowColor(JwkWindow *jw, JtkColor color)
{
	XSetWindowBackground(display, jw->window_id, x11_make_rgb(color));
	XFlush(display);
}

void jwkSetWindowBorderSize(JwkWindow *jw, int size)
{
	XSetWindowBorderWidth(display, jw->window_id, size);
	XFlush(display);
}

void jwkSetWindowBorderColor(JwkWindow *jw, JtkColor color)
{
	XSetWindowBorder(display, jw->window_id, x11_make_rgb(color));
	XFlush(display);
}

void jwkSetWindowText(JwkWindow *jw, char *text)
{	
	if(jw->top_level){
		XTextProperty ct;
		
		XmbTextListToTextProperty(display, &text, 1,
			XCompoundTextStyle, &ct);
		XSetWMName(display, jw->window_id, &ct);
		XSetWMIconName(display, jw->window_id, &ct);
		XFlush(display);
	}
}

void jwkGetWindowPos(JwkWindow *jw, JtkPoint *pos)
{
	Window w_root;
	int x, y;
	unsigned int width, height, border, depth;
	
	XGetGeometry(display, jw->window_id, &w_root,
		&x, &y, &width, &height, &border, &depth);
	
	pos->px = x;
	pos->py = y;
}

void jwkGetWindowSize(JwkWindow *jw, JtkSize *size)
{
	Window w_root;
	int x, y;
	unsigned int width, height, border, depth;
	
	XGetGeometry(display, jw->window_id, &w_root,
		&x, &y, &width, &height, &border, &depth);
	
	size->width = width;
	size->height = height;
}

void jwkSetWindowEventMask(JwkWindow *jw, int mask)
{
	int xmask = 0;
	
	if(mask & JWK_EVENTMASK_BUTTONDOWN)
		xmask |= ButtonPressMask;
	if(mask & JWK_EVENTMASK_BUTTONUP)
		xmask |= ButtonReleaseMask;
	if(mask & JWK_EVENTMASK_MOTION)
		xmask |= PointerMotionMask;
	if(mask & JWK_EVENTMASK_ENTER)
		xmask |= EnterWindowMask;
	if(mask & JWK_EVENTMASK_LEAVE)
		xmask |= LeaveWindowMask;
	if(mask & JWK_EVENTMASK_FOCUSIN)
		xmask |= FocusChangeMask;
	if(mask & JWK_EVENTMASK_FOCUSOUT)
		xmask |= FocusChangeMask;
	if(mask & JWK_EVENTMASK_MAP)
		xmask |= StructureNotifyMask;
	if(mask & JWK_EVENTMASK_UNMAP)
		xmask |= StructureNotifyMask;
	if(mask & JWK_EVENTMASK_MOVE)
		xmask |= StructureNotifyMask;
	if(mask & JWK_EVENTMASK_RESIZE)
		xmask |= StructureNotifyMask;
	if(mask & JWK_EVENTMASK_EXPOSE)
		xmask |= ExposureMask;

	if(jw->top_level){
		int im_mask;
		XGetICValues(jw->xic, XNFilterEvents, &im_mask, NULL);
		if(mask & JWK_EVENTMASK_KEYDOWN)
			xmask |= KeyPressMask;
		if(mask & JWK_EVENTMASK_KEYUP)
			xmask |= KeyReleaseMask;
		if(mask & JWK_EVENTMASK_STRING)
			xmask |= KeyPressMask;
		XSelectInput(display, jw->window_id, xmask | im_mask);
	}else{
		XSelectInput(display, jw->window_id, xmask);
	}
	jw->event_mask = mask;
}

void jwkSetWidget(JwkWindow *jw,
					void (*callback)(JwkEvent *event),
					JtkWidget *widget)
{
	jw->callback = callback;
	jw->widget = widget;
}
