package com.kurukurupapa.tryandroid.fw.view;

import android.content.Context;
import android.graphics.Canvas;
import android.util.AttributeSet;
import android.view.MotionEvent;
import android.view.View;

/**
 * 色選択ビュー
 *
 * 元ネタ<br>
 * Y.A.M の 雑記帳: Android　ColorPickerDialog を作った
 * http://y-anz-m.blogspot.com/2010/05/androidcolorpickerdialog.html
 */
public class BoxHsvColorPicker extends View {
	public interface OnColorListener {
		public void onColorChanging(int color);

		public void onColorChanged(int color);
	}

	private static final int MIN_RADIUS = 50;
	private static final float SPACE = 5;

	private float radius;

	private OnColorListener listener;

	BoxHsvColorPickerHueHelper hueHelper;
	BoxHsvColorPickerSvHelper svHelper;

	public BoxHsvColorPicker(Context context) {
		super(context);
		setup();
	}

	public BoxHsvColorPicker(Context context, AttributeSet attrs) {
		super(context, attrs);
		setup();
	}

	private void setup() {
		hueHelper = new BoxHsvColorPickerHueHelper(24);
		svHelper = new BoxHsvColorPickerSvHelper(9);
	}

	public void setColorNum(int hueNum, int svNum) {
		hueHelper.setColorNum(hueNum);
		svHelper.setColorNum(svNum);
	}

	public void setSelectColor(int color) {
		hueHelper.setSelectColorFromColor(color);
		svHelper.setSelectColorFromColor(color);
		invalidate();
	}

	public int getSelectColor() {
		return svHelper.getSelectColor(hueHelper.getHue());
	}

	public void setOnColorListener(OnColorListener l) {
		listener = l;
	}

	@Override
	protected void onDraw(Canvas canvas) {
		canvas.translate(radius, radius);
		hueHelper.drawOval(canvas);
		svHelper.drawSVRegion(canvas, hueHelper.getHue());
	}

	@Override
	protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		int size;
		if (widthMeasureSpec < heightMeasureSpec) {
			size = MeasureSpec.getSize(widthMeasureSpec);
		} else {
			size = MeasureSpec.getSize(heightMeasureSpec);
		}
		size = Math.max(size, MIN_RADIUS * 2);
		setMeasuredDimension(size, size);

		radius = size / 2.0f;
		float r = (float) ((radius - hueHelper.getArcWidth() - SPACE) * Math
				.sin(Math.PI / 4.0));
		hueHelper.setSize(radius);
		svHelper.setSize(-r, -r, r, r);
	}

	@Override
	public boolean onTouchEvent(MotionEvent event) {
		float x = event.getX() - radius;
		float y = event.getY() - radius;
		float r = (float) (java.lang.Math.sqrt(x * x + y * y));
		boolean inOval = false;
		boolean inRect = false;

		if (hueHelper.contains(r)) {
			inOval = true;
		} else if (svHelper.contains(x, y)) {
			inRect = true;
		}

		switch (event.getAction()) {
		case MotionEvent.ACTION_DOWN:
		case MotionEvent.ACTION_MOVE:
			if (inOval) {
				// 色相環を操作されている場合
				hueHelper.setSelectColorFromXY(x, y);
				invalidate();

			} else if (inRect) {
				// 明度、彩度を操作されている場合
				svHelper.setSelectColorFromXY(x, y);
				invalidate();
			}
			// リスナーへ通知
			if (listener != null) {
				listener.onColorChanging(getSelectColor());
			}
			break;

		case MotionEvent.ACTION_UP:
			// リスナーへ通知
			if (listener != null) {
				listener.onColorChanged(getSelectColor());
			}
			break;
		}
		return true;
	}

}
