package com.kurukurupapa.tryandroidui.draglist;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.PixelFormat;
import android.view.Gravity;
import android.view.View;
import android.view.WindowManager;
import android.widget.ImageView;
import android.widget.ListView;

/**
 * ドラッグ中の項目を表示するためのビューヘルパー
 */
public class DragListViewHelper {
	private static final Bitmap.Config DRAG_BITMAP_CONFIG = Bitmap.Config.ARGB_8888;
	private static final int BACKGROUND_COLOR = Color.argb(128, 0xFF, 0xFF,
			0xFF);
	// private static final int Y_GAP = 32;
	private static final int Y_GAP = 20;

	private ListView mListView;
	private WindowManager mWindowManager;
	private ImageView mDragImageView;
	private WindowManager.LayoutParams mLayoutParams;

	public DragListViewHelper(ListView listView) {
		mListView = listView;
		mWindowManager = getWindowManager();
	}

	public void startDrag(int positionFrom, View itemView) {
		// 前回使用した ImageView が残っている場合は除去（念のため？）
		if (mDragImageView != null) {
			mWindowManager.removeView(mDragImageView);
		}

		// ImageView 用の LayoutParams が未設定の場合は設定する
		if (mLayoutParams == null) {
			initLayoutParams();
		}

		// ImageView を生成し WindowManager に addChild する
		mDragImageView = createImageView(itemView);
		mWindowManager.addView(mDragImageView, mLayoutParams);
	}

	public void duringDrag(int x, int y) {
		if (mDragImageView == null) {
			return;
		}

		// ImageView の表示や位置を更新
		if (mDragImageView.getHeight() < 0) {
			mDragImageView.setVisibility(View.INVISIBLE);
		} else {
			mDragImageView.setVisibility(View.VISIBLE);
		}
		updateLayoutParams(x, y);
		getWindowManager().updateViewLayout(mDragImageView, mLayoutParams);
	}

	/**
	 * ドラッグ項目の描画を終了する
	 */
	public void stopDrag() {
		if (mDragImageView == null) {
			return;
		}

		getWindowManager().removeView(mDragImageView);
		mDragImageView = null;
	}

	/**
	 * WindowManager の取得
	 */
	private WindowManager getWindowManager() {
		return (WindowManager) mListView.getContext().getSystemService(
				Context.WINDOW_SERVICE);
	}

	/**
	 * ドラッグ中の項目を表すビューを作成する
	 */
	private ImageView createImageView(View itemView) {
		ImageView imageView = new ImageView(mListView.getContext());
		Bitmap bitmap = Bitmap.createBitmap(itemView.getWidth(),
				itemView.getHeight(), DRAG_BITMAP_CONFIG);
		Canvas canvas = new Canvas();
		canvas.setBitmap(bitmap);
		itemView.draw(canvas);
		imageView.setImageBitmap(bitmap);
		imageView.setBackgroundColor(BACKGROUND_COLOR);
		return imageView;
	}

	/**
	 * ImageView 用 LayoutParams の初期化
	 */
	private void initLayoutParams() {
		mLayoutParams = new WindowManager.LayoutParams();
		mLayoutParams.gravity = Gravity.TOP | Gravity.LEFT;
		mLayoutParams.height = WindowManager.LayoutParams.WRAP_CONTENT;
		mLayoutParams.width = WindowManager.LayoutParams.WRAP_CONTENT;
		mLayoutParams.flags = WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
				| WindowManager.LayoutParams.FLAG_NOT_TOUCHABLE
				| WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON
				| WindowManager.LayoutParams.FLAG_LAYOUT_NO_LIMITS;
		mLayoutParams.format = PixelFormat.TRANSLUCENT;
		mLayoutParams.windowAnimations = 0;
		mLayoutParams.x = mListView.getLeft();
		mLayoutParams.y = mListView.getTop();
	}

	/**
	 * ImageView 用 LayoutParams の座標情報を更新
	 */
	private void updateLayoutParams(int x, int y) {
		// ドラッグ中であることが分かるように少し上にずらす
		mLayoutParams.y = mListView.getTop() + y - Y_GAP;
	}

}
