/*
 * created  2008/06/14
 * Copyright 2008 yhj All rights reserved.
 */
package jp.que.ti.yhj.less.than.more;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;

import jp.que.ti.yhj.less.than.more.command.IllegalPositionException;

/**
 * テキストファイル参照機能を実現するためのロジッククラス。<br>
 * 大容量ファイルを参照することを前提とし、テキストファイル中の一部を参照する機能を実現します。
 *
 * @author yhj
 * @author $Author$ (last modified)
 * @version $Revision$
 */
public class TextViewer {
	/** デフォルトの読み込み byte */
	public static final int DEFAULT_READING_LENGTH = 30 * 128;
	public static final String UTF_8 = "UTF-8";
	public static final String UTF_16LE = "UTF-16LE";
	public static final String UTF_16BE = "UTF-16BE";
	public static final String UTF_16 = "UTF-16";
	public static final String ISO_8859_1 = "ISO-8859-1";
	public static final String US_ASCII = "US-ASCII";
	public static final String SHIFT_JIS = "SJIS";
	public static final String EUC_JP = "EUC-JP";

	private long position;
	private long lastReadLength;
	private int readingLength = DEFAULT_READING_LENGTH;
	private File file;
	private String encoding = Charset.defaultCharset().name();
	private String lastViewCache = "";

	private byte[] _byteTemp4view = new byte[1024];
	private byte[] _byteTemp4viewToCrLf = new byte[1024];

	private byte[] extendBytesIfTooSmall(byte[] bytes, int newLength) {
		byte[] newTemp = null;
		if (bytes.length > newLength) { // 格納可能なので問題なし
			return bytes;
		} else { // 格納スペースがないため拡張
			newTemp = new byte[newLength * 2];
			System.arraycopy(bytes, 0, newTemp, 0, bytes.length);
			return newTemp;
		}
	}

	/**
	 * encoding を取得します。
	 *
	 * @return エンコード
	 */
	public final String getEncoding() {
		return encoding;
	}

	/**
	 * 読み込んだ最終位置を取得します。
	 *
	 * @return 読み込んだ最終位置
	 */
	public final long getEndPosition() {
		return getPosition() + getLastReadLength();
	}

	/**
	 * 読み込み対象のファイル を取得します。
	 *
	 * @return 読み込み対象のファイル
	 */
	public final File getFile() {
		return file;
	}

	/**
	 * 最後に発行された {@link #view()}実行結果で読み込んだ長さを取得します
	 *
	 * @return 最後に発行された {@link #view()}実行結果で読み込んだ長さ
	 */
	public final long getLastReadLength() {
		return lastReadLength;
	}

	/**
	 * 最後に発行された {@link #view()}実行結果をキャッシュした文字列を返却します。
	 *
	 * @return 最後に発行された {@link #view()}実行結果をキャッシュした文字列
	 */
	public final String getLastViewCache() {
		return lastViewCache;
	}

	/**
	 * 現在のアドレス位置 を取得します。
	 *
	 * @return 現在のアドレス位置
	 */
	public final long getPosition() {
		return position;
	}

	/**
	 * 読み込み byte 数 を取得します。
	 *
	 * @return 読み込み byte 数
	 */
	public final int getReadingLength() {
		return readingLength;
	}

	/**
	 * 改行文字の次の文字から、引数 endPosition までのbyte 配列を返却します
	 *
	 * @param endPosition
	 *            読み込み最後尾位置
	 * @return 改行文字の次の文字から、引数 endPosition までのbyte 配列
	 */
	public byte[] readLineBackward(long endPosition) {
		BackwardByteReader reader = new BackwardByteReader(getFile());
		byte[] rtn = null;
		try {
			rtn = reader.readLine(endPosition);
		} finally {
			try {
				reader.close();
			} catch (IOException e) {
				throw new RuntimeException("ファイルクローズに失敗しました", e);
			}
		}
		return rtn;
	}

	/**
	 * 引数 startPosition から、改行文字までのbyte 配列を返却します
	 *
	 * @param startPosition
	 *            読み込み先頭位置
	 * @return 引数 startPosition から、改行文字までのbyte 配列
	 */
	public byte[] readLineForward(long startPosition) {
		ForwardByteReader reader = new ForwardByteReader(getFile());
		byte[] rtn = null;
		try {
			rtn = reader.readToCrLf(startPosition);
		} finally {
			try {
				reader.close();
			} catch (IOException e) {
				throw new RuntimeException("ファイルクローズに失敗しました", e);
			}
		}
		return rtn;
	}

	private byte[] readToCrLf(long pos, ByteReader reader) {
		reader.setPosition(pos);
		int dataIdx = 0;
		int lastCRLFPos = -1;
		boolean notEnd = true;
		while (notEnd) {
			int tmp = 0;
			tmp = reader.read();

			if (tmp < 0) {
				notEnd = false;
			} else {
				// _byteTemp4viewToCrLf サイズが足りなければ拡張
				_byteTemp4viewToCrLf = extendBytesIfTooSmall(
						_byteTemp4viewToCrLf, dataIdx);
				final byte tempByte = (byte) tmp;
				_byteTemp4viewToCrLf[dataIdx] = tempByte;
				if (reader.isForward()) {
					if (tempByte == '\n') {// ラインフィード
						lastCRLFPos = dataIdx;
					} else if (dataIdx - 1 >= 0 //
							&& _byteTemp4viewToCrLf[dataIdx - 1] == '\r') {// キャリッジリターン
						lastCRLFPos = dataIdx - 1;
					}
				} else {
					if (tempByte == '\r') {// キャリッジリターン
						lastCRLFPos = dataIdx;
					} else if (dataIdx - 1 >= 0 //
							&& _byteTemp4viewToCrLf[dataIdx - 1] == '\n') {// ラインフィード
						lastCRLFPos = dataIdx - 1;
					}
				}
				if (lastCRLFPos < 0) {// 改行が1回も出現していない
					// 終了しない
				} else {
					notEnd = false; // 読み込み終了
				}
				dataIdx++;
			}
		}

		int copyLength = 0;
		if (lastCRLFPos >= 0) {
			copyLength = lastCRLFPos + 1;
		} else {
			copyLength = dataIdx;
		}

		byte[] tmpBytes = new byte[copyLength];
		System.arraycopy(_byteTemp4viewToCrLf, 0, tmpBytes, 0, copyLength);
		return tmpBytes;

	}

	/**
	 * エンコード を設定します。
	 *
	 * @param encoding
	 *            エンコード
	 */
	public final void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	/**
	 * 読み込み対象のファイル を設定します。
	 *
	 * @param file
	 *            読み込み対象のファイル
	 */
	public final void setFile(File file) {
		this.file = file;
	}

	final void setLastReadLength(long lastReadLength) {
		this.lastReadLength = lastReadLength;
	}

	private final void setLastViewCache(String lastViewCache) {
		this.lastViewCache = lastViewCache;
	}

	/**
	 * 現在のアドレス位置 を設定します。
	 *
	 * @param position
	 *            現在のアドレス位置
	 */
	public final void setPosition(long position) {
		this.position = position;
	}

	/**
	 * 読み込み byte 数 を設定します。
	 *
	 * @param readingLength
	 *            読み込み byte 数
	 */
	public final void setReadingLength(int readingLength) {
		this.readingLength = readingLength;
	}

	/**
	 * ファイルの内容を現在のアドレス位置から読み込みサイズ分だけ読み込み、読み込み文字列を返却します。
	 *
	 * @return 読み込み文字列
	 */
	public String view() {
		ByteReader reader = null;
		byte[] record = null;
		int dataLen = 0;

		try {
			reader = new ForwardByteReader(getFile());
			boolean notEnd = true;
			record = readToCrLf(getPosition(), reader);
			int tmpLen = dataLen + record.length;
			if (tmpLen <= getReadingLength()) {
			} else {
				notEnd = false;
			}

			// 内部byteバッファへの格納
			_byteTemp4view // サイズが小さい場合バッファ拡張
			= extendBytesIfTooSmall(_byteTemp4view, tmpLen);
			System.arraycopy(record, 0//
					, _byteTemp4view, dataLen, record.length);
			dataLen = dataLen + record.length;

			long fileLastPos = getFile().length() - 1;
			if (fileLastPos >= getPosition() + dataLen) {
			} else {
				notEnd = false;
			}

			while (notEnd) {
				record = readToCrLf(getPosition() + dataLen, reader);
				tmpLen = dataLen + record.length;
				if (tmpLen <= getReadingLength()) {

					// 内部byteバッファへの格納
					_byteTemp4view // サイズが小さい場合バッファ拡張
					= extendBytesIfTooSmall(_byteTemp4view, tmpLen);
					System.arraycopy(record, 0//
							, _byteTemp4view, dataLen, record.length);
					dataLen = dataLen + record.length;

					if (fileLastPos >= getPosition() + dataLen) {
					} else {
						notEnd = false;
					}
				} else {
					notEnd = false;
				}
			}

		} finally {
			try {
				reader.close();
			} catch (IOException e) {
				throw new RuntimeException("ファイルクローズに失敗しました", e);
			}
		}

		byte[] tmpBytes = new byte[dataLen];
		System.arraycopy(_byteTemp4view, 0, tmpBytes, 0, dataLen);

		try {
			setLastViewCache(new String(tmpBytes, getEncoding()));
			setLastReadLength(tmpBytes.length);
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e.getMessage(), e);
		}

		return getLastViewCache();
	}

	/**
	 * ファイルの内容を現在のアドレス位置から読み込みサイズ分だけ読み込み、読み込み文字列を返却します。
	 *
	 * @return 読み込み文字列
	 */
	public String viewNextPage() {
		if (getFile() == null) {
			final String msg = "file is null";
			throw new NullPointerException(msg);
		}
		final long pos = getEndPosition();
		if (pos >= getFile().length()) {
			final String msg = "ファイルの終端に達しています。position="//
					+ pos + " : fileLength=" + getFile().length()
					+ " file="
					+ getFile().getAbsolutePath();
			throw new IllegalPositionException(msg);
		}
		setPosition(pos);
		view();
		return getLastViewCache();
	}
}
