package jp.que.ti.yhj.utils.file;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * ファイルに関するユーティリティ
 * 
 * @author Hajime.Yanagawa
 * @author $Author: yanagawa.hajime $ (last modified)
 * @version $Revision: $
 */
public class FileUtils {
	private FileUtils() {
	}

	private static final Pattern YEN_PATTERN = Pattern.compile("\\\\");

	/**
	 * 相対パスを文字列で返却する
	 * 
	 * @param baseDir
	 *            相対パスの基準となるベースディレクトリ
	 * @param file
	 *            対象のファイルまたは、ディレクトリ
	 * @return 相対パスの文字列
	 * @throws IllegalArgumentException
	 *             引数に null が設定されている場合スローされる
	 */
	public static String relativePath(File baseDir, File file) {
		if (file == null) {
			throw new IllegalArgumentException("引数 file が設定されていません");
		}
		if (baseDir == null) {
			throw new IllegalArgumentException("引数 baseDir が設定されていません");
		}

		final int len = (baseDir.getAbsolutePath()).length();
		final String longPath = file.getAbsolutePath();
		String shortPath = longPath.substring(len);
		if (shortPath.indexOf("\\") < 0) { // 円マークが存在しない場合
		} else { // 円マークが存在する場合
			shortPath = YEN_PATTERN.matcher(shortPath).replaceAll("/");
		}
		if (shortPath.indexOf("/") == 0) { // 先頭が "/" の場合
			shortPath = shortPath.substring(1);
		}
		return shortPath;
	}

	/**
	 * 引数 file から、 {@link FileOutputStream}オブジェクトを生成する。<br>
	 * 引数 file の親ディレクトリがない場合はディレクトリを生成する。<br>
	 * 当メソッドは、引数 file がディレクトリであることを考慮しない。呼び出し元でファイルを指定すること。
	 * 
	 * @param file
	 *            生成対象のファイル
	 * @return 生成した {@link FileOutputStream}オブジェクト
	 */
	public static FileOutputStream createFileOutputStream(File file) {
		File parentFile = file.getParentFile();
		if (parentFile != null) {
			if (parentFile.exists()) {
			} else {
				parentFile.mkdirs();
			}
		}
		FileOutputStream fos = null;
		try {
			fos = new FileOutputStream(file);
		} catch (FileNotFoundException e) {
			throw new RuntimeException("出力ファイルを開けませんでした:"
					+ file.getAbsolutePath(), e);
		}
		return fos;
	}

	/** すべてのファイルを必要と判断する {@link FileFilter}オブジェクト */
	public static final FileFilter NEED_ALL_FILTER = new FileFilter() {
		public boolean isNeed(FilteredFile file) {
			return true;
		}
	};

	/**
	 * 引数 fromFileOrDir の配下のに含まれるファイルおよびディレクトリの一覧を取得する
	 * 
	 * @param fromFileOrDir
	 * @return 引数 fromFileOrDir の配下のに含まれるファイルおよびディレクトリの一覧
	 */
	public static List<File> fileListWithDir(File fromFileOrDir) {
		return fileListWithDir(fromFileOrDir, NEED_ALL_FILTER);
	}

	/**
	 * 引数 fromFileOrDir の配下のに含まれるファイルおよびディレクトリの一覧を取得する
	 * 
	 * @param fromFileOrDir
	 *            ファイルまたはディレクトリ
	 * @param filter
	 *            返却する一覧に対するフィルター
	 * @return 引数 fromFileOrDir の配下のに含まれるファイルおよびディレクトリの一覧
	 */
	public static List<File> fileListWithDir(File fromFileOrDir,
			FileFilter filter) {
		return fileList(fromFileOrDir, true, filter);
	}

	/**
	 * 引数 fromFileOrDir の配下のに含まれるファイルの一覧を取得する
	 * 
	 * @param fromFileOrDir
	 * @return 引数 fromFileOrDir の配下のに含まれるファイルの一覧
	 */
	public static List<File> fileListWithNoDir(File fromFileOrDir) {
		return fileListWithNoDir(fromFileOrDir, NEED_ALL_FILTER);
	}

	/**
	 * 引数 fromFileOrDir の配下のに含まれるファイルの一覧を取得する
	 * 
	 * @param fromFileOrDir
	 *            ファイルまたはディレクトリ
	 * @param filter
	 *            返却する一覧に対するフィルター
	 * @return 引数 fromFileOrDir の配下のに含まれるファイルの一覧
	 */
	public static List<File> fileListWithNoDir(File fromFileOrDir,
			FileFilter filter) {
		return fileList(fromFileOrDir, false, filter);
	}

	private static final List<File> NULL_LIST = Collections
			.unmodifiableList(new ArrayList<File>(0));

	private static List<File> fileList(//
			File fromFileOrDir, boolean withDir, FileFilter filter) {
		List<File> rtn = new LinkedList<File>();

		FilteredLocalFile ffile = new FilteredLocalFile(fromFileOrDir);
		if (filter.isNeed(ffile)) {
		} else {// フィルターで不要と判断される場合は、空のリストを返却
			return NULL_LIST;
		}
		if (fromFileOrDir.isFile()) {
			rtn.add(fromFileOrDir);
		} else {
			if (withDir) {
				rtn.add(fromFileOrDir);
			}

			File[] childs = fromFileOrDir.listFiles();
			if (childs == null) {// null 以外の場合処理する
			} else {
				for (int idx = 0; idx < childs.length; idx++) {
					final File child = childs[idx];
					final List<File> grandchilds = fileList(child, withDir,
							filter);
					rtn.addAll(grandchilds);
				}
			}
		}
		return rtn;
	}
}
