/*
 * created  2008/07/22
 * Copyright 2008 yhj All rights reserved.
 */
package jp.que.ti.yhj.less.than.more;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.que.ti.yhj.less.than.more.command.IllegalPositionException;

/**
 * {@link java.io.InputStream} で実現できない逆方向ファイル読み込みを実現するためのクラスです
 *
 * @author yhj
 * @author $Author$ (last modified)
 * @version $Revision$
 */
public class BackwardByteReader extends ByteReaderBase {
	final private Logger log = Logger.getLogger(getClass().getName());

	private static final byte[] ZERO_BYTE_ARRAY = new byte[0];
	private File file;

	private FileChannel fileChannel;
	private long position = 0;
	private byte[] bufferedByts = ZERO_BYTE_ARRAY;
	private int bufferPosition = -1;

	private int bufferAllocateSize = 1024;

	/**
	 * コンストラクタ
	 *
	 * @param fileChannel
	 */
	public BackwardByteReader(File file) {
		if (file == null) {
			throw new RuntimeException("file is null. ファイルが指定されていません。");
		} else {
			setFile(file);
		}
		if (getFile().isDirectory()) {
			throw new RuntimeException("file is directory. ディレクトリは指定できません");
		}
		RandomAccessFile raf = null;

		try {
			raf = new RandomAccessFile(getFile(), "r");
		} catch (FileNotFoundException e) {
			throw new RuntimeException("ファイルが存在しません。getFile()="
					+ getFile().getAbsolutePath(), e);
		}
		setFileChannel(raf.getChannel());
		setPosition(getFile().length() - 1);
		setBufferPosition(-1);
	}

	/**
	 * {@link #getFileChannel()} で取得したオブジェクトに対して {@link FileChannel#close()}
	 * を呼び出し、リソースを開放します
	 */
	@Override
	public void close() throws IOException {
		if (getFileChannel().isOpen()) {
			getFileChannel().close();
		}
	}

	private final int getBufferAllocateSize() {
		return bufferAllocateSize;
	}

	private final byte[] getBufferedByts() {
		return bufferedByts;
	}

	private final int getBufferPosition() {
		return bufferPosition;
	}

	@Override
	public final File getFile() {
		return file;
	}

	private final FileChannel getFileChannel() {
		return fileChannel;
	}

	/**
	 * 次回呼び出される {@link #read()} により取得するアドレスを取得します。
	 *
	 * @return 次回呼び出される {@link #read()} により取得するアドレス
	 */
	@Override
	public final long getPosition() {
		return position;
	}

	/**
	 * 後方方向であることを意味する false を返却します。
	 *
	 * @return false を返却します。
	 * @see jp.que.ti.yhj.less.than.more.ByteReader#isForward()
	 */
	@Override
	final public boolean isForward() {
		return false;
	}

	/**
	 * このメソッドは現在の位置から逆方向に1byte読み込みます。<br>
	 * 値のバイトは、0 〜 255 の範囲の int として返されます。ストリームの終わりに達したために読み込むバイトがない場合は、値 -1
	 * が返されます。 入力データが読み込めるようになるか、ファイルの終わりが検出されるか、または例外が発生するまで、このメソッドはブロックされます。<br>
	 *
	 * @return データの次のバイト。ストリームの終わりに達した場合は -1
	 */
	@Override
	public int read() {
		if (getPosition() < 0) {
			return -1;
		}

		if (getBufferPosition() < 0) {// バッファの残りがなければバッファに読み込み
			setBufferedByts(readToBuffer(getPosition()));
			setBufferPosition(getBufferedByts().length - 1);
		}
		final byte rtn = getBufferedByts()[getBufferPosition()];
		setBufferPosition(getBufferPosition() - 1);
		setPositionOnly(getPosition() - 1);
		final int rtnInt = rtn & 0x00ff;
		return rtnInt;
	}

	private byte[] readToBuffer(long endPosition) {
		if (endPosition < 0) {
			throw new FileEdgeException("ファイルの先端に達しています。引数 endPosition="
					+ endPosition);
		}

		long startPos = endPosition - getBufferAllocateSize() + 1;
		if (startPos < 0) {
			startPos = 0;
		}
		int maxReadLen = (int) (endPosition - startPos + 1);

		ByteBuffer buf = ByteBuffer.allocate(maxReadLen);
		buf.clear();
		try {
			getFileChannel().position(startPos);
		} catch (IOException e) {
			throw new RuntimeException(e.getMessage(), e);
		}
		int readLen;
		try {
			readLen = getFileChannel().read(buf);
		} catch (IOException e) {
			throw new RuntimeException(e.getMessage(), e);
		}
		buf.position(0);
		byte[] nowBytes = new byte[readLen];

		buf.get(nowBytes);
		if (log.isLoggable(Level.FINEST)) {
			final String msg = "startPos=" + startPos + ":maxReadLen="
					+ maxReadLen + ":endPos=" + endPosition + ":readLen="
					+ readLen;
			log.finest(msg);
			log.finest("readToBuffer=" + new String(nowBytes));
		}
		return nowBytes;
	}

	public final void setBufferAllocateSize(int bufferAllocateSize) {
		this.bufferAllocateSize = bufferAllocateSize;
	}

	private final void setBufferedByts(byte[] bufferdByts) {
		this.bufferedByts = bufferdByts;
	}

	private final void setBufferPosition(int bufferdPosition) {
		this.bufferPosition = bufferdPosition;
	}

	private final void setFile(File file) {
		this.file = file;
	}

	private final void setFileChannel(FileChannel fileChannel) {
		this.fileChannel = fileChannel;
	}

	/**
	 * 次回呼び出される {@link #read()} により取得するアドレスを設定します。
	 *
	 * @param position
	 *            次回呼び出される {@link #read()} により取得するアドレス
	 */
	@Override
	public final void setPosition(long position) {
		if (getFile().length() <= position) {
			final String msg = //
			"ファイルの終了ポジションを越えています。position=" + position
					+ ": ファイルのポジションは0から始まりファイルサイズ (" //
					+ (getFile().length()) + ") までです。";
			throw new IllegalPositionException(msg);
		}

		final long gap = position - getPosition();
		final long newBufPos = getBufferPosition() + gap;
		if (newBufPos < getBufferAllocateSize()//
				&& newBufPos >= 0) {
			setBufferPosition((int) newBufPos);
		} else {
			setBufferPosition(-1);
		}

		setPositionOnly(position);
	}

	/**
	 * 次回呼び出される {@link #read()} により取得するアドレスを設定します。 このメソッドは、内部バッファーをクリアしません。
	 *
	 * @param position
	 *            次回呼び出される {@link #read()} により取得するアドレス
	 */
	private final void setPositionOnly(long position) {
		this.position = position;
	}

	/**
	 * 改行文字の次の文字から、引数 endPosition までのbyte 配列を返却します
	 *
	 * @param endPosition
	 *            読み込み最後尾位置
	 * @return 改行文字の次の文字から、引数 endPosition までのbyte 配列
	 */
	@Override
	public byte[] readLine(long endPosition) {
		byte[] rtn1 = null;
		byte[] rtn2 = BYTES_NODATA;
		byte[] rtnCrLf = null;
		byte[] rtnRecord = null;
		int rtnCrlfLength = 0;
		int rtnRecordLength = 0;

		rtn1 = readToCrLf(endPosition);

		if (isCrLfOnly(rtn1)) {
			rtnCrLf = rtn1;
			rtn2 = readToCrLf(endPosition - rtn1.length);
			rtnRecord = rtn2;
		} else {
			rtnCrLf = BYTES_NODATA;
			rtnRecord = rtn1;
		}
		rtnCrlfLength = rtnCrLf.length;
		rtnRecordLength = rtnRecord.length;

		if (rtnRecordLength > 0 && rtnRecord[rtnRecordLength - 1] == '\r') {
			rtnRecordLength--;
		}
		if (rtnRecordLength > 0 && rtnRecord[rtnRecordLength - 1] == '\n') {
			rtnRecordLength--;
		}
		if (rtnRecord.length == rtnRecordLength) {
		} else {
			byte[] tmpBytes = new byte[rtnRecordLength];
			System.arraycopy(rtnRecord, 0, tmpBytes, 0, rtnRecordLength);
			rtnRecord = tmpBytes;
		}
		int rtnLength = rtnCrlfLength + rtnRecordLength;
		byte[] rtn = new byte[rtnLength];

		if (rtnCrlfLength != 0) {
			System.arraycopy(rtnCrLf, 0, rtn, 0, rtnCrlfLength);
		}
		if (rtnRecordLength != 0) {
			System.arraycopy(rtnRecord, 0, rtn, rtnCrlfLength, rtnRecordLength);
		}

		// log start
		if (log.isLoggable(Level.FINEST)) {
			String msg = null;
			msg = "\n:rtn1=" + new String(rtn1)//
					+ "\n:rtn2=" + new String(rtn2)//
					+ "\n:rtnCrLf=" + new String(rtnCrLf)//
					+ "\n:rtnRecord=" + new String(rtnRecord)//
					+ "\n:end";
			log.finest(msg);
		} // log end

		return reverse(rtn);
	}

	/**
	 * 引数のbyte配列の並び順を逆にして返却します。
	 *
	 * @param bt
	 *            byte配列
	 * @return 並び順を逆にしたbyte配列
	 */
	public byte[] reverse(byte[] bt) {
		final int length = bt.length;
		byte[] rtn = new byte[length];
		for (int i = 0; i < length; i++) {
			final int pos = length - i - 1;
			rtn[pos] = bt[i];
		}
		return rtn;
	}

	private boolean isCrLfOnly(byte[] rtnBytes) {
		if (rtnBytes.length == 1 //
				&& (rtnBytes[0] == '\n' || rtnBytes[0] == '\r')//
		) {
			return true;
		} else if (rtnBytes.length == 2 //
				&& (rtnBytes[0] == '\n' && rtnBytes[1] == '\r')//
		) {
			return true;
		} else {
			return false;
		}
	}
}
