package jp.que.ti.yhj.utils.file;

import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 正規表現で不要なファイルのルールを設定し、ファイルの要不要を判定するフィルタークラス
 * 
 * @author Hajime.Yanagawa
 * 
 */
public class UnnecessaryLocalFileFilter implements FileFilter {
	private static final Logger LOGGER = Logger
			.getLogger(UnnecessaryLocalFileFilter.class.getName());

	private List<Pattern> directoryPatterns = new LinkedList<Pattern>();

	private List<Pattern> filePatterns = new LinkedList<Pattern>();

	/**
	 * 
	 * @see jp.que.ti.yhj.utils.file.FileFilter#isNeed(jp.que.ti.yhj.utils.file.FilteredFile)
	 */
	public boolean isNeed(FilteredFile file) {
		List<Pattern> ptns = null;
		if (file.isDirectory()) {
			ptns = this.directoryPatterns;
		} else {
			ptns = this.filePatterns;
		}
		for (Pattern pattern : ptns) {
			final Matcher mt = pattern.matcher(file.getName());
			if (LOGGER.isLoggable(Level.INFO)) {
				LOGGER.info(mt + "::" + file.getName() + "::" + mt.matches());
			}
			if (mt.matches()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * 不要なディレクトリの一覧を、正規表現パターンで指定する。
	 * 
	 * @param patterns
	 *            不要ディレクトリを示す正規表現パターンの一覧。
	 */
	private void setDirectoryRegularExpressions(String... patterns) {
		this.directoryPatterns = new LinkedList<Pattern>();
		if (patterns == null) {
			return;
		}
		for (String pattern : patterns) {
			if ("".equals(pattern)) {
			} else {
				this.directoryPatterns.add(Pattern.compile(pattern));
			}
		}
	}

	/**
	 * 不要なディレクトリの一覧を、平易な文字列表現で指定する。
	 * 
	 * @param patterns
	 *            不要ディレクトリを示す平易な文字列表現の一覧。ワイルドカードに*を使用できる
	 */
	public void setDirectorySimpleRegularExpressions(String... patterns) {
		final String[] paramPtns = simpleRegularExpressions(patterns);
		setDirectoryRegularExpressions(paramPtns);
	}

	private static final String[] NULL_STRING_ARRAY = new String[0];

	private String[] simpleRegularExpressions(String... patterns) {
		if (patterns == null) {
			return NULL_STRING_ARRAY;
		}
		String[] paramPtns = new String[patterns.length];
		for (int idx = 0; idx < patterns.length; idx++) {
			final char[] crs = patterns[idx].toCharArray();
			StringBuilder sb = new StringBuilder(crs.length * 3);
			sb.append("^");// 文字列の先頭
			for (char cr : crs) {
				if (cr == '*') {
					sb.append(".*");// ワイルドカード
				} else {
					sb.append("[");
					sb.append(cr);
					sb.append("]");
				}
			}
			sb.append("$");// 文字列の末尾
			paramPtns[idx] = sb.toString();
		}
		return paramPtns;
	}

	/**
	 * 不要なファイルの一覧を、正規表現パターンで指定する。
	 * 
	 * @param patterns
	 *            不要ファイルを示す正規表現パターンの一覧。ワイルドカードに*を使用できる
	 */
	private void setFileRegularExpressions(String... patterns) {
		this.filePatterns = new LinkedList<Pattern>();
		if (patterns == null) {
			return;
		}
		for (String pattern : patterns) {
			if ("".equals(pattern)) {
			} else {
				this.filePatterns.add(Pattern.compile(pattern));
			}
		}
	}

	/**
	 * 不要なファイルの一覧を、平易な文字列表現で指定する。
	 * 
	 * @param patterns
	 *            不要ファイルを示す平易な文字列表現の一覧。ワイルドカードに*を使用できる
	 */
	public void setFileSimpleRegularExpressions(String... patterns) {
		final String[] paramPtns = simpleRegularExpressions(patterns);
		setFileRegularExpressions(paramPtns);
	}
}
