#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include "ar.h"

static off_t
store(FILE *rfp, FILE *wfp, uint16_t *crc)
{
    size_t n;
    char buf[MAXDICSIZ];
    off_t sz = 0;

    *crc = INIT_CRC;
    while ((n = fread(buf, 1, sizeof(buf), rfp)) != 0) {
        fwrite_crc(buf, n, wfp, crc);
        sz += n;
    }
    return sz;
}

int
add_dir(struct lzh_ostream *wp, int replace_flag, struct lzh_header *h)
{
    h->origsize = h->compsize = 0;
    h->file_crc = INIT_CRC;

    write_header(wp->fp, h);

    if (opts.quiet < 2) {
        if (replace_flag)
            printf("Replacing %s/ ", h->filename);
        else
            printf("Adding %s/ ", h->filename);

        printf("(directory)\n");
    }
    return 1;                   /* success */
}

static int
add_1(struct lzh_ostream *wp, int replace_flag, struct lzh_header *h)
{
    long headerpos, arcpos, arcendpos;
    unsigned int r;
    FILE *rfp;

    if ((rfp = fopen(h->filename, "rb")) == NULL) {
        fprintf(stderr, "Can't open %s\n", h->filename);
        return 0;               /* failure */
    }
    if (opts.quiet < 2) {
        if (replace_flag)
            printf("Replacing %s ", h->filename);
        else
            printf("Adding %s ", h->filename);
    }

    headerpos = ftell(wp->fp);
    write_header(wp->fp, h);
    arcpos = ftell(wp->fp);

    wp->origsize = wp->compsize = 0;
    wp->crc = INIT_CRC;
    if (opts.nocompress) {
        wp->unpackable = 1;
    }
    else {
        wp->unpackable = 0;
        encode(wp, rfp);
    }

    if (wp->unpackable) {
        memcpy(h->method, "-lh0-", sizeof(h->method));  /* store */
        rewind(rfp);
        fseek(wp->fp, arcpos, SEEK_SET);
        h->compsize = h->origsize = store(rfp, wp->fp, &wp->crc);
    }
    else {
        h->compsize = wp->compsize;
        h->origsize = wp->origsize;
    }
    h->file_crc = wp->crc ^ INIT_CRC;
    fclose(rfp);

    /* rewrite compsize and origsize to the header */
    arcendpos = ftell(wp->fp);
    fseek(wp->fp, headerpos, SEEK_SET);
    write_header(wp->fp, h);
    fseek(wp->fp, arcendpos, SEEK_SET);

    r = ratio(wp->compsize, wp->origsize);
    if (opts.quiet < 2)
        printf(" %d.%d%%\n", r / 10, r % 10);
    return 1;                   /* success */
}

int
add(struct lzh_ostream *wp, int replace_flag, char *filename, int namelen)
{
    struct lzh_header h;
    struct stat st;

    memset(&h, 0, sizeof(h));

    h.level = opts.header_level;

    strcpy(h.filename, filename);
    h.namelen = namelen;

    stat(h.filename, &st);

    h.mtime = st.st_mtime;
    if (S_ISDIR(st.st_mode)) {
        /* directory */
        DIR *dir;
        struct dirent *ent;

        memcpy(h.method, "-lhd-", sizeof(h.method));  /* directory */
        add_dir(wp, replace_flag, &h);

        /* recursive call */
        dir = opendir(h.filename);
        if (dir == NULL)
            error("cannot open directory: \"%s\"", h.filename);

        while ((ent = readdir(dir)) != 0) {
            char filename[1024];

            if (string_equal(ent->d_name, ".") ||
                string_equal(ent->d_name, ".."))
                continue;

            h.namelen = path_addsep(h.filename, sizeof(h.filename));

            string_cat(filename, sizeof(filename),
                       h.filename, ent->d_name, NULL);

            add(wp, replace_flag, filename, strlen(filename));
        }
        closedir(dir);
    }
    else {
        /* regular file */
        memcpy(h.method, opts.method->id, sizeof(h.method));
        add_1(wp, replace_flag, &h);
    }

    return 0;
}
