/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
maskat.lang.Class.declare("maskat.dialog.rialto.RialtoAlertDialog")
	.extend("maskat.ui.AlertDialog", {

	/**
	 * Rialtoのアラートコンストラクタ
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param icon アイコン種別
	 * @param onClick ボタン押下時の呼び出し先関数名
	 */
	initialize: function (title, message, icon, onClick) {
		/* 引数を内部メンバとして設定 */
		this.base.apply(this, arguments);
		/* 改行文字列の置換 */
		this.message = this.message.replace( /\r?\n/g, "<br />\n");
		/* ウインドウサイズを取得 (最小サイズ幅:300 高:80)  */
		this.getOffset(300,80)
		/* ボタンの幅 */
		this.buttonWidth = 100;
	},

	/**
	 * RialtoのアラートをPOPUPして表示します。
	 *
	 */
	open: function () {
	
		/* 空ウィンドウの表示 */
		var oThis = this;
		var widthGlobal = document.body.clientWidth;
		var heightGlobal = document.body.clientHeight;
		var top = (heightGlobal / 2) - (this.height / 2);
		var left = (widthGlobal / 2) - (this.width / 2);
		this.width = getTailleTexte(this.message) + 40;
		if (this.width < 200) {
  			/* 最小サイズの設定 */
			this.width = 200;
		}
		this.popup = new rialto.widget.PopUp("popup", 
							top, 
							left, 
							this.width, 
							this.height, 
							"", 
							this.title, 
							"inherit",
							{withCloseButon:false}
							);
							
		this.divData = document.createElement("DIV");
		/* イメージの表示 */
		this.icon = this.icon ? this.icon : "";
		if (this.icon!="") {
			this.message = "<IMG src=\"" +
						maskat.location +
						this.icon +
						"\">" +
						" " +  
						this.message;
		}
		/* アラート内表示メッセージの作成 */
		this.divData.innerHTML = this.message;
		this.divData.style.position = "absolute";
		this.divData.style.overflow = "auto";
		this.popup.add(this.divData);
		/* [OK]ボタンの追加 */
		var posxOk = (this.width / 2) - (this.buttonWidth / 2);
		var hightButton = 25;
		var posyOk = this.height - hightButton;
		this.buttonClose = new rialto.widget.Button(posyOk, 
									posxOk, 
									maskat.ui.Dialog.OK, 
									maskat.ui.Dialog.OK
									);
		this.buttonClose.onclick = function () {
			oThis.popup.closeWindow();
			oThis.onClick();
		};
		this.popup.add(this.buttonClose);
	},

	/**
	 * ダイアログサイズを算出します
	 *
	 * @param minWidth 最小幅
	 * @param minHeight 最小高さ
	 */
	getOffset: function (minWidth, minHeight) {
		var div = document.createElement("DIV");
		div.innerHTML = this.message;
		div.style.position = "absolute";
		div.style.visibility = "hidden";
		div.style.whiteSpace = "nowrap";
		document.body.appendChild(div);
		this.width = div.offsetWidth < minWidth ? minWidth : div.offsetWidth + 0;
		this.width = this.width * 1.05;
		this.height = div.offsetHeight < minHeight ? minHeight : div.offsetHeight;
		this.height += 50;
		document.body.removeChild(div);
		div = null;
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
/**
 * Rialtoのダイアログ表示クラスのインスタンスを生成するためのクラスです。
 *
 */
maskat.lang.Class.declare("maskat.dialog.rialto.RialtoDialogFactory")
	.extend("maskat.ui.DialogFactory", {
	
	_static: {
		/**
		 * このクラスの唯一のインスタンスを返します。(Singleton パターン)
		 *
		 * @return このクラスの唯一のインスタンス
		 */
		getInstance: function(){
			var self = arguments.callee;
			if (!self.instance) {
				self.instance = new this();
			}
			return self.instance;
		}
	},

	/**
	 * 確認ダイアログを表示するためのクラスを生成します。
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param icon アイコン種別
	 * @param onClick ボタン押下時の呼び出し先関数名
	 */
	createMessage: function (title, message, icon, onClick) {
		return new maskat.dialog.rialto.RialtoMessageDialog (title, message, icon, onClick);
	},
	
	/**
	 * アラートダイアログを表示するためのクラスを生成します。
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param onClick ボタン押下時の呼び出し先関数名
	 */
	createAlert: function (title, message, icon, onClick) {
		return new maskat.dialog.rialto.RialtoAlertDialog (title, message, icon, onClick);
	},
	
	/**
	 * 入力ダイアログを表示するためのクラスを生成します。
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param onClick ボタン押下時の呼び出し先関数名
	 */
	createInput: function (title, message, icon, onClick) {
		return new maskat.dialog.rialto.RialtoInputDialog (title, message, icon, onClick);
	},
	
	/**
	 * 進捗表示ダイアログを表示するためのクラスを生成します。
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param goalValue 終了進捗値(オプション)
	 */
	createProgress: function (title, message, goalValue) {
		return new maskat.dialog.rialto.RialtoProgressDialog (title, message, goalValue);
	}
	
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
maskat.lang.Class.declare("maskat.dialog.rialto.RialtoInputDialog")
	.extend("maskat.ui.InputDialog", {

	/**
	 * Rialtoの入力ダイアログコンストラクタ
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param icon アイコン種別
	 * @param onClick ボタン押下時の呼び出し先関数名
	 */
	initialize: function (title, message, icon, onClick) {
		/* 引数を内部メンバとして設定 */
		this.base.apply(this, arguments);
		/* 改行文字列の置換 */
		this.message = this.message.replace( /\r?\n/g, "<br />\n");
		/* ウインドウサイズを取得 (最小サイズ幅:400 高:100)  */
		this.getOffset (400, 50);
		/* ボタンの幅 */
		this.buttonWidth = 100;
	},

	/**
	 * Rialtoの入力ダイアログをPOPUPして表示します。
	 *
	 */
	open: function () {
		/* 戻り関数の存在チェック、無いときはEmpty関数をセット*/
		if (typeof(this.onClick) != "function") {
			this.onClick = maskat.lang.EmptyFunction;
		}
		/* 空ウィンドウの表示 */
		var oThis = this;
		var widthGlobal = document.body.clientWidth;
		var heightGlobal = document.body.clientHeight;
		var top = (heightGlobal / 3) - (this.height / 2);
		var left = (widthGlobal / 2) - (this.width / 2);
		this.popup = new rialto.widget.PopUp ("popup", 
								top, 
								left, 
								this.width, 
								this.height,
								"", 
								this.title, 
								"transparent", 
								{withCloseButon:false});  

		this.divData = document.createElement ("DIV");
		/* イメージの表示 */
		this.icon = this.icon ? this.icon : "";
		if (this.icon!="") {
			this.message = "<IMG src=\"" +
						maskat.location +
						this.icon +
						"\">" +
						" " +  
						this.message;
		}
		/* ダイアログ表示メッセージの作成 */
		this.divData.innerHTML = " " + this.message;
		this.popup.add (this.divData);
		/* 入力テキストボックスの作成 */
		this.TextBox = new rialto.widget.Text(this.name, 
								this.height - 60, 
								this.width * 0.01, 
								this.width * 0.9, 
								"A", 
								this.divExt
								);
		this.TextBox.popup = this.popup;
		this.popup.add(this.TextBox) ;
		this.TextBox.setFocus();
		/* [OK]ボタンの作成 */
		var posxOk = (this.width / 2) - this.buttonWidth;
		var posxCancel = posxOk + this.buttonWidth;
		var hBtn = this.height - 25;
		this.buttonOk = new rialto.widget.Button(hBtn,
								posxOk, 
								maskat.ui.Dialog.OK, 
								maskat.ui.Dialog.OK);
		this.buttonOk.onclick = function () {
			oThis.onClick(oThis.TextBox.getValue());
			oThis.popup.closeWindow();
		};
		this.buttonOk.popup = this.popup;
		this.popup.add(this.buttonOk);
		/* [CANCEL]ボタンの作成 */
		this.buttonCancel = new rialto.widget.Button(hBtn, 
									posxCancel, 
									maskat.ui.Dialog.CANCEL, 
									maskat.ui.Dialog.CANCEL);
		this.buttonCancel.onclick = function () {
			oThis.onClick(null);
			oThis.popup.closeWindow();
		};
		this.buttonCancel.popup = this.popup;
		this.popup.add(this.buttonCancel) ;
	},

	/**
	 * ダイアログサイズを算出します
	 *
	 * @param minWidth 最小幅
	 * @param minHeight 最小高さ
	 */
	getOffset: function (minWidth, minHeight) {
		var div = document.createElement("DIV");
		div.innerHTML = this.message;
		div.style.position = "absolute";
		div.style.visibility = "hidden";
		div.style.whiteSpace = "nowrap";
		document.body.appendChild(div);
		this.width = div.offsetWidth < minWidth ? minWidth : div.offsetWidth + 0;
		this.width = this.width * 1.05
		this.height = div.offsetHeight < minHeight ? minHeight : div.offsetHeight;
		this.height += 60;
		document.body.removeChild(div);
		div = null;
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.dialog.rialto.RialtoMessageDialog")
	.extend("maskat.ui.MessageDialog", {

	/**
	 * rialtoの確認ダイアログコンストラクタ
	 * 引数のtitle, iconは無視されます
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param icon アイコン種別
	 * @param onClick ボタン押下時の呼び出し先関数名
	 */
	initialize: function (title, message, icon, onClick) {
		/* 引数を内部メンバとして設定 */
		this.base.apply(this, arguments);
		/* 改行文字列の置換 */
		this.message = this.message.replace( /\r?\n/g, "<br />\n");
		/* ウインドウサイズを取得 (最小サイズ幅:400 高:50)  */
		this.getOffset (400, 50);
		/* ボタンの幅 */
		this.buttonWidth = 100;
		
	},

	/**
	 * rialtoの確認ダイアログをPOPUPして表示します。
	 *
	 */
	open: function () {
	
		/* ダイアログボックス表示 */
		var oThis = this;
		var widthGlobal = document.body.clientWidth;
		var heightGlobal = document.body.clientHeight;
		var top = (heightGlobal / 2) - (this.height / 2);
		var left = (widthGlobal / 2) - (this.width / 2);
		this.popup = new rialto.widget.PopUp("popup", 
							top, 
							left, 
							this.width, 
							this.height, 
							"", 
							this.title, 
							"transparent", 
							{withCloseButon:false}
							);  
		this.divData = document.createElement("DIV");
		/* イメージの表示 */
		this.icon = this.icon ? this.icon : "";
		if (this.icon!="") {
			this.message = "<IMG src=\"" +
						maskat.location +
						this.icon +
						"\">" +
						" " +  
						this.message;
		}
		/* ダイアログ表示メッセージの作成 */
		this.divData.innerHTML = this.message;
		this.divData.style.position = "absolute";
		this.divData.style.overflow = "auto";
		this.popup.add(this.divData);
		/* [OK]ボタンの追加 */
		var posxOk = ( this.width / 2 ) - this.buttonWidth;
		var posyCancel = posxOk + this.buttonWidth;
		var hightButton = this.height - 25;
    
		this.buttonYes = new rialto.widget.Button(hightButton, 
								posxOk, 
								maskat.ui.Dialog.OK, 
								maskat.ui.Dialog.OK
								);
		this.buttonYes.onclick = function () {
			oThis.popup.closeWindow();
			oThis.onClick(maskat.ui.Dialog.OK);
		};
		this.popup.add(this.buttonYes);
		/* [Cancel]ボタンの追加 */
		this.buttonNo = new rialto.widget.Button(hightButton, 
								posyCancel, 
								maskat.ui.Dialog.CANCEL, 
								maskat.ui.Dialog.CANCEL
								);
		this.buttonNo.onclick = function () {
			oThis.popup.closeWindow();
			oThis.onClick(maskat.ui.Dialog.CANCEL);
		};
		this.popup.add(this.buttonNo) ;

	},

	/**
	 * ダイアログサイズを算出します
	 *
	 * @param minWidth 最小幅
	 * @param minHeight 最小高さ
	 */
	getOffset: function (minWidth, minHeight) {
		var div = document.createElement("DIV");
		div.innerHTML = this.message;
		div.style.position = "absolute";
		div.style.visibility = "hidden";
		div.style.whiteSpace = "nowrap";
		document.body.appendChild(div);
		this.width = div.offsetWidth < minWidth ? minWidth : div.offsetWidth + 0;
		this.width = this.width * 1.05;
		this.height = div.offsetHeight < minHeight ? minHeight : div.offsetHeight;
		this.height += 50;
		document.body.removeChild(div);
		div = null;
	}
	
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
maskat.lang.Class.declare("maskat.dialog.rialto.RialtoProgressDialog")
	.extend("maskat.ui.ProgressDialog", {

	/**
	 * Rialtoの進捗ダイアログコンストラクタ
	 *
	 * @param title ダイアログのタイトル文字列
	 * @param message ダイアログメッセージ文字列
	 * @param total 終了進捗値(オプション) または、doneを実行するまで終了しない処理中表示(POSITIVE_INFINITY)
	 */
	initialize: function (title, message, total) {
	
		/* 引数を内部メンバとして設定 */
		this.base.apply(this, arguments);
		
		/* totalのデフォルト設定 */
		if (!this.total || this.total == "") {
			this.total = 100;
		}
		
		/* 表示タイプの判定 */
		this.endlessMode = false;
		if (this.total == Number.POSITIVE_INFINITY) {
			this.endlessMode = true;
			this.total = 100;
		}
		
		/* 改行文字列の置換 */
		if (typeof(this.message) == "string") {
			this.message = this.message.replace( /\r?\n/g, "<br />\n");
		}

		/* ウインドウサイズを取得 (最小サイズ幅:500 高:50)  */
		this.getOffset (250, 50);
		/* 進捗バーの幅 */
		this.barWidth = this.width * 0.8;
		/* カレントの進捗値 */
		this.count = 0;
		/* 進捗バーのElement ID */
		this.barElementId = "";
		this.oThis = this;
		
	},
 
 	/**
	 * Rialtoの進捗ダイアログをPOPUPして表示します。
	 *
	 */
	open: function () {
	
		/* 空ウィンドウのPOPUP */
		var oThis = this;
		var widthGlobal = document.body.clientWidth;
		var heightGlobal = document.body.clientHeight;
		var top = (heightGlobal / 2) - (this.height / 2);
		var left = (widthGlobal / 2) - (this.width / 2);
		this.popup = new rialto.widget.PopUp("popup",
								top, 
								left, 
								this.width, 
								this.height, 
								"", 
								this.title, 
								"transparent", 
								{withCloseButon:false}
								);         

		/* メッセージの表示 */
		var divData = document.createElement("DIV");
		divData.id = 'processingMessage';
		divData.innerHTML = this.message;
		this.popup.add(divData);
		
		/* 進捗バーの位置・サイズ算出 */
		var barHeight = 16; 
		var barTop = this.height - barHeight -10;
		var barLeft = this.width * 0.02;
		
		if (this.endlessMode) {
			
			/* 進捗バー表示枠の設定 */
			var barProgress = document.createElement ("div");
			barProgress.id = 'pbar';
			barProgress.style.top = barTop;
			barProgress.style.left = (this.width - this.barWidth) / 2 - 10;
			barProgress.style.position = "absolute";

			/* 進捗バーの設定 */
			var imagePath = maskat.location + "core/images/"
			var imgTag = document.createElement("img");
			imgTag.src = imagePath + "endless_progress.gif";
			imgTag.width = this.barWidth ;
			imgTag.height = 14;
			
			barProgress.appendChild(imgTag);
			this.popup.add(barProgress);
		
		} else {
		
			/* 進捗率の表示 */
			var rateMargine = 35;
			this.textRate = document.createElement("DIV");
			this.textRate.innerHTML = "0%";
			this.textRate.style.position = "absolute";
			this.textRate.style.top = barTop;
			this.textRate.style.left = barLeft;
			this.popup.add(this.textRate);

			/* 進捗バー枠の表示 */
			var barBorder = document.createElement("DIV");
			barBorder.style.top = barTop;
			barBorder.style.left = barLeft + rateMargine;
			barBorder.style.width = this.barWidth;
			barBorder.style.height = barHeight;
			barBorder.style.backgroundColor = "white"; 
			barBorder.style.borderColor = "black";
			barBorder.style.borderStyle = "solid";
			barBorder.style.borderWidth = "1px"; 
			barBorder.style.position = "absolute";
			this.popup.add(barBorder);

			/* 進捗バーの設定 */
			var barColor = "#003399";
			this.barProgress = document.createElement("DIV");
			this.barProgress.id = "progressBar";
			this.barProgress.style.top = barTop;
			this.barProgress.style.left = barLeft + rateMargine;
			this.barProgress.style.width = 0;
			this.barProgress.style.height = barHeight;
			this.barProgress.style.backgroundColor = barColor;
			this.barProgress.style.borderColor = barColor;
			this.barProgress.style.borderStyle = "solid";
			this.barProgress.style.borderWidth = "1px"; 
			this.barProgress.style.position = "absolute";
			this.popup.add(this.barProgress);
			return this;
			
		}
	},
	
	/**
	 * rialtoの進捗状況ウィンドウの進捗状況を更新します。
	 *
	 * @param count 進捗総数
	 */
	setProgress:function (count) {
		/* 更新対象の存在チェック */
		if (!this.oThis) {
			return;
		}
		if (!this.isNumeric(count)) {
		    return;
		}
		this.count = Number(count);
		/* トータルを超えている場合 */
		if (this.count > this.total) {
			this.count = this.total;
		}
		/* マイナス値の場合 */
		if (this.count < 0) {
			this.count = 0;
		}
		/* 表示の更新 */
		this.updateProgress();
		/* 終了時のClose */
		if (this.count >= this.total) {
			this.oThis.popup.closeWindow();
			this.oThis = null;
		}
	},

	/**
	 * rialtoの進捗状況ウィンドウの進捗状況を加算更新します。
	 *
	 * @param addValue 進捗数
	 */
	worked:function (addValue) {
		if (this.isNumeric(addValue)) {
		    this.setProgress (Number(addValue) + Number(this.count));
		} else {
		    return;
		}
	},

	/**
	 * 進捗状況ウィンドウを閉じます。
	 *
	 */
	done: function() {
		this.setProgress(this.total);
	},

	/**
	 * 進捗状況ウィンドウの進捗状況を更新します。
	 *
	 */
	updateProgress:function () {
	
		/* 進捗率表示無し */
		if (this.endlessMode) {
			return;
		}
		
		/* 更新対象の存在チェック */
		if (!this.oThis) {
		    return;
		}
		
		/* 進捗率の計算 */
		var rate = this.count * 100 / this.total;
		
		this.textRate.innerHTML = Math.round (rate) + "%";
		
		/* 進捗バーの更新 */
		this.barProgress.style.width
					= (this.barWidth * rate) / 100;
	},

	/**
	 * ダイアログサイズを算出します
	 *
	 * @param minWidth 最小幅
	 * @param minHeight 最小高さ
	 */
	getOffset: function (minWidth, minHeight) {
		var div = document.createElement("DIV");
		div.innerHTML = this.message;
		div.style.position = "absolute";
		div.style.visibility = "hidden";
		div.style.whiteSpace = "nowrap";
		document.body.appendChild(div);
		this.width = div.offsetWidth < minWidth ? minWidth : div.offsetWidth + 0;
		this.width = this.width * 1.05;
		this.height = div.offsetHeight < minHeight ? minHeight : div.offsetHeight;
		this.height += 50;
		document.body.removeChild(div);
		div = null;
	},
	
	/**
	 * 先頭にプラス・マイナスを含んだ数字であることをチェックします
	 *
	 */
	isNumeric: function (value) {
		if (typeof(value) != "string") { 
			return true;
		}
		return value.match(/[^+^-][\D]/g) ? false : true;
	},

	/**
	 * 表示中のメッセージを置き換えます
	 * 制限として、メッセージの長さによるウィンドウのリサイズは行いません
	 *
	 * @param newMessage 置き換え後のメッセージ
	 */
	replaceMessage: function (newMessage) {
	
		/* 更新対象の存在チェック */
		if (!this.oThis) {
			return;
		}
	
		if (typeof(newMessage) != "string") { 
			return;
		}

		newMessage = newMessage.replace( /\r?\n/g, "<br />\n"); 
		document.getElementById('processingMessage').innerHTML = newMessage;
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.AlertWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
    	var alertWrapper = new rialto.widget.Alert(this.mess);
    	
    	var self = this;
    	
    	alertWrapper.onclose = function() { self.notifyEvent("onclose"); };
		
		
		this.widget = alertWrapper;
		return alertWrapper;
    },

	setParent: function(parent) {
		if (this.parent == parent) {
			return;
		}
		this.parent = parent;
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.ButtonWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
    	var button = new rialto.widget.Button(
    		this.top,
    		this.left,
    		this.title,
    		this.alt,
    		null, /* parent */
    		this);

		var self = this;
		button.onclick = function() { self.notifyEvent("onclick"); };
    	
		this.widget = button;
		return button;
    },
    
    postCreateWidget: function() {
		this.widget.divExt.tabIndex = this.getTabIndex();
		this.addKeyEventListener(this.widget.divExt);
    },

	handleKeyEvent: function(event) {
		if (event.keyCode == 13 || event.keyCode == 32) {
			this.widget.onclick();
			return false;
		}
		return true;
	},
	
	setFocus: function() {
		this.widget.divExt.focus();
	}
});

/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.CheckboxWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
    	var checkbox = new rialto.widget.Checkbox(
            this.name,
            this.top,
            this.left,
            null, /* parent */
            this.text,
            this.checked,
			this.className);
		checkbox.setEnable(this.enable);

		var self = this;
		checkbox.onclick = function() { self.notifyEvent("onclick"); };

		this.widget = checkbox;
		return checkbox;
    },

	getControlElement: function() {
		return this.widget.checkbox;
	},

	getValue: function() {
		return this.widget.isCheck();
	},

	setValue: function(value) {
		if (value == "true") {
			this.widget.setCheck(true);
		} else if (value == "false") {
			this.widget.setCheck(false);
		}
	},
	
	clear: function() {
		this.widget.setCheck(false);
	},

    postCreateWidget: function() {
		this.widget.divExt.tabIndex = this.getTabIndex();
		this.addKeyEventListener(this.widget.divExt);
    },

	handleKeyEvent: function(event) {
		if (event.keyCode == 13 || event.keyCode == 32) {
			this.widget.setCheck(!this.widget.isCheck());
			this.setFocus();
			return false;
		}
		return true;
	},
	
	setFocus: function() {
		this.widget.divExt.focus();
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.CodeLibWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent) {
		var codeLib = new rialto.widget.codeLabel(
			this.name,
			this.top,
			this.left,
			this.width,
			null, /* parent */
			this);
		
		if(this.enable == false){
			codeLib.setEnable(false);
		}	
		
		this.widget = codeLib;
		return codeLib;
	},

    postCreateWidget: function() {
		this.widget.code.champs.tabIndex = this.getTabIndex();
		this.widget.img.divExt.tabIndex = this.getTabIndex();
		this.addKeyEventListener(this.widget.divExt);
    },

	getValue: function() {
		return this.widget.getValue();
	},

	setValue: function(value) {
		this.widget.setValue(value);
	},

	handleKeyEvent: function(event) {
		var element = event.target || event.srcElement;
		if (event.keyCode == 9) {
			if (element == this.widget.code.champs) {
				if (!event.shiftKey) {
					this.widget.img.divExt.focus();
					return false;
				}
			} else if (event.shiftKey) {
				this.widget.code.champs.focus();
				return false;
			}
		} else if (this.focusPos == 1 && event.keyCode == 13 || event.keyCode == 32) {
			this.widget.img.onclick();
			return false;
		}
		return true;
	},
	
	setFocus: function() {
		this.widget.code.champs.focus();
	},
	
	getControlElement: function(){
		// codeLabel の コード入力用要素を返却します。
		return this.widget.code.getHtmlExt().firstChild;
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.ComboWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {

	   	/* tabdataがnullの場合、空配列を設定する */
	   	if(typeof(this.tabData) == "undefined"){
    		this.tabData = new Array;
    	}

    	var combo = new rialto.widget.Combo(
    		this.tabData,
    		this.name,
    		this.top,
    		this.left,
    		this.width,
    		null, /* parent */
    		this);

		var self = this;
		combo.onclick = function() {
			self.setFocus();
			self.notifyEvent("onclick");
		};
    	combo.onblur = function() { self.notifyEvent("onblur"); };
    		
    	this.widget = combo;
    	return combo;
    },

    postCreateWidget: function() {
		this.widget.divExt.tabIndex = this.getTabIndex();
		this.addKeyEventListener(this.widget.divExt);
    },
	
	getControlElement: function(){
		return this.widget.text.getHtmlExt().firstChild;
	},

	getValue: function() {
		return this.widget.getSelValue();
	},

	setValue: function(value) {
		this.widget.selWithValue(value);
	},
	
	setFocus: function() {
		this.widget.text.divExt.childNodes[0].focus();
	}
});

/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.DivSplitWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent) {
		var splitter = parent.getWidget();

		var children = parent.getChildren();

		for( var cnt=0; cnt<children.length; cnt++ ){
			if(children[cnt] == this){
				var key = cnt + 1;
				this.div = splitter["div" + key];
				this.div.style.backgroundColor = this.backgroundColor;
			}
		}
		
		return null;
	},
	
	getElement: function() {
		return this.div;
	},

	setParent: function(parent){
		this.parent = parent;
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.FrameWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent){
		var frame = new rialto.widget.Frame(this);

		var self = this;
		frame.onSetDisplay = function() { self.notifyEvent("onSetDisplay"); };
    		
		this.widget = frame;
	}

});
maskat.lang.Class.declare("maskat.widget.rialto.GridWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
		this.TabEntete = [];
		this.tabTypeCol = [];
    
    	if (this.headers) {
    		for (var i = 0; i < this.headers.length; i++) {
    			var header = this.headers[i];
    			this.TabEntete.push(header.title);
    			this.tabTypeCol.push([header.type, header.width, header.tabData]);
    		}
    	}

		var grid = new rialto.widget.Grid(this);

		if(typeof(this.onCellEdit) != "undefined"){
	    	grid.onCellEdit = this.onCellEdit;
	    }
		
		var self = this;
		grid.onclick = function() { self.notifyEvent("onclick"); };
		grid.ondbleclick = function() { self.notifyEvent("ondbleclick"); };
		grid.onCellWrite = function() { self.notifyEvent("onCellWrite"); };
    		
		this.widget = grid;
		return this.widget;
    },

	postCreateWidget: function() {
		this.widget.divExt.tabIndex = -1;
		this.addKeyEventListener(this.widget.divExt);

		/* グリッドの初期ソート状態を設定 */
		if (this.sortable) {
			this.widget.triColonne(this.sortCol, this.boolOrder);
		}

		/* グリッドの初期データ (gridLine/gridCell) を登録 */
		this.setTabData(this.tabData);
	},

	setTabData: function(data) {
		if (data) {
			/*
			 * グリッドがソートされている場合、データ更新後も現在のソート
			 * 状態を保存する
			 */
			var oldColCLic = this.widget.oldColCLic;
			if (oldColCLic) {
				this.widget.fillGrid(data, oldColCLic.ind, oldColCLic.boolOrder);
			} else {
				this.widget.fillGrid(data);
			}

			var index = this.getTabIndex();
			for (var i = 0; i < this.widget.tableauHTML.childNodes.length; i++) {
				if (this.widget.cellActive) {
					var nodes = this.widget.tableauHTML.childNodes[i];
					for (var j = 1; j < nodes.childNodes.length; j += 2) {
						if (this.widget.NbreCol * 2 < j) {
							break;
						}
						if (nodes.childNodes[j].className == "grid_cellData") {
							nodes.childNodes[j].tabIndex = index;
						}
					}
				} else {
					this.widget.tableauHTML.childNodes[i].tabIndex = index;
				}
			}
		}
	},

	getValue: function(key) {
		if (key && this.getters[key]) {
			return this.getters[key].call(this);
		}
		return this.widget.tabData;
	},

	getters: {
		selectedIndex: function() {
			return this.widget.indLineClic;
		},

		selectedIndexes: function() {
			var indexes = []
			var tabData = this.widget.tabData;
			for (var i = 0; i < tabData.length; i++) {
				if (tabData[i].sel) {
					indexes.push(i);
				}
			}
			return indexes;
		}
	},

	setValue: function(value, key) {
		
		if(typeof(value) != "undefined"){
			
			for(i=0;i < value.length;i++){
				var tabdata = value[i];
				
				for(j=0;j < this.widget.tabEntete.length;j++){
					if(typeof(tabdata[j])=="undefined"){
						tabdata[j]="";
					}
				}		
			}
		
		}else{
			this.clear();
		}
	
	
		if (key && this.setters[key]) {
			this.setters[key].call(this, value);
		}
		this.setTabData(value);
	},
	
	setters: {
	},

	clear: function() {
		this.widget.initTab();
	},

	handleKeyEvent: function(event) {
		var element = event.target || event.srcElement;
		var line = parseInt(this.widget.getLineIndex(element), 0);
		/*
		 * セル編集モードで移動させた場合、イベント通知は移動後の
		 * エレメントへ送られる。prenetNodeが"field_global"だと編集中
		 */
		if (line == -1) {
			if (element.parentNode.className == "field_global") {
				if (event.keyCode == 9 || event.keyCode == 13) {
					return false;
				}
			}
			return true;
		}
		if (this.widget.cellActive) {
			var col = parseInt(this.widget.getCellIndex(element), 10);
			if (this.widget.NbreCol - 1 > 0 && col < 0) {
				col = 0;
			}
			switch (event.keyCode) {
			case 13:
			case 32:
				this.widget.afterOnClick(line, col);
				break;
			case 37:
				if (0 < col) {
					this.widget.getHtmlCellFromIndex(line, col - 1).focus();
				}
				break;
			case 38:
				if (0 < line) {
					this.widget.getHtmlCellFromIndex(line - 1, col).focus();
				}
				break;
			case 39:
				if (this.widget.NbreCol - 1 > col) {
					this.widget.getHtmlCellFromIndex(line, col + 1).focus();
				}
				break;
			case 40:
				if (this.widget.NbreLig - 1 > line) {
					this.widget.getHtmlCellFromIndex(line + 1, col).focus();
				}
				break;
			default:
				return true;
			}
		} else {
			switch (event.keyCode) {
			case 13:
			case 32:
				this.widget.afterOnClick(line, col);
				break;
			case 38:
				if (0 < line) {
					this.widget.getHtmlLineFromIndex(line - 1).focus();
				}
				break;
			case 40:
				if (this.widget.NbreLig - 1 > line) {
					this.widget.getHtmlLineFromIndex(line + 1).focus();
				}
				break;
			default:
				return true;
			}
		}
		return false;
	},
	
	setFocus: function() {
		if (this.widget.NbreLig > 0) {
			if (this.widget.cellActive) {
				if (this.widget.NbreCol > 0) {
					this.widget.tableauHTML.childNodes[0].
						childNodes[1].focus();
				}
			} else {
				this.widget.tableauHTML.childNodes[0].focus();
			}
		} else {
			this.widget.tableauHTML.focus();
		}
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.ImageWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
		var orgName = this.name;
    	var image = new rialto.widget.Image(
    		this.imageOut,
    		this.left,
    		this.top,
    		null, /* parent */
    		this.alternateText,
			this.imageOn,
			this);

		var self = this;
		image.onclick = function() { self.notifyEvent("onclick"); };

		this.name = orgName;
		this.widget = image;
		return image;
    },

	setValue: function(value) {
		this.widget.setImages(value);
	},

	clear: function() {
		this.widget.setImages("");
	}
});

/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.LabelWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent) {
		var label = new rialto.widget.Label(
			this.name,
			this.top,
			this.left,
			null, /* parent */
			this.text,
			this.className,
			this);

		this.widget = label;
		return label;
	},

	getValue: function() {
		return this.widget.text;
	},

	setValue: function(value) {
		this.widget.setText(value);
	},

	clear: function() {
		this.widget.setText("");
	}

});
 /*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.PopupWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
		var popup= new rialto.widget.PopUp(
			this.name,
			this.top,
			this.left,
			this.width,
			this.height,
			this.content, /* parent */
			this.title,
			this.suffFond,
			this);
		
		var self = this;
		popup.onClose = function() { self.notifyEvent("onClose"); };
		
		this.widget = popup;
		return popup;
	},

	setParent: function(parent) {
		if (this.parent == parent) {
			return;
		}
		this.parent = parent;
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.RadioGroupWrapper")
	.extend("maskat.layout.Widget", {

	_static: {
		groupMap: {},
		
		getGroup: function(name) {
			return maskat.widget.rialto.RadioGroupWrapper.groupMap[name];
		}
	},
	
   	getWidgetId: function() {
		return this.name;
	},

	postCreateWidget: function() {
		for (var i = 0; i < this.members.length; i++) {
			maskat.widget.rialto.RadioGroupWrapper.groupMap[this.members[i]] = this.name;
		}
	},

	setFocus: function() {
		var widget = this.getSelectWidget();
		if (widget) {
			widget.radio.focus();
		}
	},
	
	getValue: function() {
		var widget = this.getSelectWidget();
		if (widget) {
			return widget.name;
		}
	},

	setValue: function(value) {
		if (this.getLayout().widgets[value]) {
			this.getLayout().widgets[value].widget.setCheck(true);
		}
	},
	
	getSelectWidget: function() {
		var widgets = this.getLayout().widgets;
		for (var i = 0; i < this.members.length; i++) {
			if (widgets[this.members[i]].widget.isCheck()) {
				return widgets[this.members[i]].widget;
			}
		}
		return null;
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.RadioWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent) {
		this.group = maskat.widget.rialto.RadioGroupWrapper.getGroup(this.name);
		var radio = new rialto.widget.Radio(
			this.name,
			this.top,
			this.left,
			null, /* parent */
			this.group,
			this.text,
			this.checked,
			this.className);
		radio.setEnable(this.enable);

		var self = this;
		radio.onclick = function() { self.notifyEvent("onclick"); };
		this.widget = radio;
		
		this.tabIndexGroupName = this.group;
		this.widget.radio.tabIndex = 0;
		rialto.widget.Form.prototype.tabIndex = 0;
		return radio;
    },

    postCreateWidget: function() {
		this.addKeyEventListener(this.widget.radio);
    },

	handleKeyEvent: function(event) {
		if (event.keyCode == 13 || event.keyCode == 32) {
			this.widget.setCheck(true);
			
		} else if (event.keyCode == 37 || event.keyCode == 38) {
			var members = this.getLayout().widgets[this.group].members;
			var radio = this.getLayout().widgets;
			for (var i = 1; i < members.length; i++) {
				if (radio[members[i]].widget.name == this.name) {
					this.getLayout().widgets[members[i]].widget.radio.focus();
					break;
				}
			}
		} else if (event.keyCode == 39 || event.keyCode == 40) {
			var members = this.getLayout().widgets[this.group].members;
			var radio = this.getLayout().widgets;
			for (var i = 0; i < members.length - 1; i++) {
				if (radio[members[i]].widget.name == this.name) {
					this.getLayout().widgets[members[i]].widget.radio.focus();
					break;
				}
			}
		}
		return true;
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.RialtoPlugin")
	.extend("maskat.core.Plugin", {

	_static: {
		initialize: function() {
			maskat.core.Plugin.register(this);
		}
	},

	initialize: function() {
		this.properties = new maskat.util.Properties({
			version: { type: "number", defaultValue: 0.9 },
			isDebug: { type: "boolean", defaultValue: false },
			traceLevel: { type: "number", defaultValue: 0 },	
			isTestVersion: { type: "boolean", defaultValue: false },
			language: { type: "string", defaultValue: "en" }
		});
	},

	getPluginId: function() {
		return "rialto";
	},

	getVersion: function() {
		return "1.0.0.v20080521";
	},

	isLoaded: function() {
		return (typeof(rialto) != "undefined") &&
			(typeof(rialtoConfig) != "undefined");
	},

	load: function(app) {
		var pathRialtoE = maskat.location + "rialto/rialtoEngine/";
		maskat.app.loadStyleSheet(pathRialtoE + "style/rialto.css");
		maskat.app.loadStyleSheet(pathRialtoE + "style/behavior.css");
		maskat.app.loadStyleSheet(pathRialtoE + "style/defaultSkin.css");

		maskat.app.loadJavaScript(pathRialtoE + "config.js", false);
		maskat.app.loadJavaScript(pathRialtoE + "javascript/rialto.js", true);
	},
	
	start: function() {
		maskat.lang.Object.populate(rialtoConfig, this.properties.getProperties());
		rialtoConfig.pathRialtoE = maskat.location + "rialto/rialtoEngine/";
		rialto.onLoad();
		
		var reader = maskat.layout.LayoutXMLReader.getInstance();
		var library = new maskat.widget.rialto.RialtoWidgetLibrary();
		reader.addWidgetLibrary(library);
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.RialtoWidgetLibrary")
	.extend("maskat.layout.WidgetLibrary", {

	getBindingConfiguration: function() {
		var config = {
			window: {
				type: maskat.widget.rialto.SimpleWindowWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number" },
					left: { type: "number" },
					width: { type: "number" },
					height: { type: "number" },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					title: { type: "string" },
					icone: { type: "string" },
					style: {
						type: "enum",
						values: [ "default", "fenetre" ],
						defaultValue: "fenetre"
					}
				},
				children: {
					"*": { property: "children", repeat: true }
				}
			},

			image: {
				type: maskat.widget.rialto.ImageWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number" },
					height: { type: "number" },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					imageOut: { type: "string", required: true },
					alternateText: { type: "string" },
					imageOn: { type: "string" },
					imageDisabled: { type: "string" },
					boolFloatRight: { type: "boolean" },
					boolFloatLeft: { type: "boolean" }
				}
			},

			tabFolder: {
				type: maskat.widget.rialto.TabFolderWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number", required: true },
					height: { type: "number", required: true },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					autoResizeContenu: { type: "boolean" },
					autoResizeParent: { type: "boolean" },
					autoRedimTab: { type: "boolean" },
					isClosable: { type: "boolean" },
					draggableItem: { type: "boolean", defaultValue: false },
					orientation: {
						type: "enum",
						values: ["t", "b", "l", "r"]
					},
					widthTabName: { type: "number" },
					noActiveTab: { type: "number", defaultValue: 1 },  
					tabIndex: { type: "number", defaultValue: -1 }
				},
				children: {
					tabItem: { property: "children", repeat: true }
				}
			},

			tabItem: {
				type: maskat.widget.rialto.TabItemWrapper,
				attributes: {
					name: { type: "string", required: true },
					title: { type: "string" }
				},
				children: {
					"*": { property: "children", repeat: true }
				}
			},

			splitter: {
				type: maskat.widget.rialto.SplitterWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number" },
					height: { type: "number" },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					prop: { type: "number" },
					orientation: { type: "enum", values: ["v", "h"] },
					autoResizeContenu: { type: "boolean" },
					autoResizeParent: { type: "boolean" , defaultValue: false},
					autoResizableH: { type: "boolean" , defaultValue: false},
					autoResizableW: { type: "boolean" , defaultValue: false},
					style: { type: "enum", values: ["normal", "3D"] },
					overflow: { type: "enum", values: ["auto", "hidden"] },
					modeLim: { type: "enum", values: ["%", "abs"] },
					limInf: { type: "number" },
					limSup: { type: "number" },
					reverseClose: { type: "boolean" },
					withImg: { type: "boolean" },
					tailleCurs: { type: "number" }
				},
				children: {
					divSplit: { property: "children", repeat: true }
				}
			},

			divSplit: {
				type: maskat.widget.rialto.DivSplitWrapper,
				attributes: {
					name: { type: "string", required: true },
					backgroundColor: { type: "string", defaultValue: "white" }
				},
				children: {
					"*": { property: "children", repeat: true }
				}
			},
				
			label: {
				type: maskat.widget.rialto.LabelWrapper,
			    attributes: {
					name: { type: "string", required: true },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					text: { type: "string" },
					className: { type: "string" },
					position: {
						type: "enum",
						values: [ "static", "relative", "absolute" ]
					}
				}
			},
	
			text: {
				type: maskat.widget.rialto.TextWrapper,
				attributes: {
					name: { type: "string", required: true },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number" },
					datatype: {
						type: "enum",
						values: ["T", "P", "A", "N", "I", "D", "H", "Hi"]
					},
					position: {
						type: "enum",
						values: ["static", "relative", "absolute"]
					},
					nbchar: { type: "number" },
					autoUp: { type: "boolean" },
					disable: { type: "boolean" },
					isRequired: { type: "boolean" },
					rows: { type: "number" },
					initValue: { type: "string" },
					accessKey: { type: "string" },
					tabIndex: { type: "number" }
				}
			},
			
			button: {
				type: maskat.widget.rialto.ButtonWrapper,
				attributes: {
			        name: { type: "string", required: true },
			        top: { type: "number", required: true },
			        left: { type: "number", required: true },
			        title: { type: "string" },
			        alt: { type: "string" },
			        width: { type: "number" },
			        widthMin:{ type: "number" },
			        adaptToText: { type: "boolean" },
			        enable: { type: "boolean", defaultValue: true },
			        tabIndex: { type: "number", defaultValue: -1 }
			    }
			},
	
			combo: {
				type: maskat.widget.rialto.ComboWrapper,
				attributes: {
					name: { type: "string", required: true },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number" },
					parent: { type: "object" },
					position: {
						type: "enum",
						values: ["static", "relative", "absolute"]
					},
					suggest: { type: "boolean" },
			        enable: { type: "boolean", defaultValue: true },
					heightItem: { type: "number" },
					tabIndex: { type: "number", defaultValue: -1 }
				},
				children: {
					comboItem: { property: "tabData", repeat: true }
				}
			},
			
			comboItem: {
				type: Array,
				attributes: {
					text: { type: "string", property: "1" ,required: true },
					value: { type: "string", property: "0" ,required: true }
				}
			},
	
			radioGroup: {
				type: maskat.widget.rialto.RadioGroupWrapper,
				attributes: {
					name: {type: "string", required: true},
					tabIndex: {type: "number", defaultValue: -1}
				},
				children: {
					radioMember: { property: "members", repeat: true, value: "ref" }
				}
			},

			radioMember: {
				attributes: {
					ref: { type: "string" ,required: true }
				}
			},

			radio: {
				type: maskat.widget.rialto.RadioWrapper,
				attributes: {
					name: { type: "string", required: true },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					parent: { type: "object" },
					// レイアウト定義XML的には使用してはいけないが、処理上許可
					group: { type: "string" }, 
					text: { type: "string" },
					checked: { type: "boolean" },
					className: { type: "string" },
			        enable: { type: "boolean", defaultValue: true }
				}
			},
			
			checkbox: {
				type: maskat.widget.rialto.CheckboxWrapper,
				attributes: {
					name: { type: "string", required: true },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					parent: { type: "object" },
					text: { type: "string" },
					checked: { type: "boolean" },
					className: { type: "string" },
					tabIndex: { type: "number", defaultValue: -1},
			        enable: { type: "boolean", defaultValue: true }
				}
			},
	
			alert: {
				type: maskat.widget.rialto.AlertWrapper,
				attributes: {
					name: { type: "string" , required: true},
					mess: { type: "string" , required: true}
				}
			},

			frame: {
				type: maskat.widget.rialto.FrameWrapper,
			    attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number", required: true },
					height: { type: "number", required: true },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
				    dynamic: { type: "boolean" },
				    open: { type: "boolean" },
				    title: { type: "string" },
				    printTitle: { type: "string" },
				    autoResizeContenu: { type: "boolean" },
				    autoResizeParent: { type: "boolean" },
				    boolPrint: { type: "boolean" },
				    boolMaxi: { type: "boolean" }
				},
				children: {
					"*": { property: "children", repeat: true }
				}
			},
	
			popup: {
				type: maskat.widget.rialto.PopupWrapper,
				attributes: {
					name: { type: "string", required: true },  
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number",required: true },
					height: { type: "number",required: true },
					content: { type: "string" },
					visible: { type: "boolean", defaultValue: false },
					withCloseButon: { type: "boolean", defaultValue: true },
					position: {
						type: "enum",
						values: ["static", "absolute", "relative"],
						defaultValue:  "absolute"
					},
					draggable: { type: "boolean", defaultValue: false},
					title: { type: "string" ,defaultValue: "Window"},  
					suffFond: {
						type: "enum",
						values: [ "Gris", "Transparent" ],
						defaultValue: "Transparent"
					},
					modeContainer: {
						type: "enum",
						values: ["AutoScroll", "nonFen"]
					},  
					bSansBtonClose: { type: "boolean" }
				},  
				children: {
					"*": { property: "children", repeat: true }
				}
			},
		  
			treeview: {
				type: maskat.widget.rialto.TreeWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string",required: true },
					type: { type: "string" },
					top: { type: "number" ,required: true},
					left: { type: "number" , required: true},
					width: { type: "number" },
					height: { type: "number" },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
				    boolSelActive: { type: "boolean" },
				    withT: { type: "boolean" },
				    rootOpen: { type: "boolean" },
				    withRoot: { type: "boolean" },
				    autoResizableH: { type: "boolean" },
				    autoResizableW: { type: "boolean" },
				    draggableNode: { type: "boolean" },
				    rootNode: { type: "object" },
				    parent: { type: "object" },
				    tabIndex: { type: "number" }
				},
				children: {
					treeNode: { property: "rootNodeData" }
				}
			},
	
			treeNode: {
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string" ,required: true},
					type: { type: "string" },
					top: { type: "number" },
					left: { type: "number" },
					width: { type: "number" },
					height: { type: "number" },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					typeInfo: { type: "string" },
					sText: { type: "string", property: "text" },
					sIcon: { type: "string", property: "icon" },
					sIcon2: { type: "string", property: "icon2" },
					onclick: { type: "string" },
					open: { type: "boolean" },
					reload: { type: "boolean" },
					url: { type: "string" }
				},
				children: {
					treeNode: { repeat: true, property: "children" }
				}
			},
	
			grid: {
				type: maskat.widget.rialto.GridWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number" },
					height: { type: "number" , defaultValue: 100 },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					TabEntete: { type: "object" },
					bNavig: { type: "boolean" },
					cellActive: { type: "boolean" },
					multiSelect: { type: "boolean" },
					lineHeight: { type: "number" },
					rang: { type: "number" },
					widthLastCell: { type: "number" },
					actifClic: { type: "boolean" },
					boolPrint: { type: "boolean" },
					switchable: { type: "boolean" },
					printTitle: { type: "string" },
					autoResizeContenu: { type: "boolean" },
					autoResizeParent: { type: "boolean" },
					writable: { type: "boolean" },
					tabTypeCol: { type: "object" },
					parent: { type: "object" },
					tabIndex: { type: "number", defaultValue: -1 },
					sortable: { type: "boolean", defaultValue: true },
					sortCol: { type: "number", defaultValue: 0 },
					sortType: {
						type: "enum",
						values: { "asc": false, "desc": true },
						defaultValue: "asc",
						property: "boolOrder"
					},
					onCellEdit:{ type: "function"}
				},
				children: {
					gridHeader: { property: "headers", repeat: true },
					gridLine: { property: "tabData", repeat: true, value: "cells" }
				}
			},
			
			gridHeader: {
				attributes: {
					width: { type: "number", property: "width", required: true },
					title: { type: "string", property: "title" },
					type: {
						type: "enum",
						values: [ "number", "date", "string" ],
						property: "type",
						required: true
					}
				},
				children: {
					gridCellCombo: { property: "tabData", repeat: true }
				}
			},
			
			gridCellCombo: {
				type: Array,
				attributes: {
					value: { property: "0", type: "string"},
					text: { property: "1", type: "string" }
				}
			},
	
			gridLine: {
				children: {
					gridCell: { property: "cells", repeat: true }
				}
			},
	
			gridCell: {
				attributes: {
					value: { type: "string" }
				}
			},
			
			codeLib: {
				type: maskat.widget.rialto.CodeLibWrapper,
				attributes: {
					bWithoutPlaceIn: { type: "boolean" },
					name: { type: "string", required: true },
					type: { type: "string" },
					top: { type: "number", required: true },
					left: { type: "number", required: true },
					width: { type: "number", required: true },
					position: {
						type: "enum",
						values: [ "static", "absolute", "relative" ],
						defaultValue: "absolute"
					},
					enable: { type: "boolean", defaultValue: true },
					draggable: { type: "boolean" },
					resizable: { type: "boolean" },
					autoResizableH: { type: "boolean" },
					autoResizableW: { type: "boolean" },
					parent: { type: "object"  },
					arrValue: { type: "object"  },
					boolWithLabel: { type: "boolean", property: "withLabel" },
					url: { type: "string", required: true },
					submitOnload: { type: "boolean" },
					alwaysRefresh: { type: "boolean" },
					codeWidth: { type: "number" },
					tabIndex: { type: "number", defaultValue: -1 }
				}
			}
		};

		return config;
	}


});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.RialtoWidgetWrapper")
	.extend("maskat.layout.Widget", {
	
	createWidget: function(parent) {
		return null;
	},

	dispose: function() {
		/* Rialto コンポーネントの破棄 */
		if (this.widget) {
			this.widget.remove();
		}
		
		/* ラッパーオブジェクトの属性を破棄 */
		maskat.lang.Object.dispose(this);
	},

	getWidget: function() {
		return this.widget;
	},

	getWidgetId: function() {
		return this.name;
	},

	getElement: function() {
		return this.widget.getHtmlExt();
	},

	setParent: function(parent){
		if (this.parent == parent) {
			return;
		}
		this.parent = parent;

		var parentWidget = parent.getWidget();
		if (parentWidget instanceof rialto.widget.AbstractContainer) {
			parentWidget.add(this.widget);
		} else {
			this.widget.placeIn(parent.getElement());
		}
	},
	
    isEnabled: function() {
		return this.enable != false;
    }
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.SimpleWindowWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent) {
		var simpleWindow = new rialto.widget.SimpleWindow(this);

		var self = this;
		simpleWindow.onSetTitle = function() { self.notifyEvent("onSetTitle"); };
		simpleWindow.onClose = function() { self.notifyEvent("onclose"); };

		this.widget = simpleWindow;
		return simpleWindow;
    }

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.SplitterWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
		var splitter = new rialto.widget.Splitter(this);

    	this.widget = splitter;
    	return splitter;
    }

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.TabFolderWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
  		var folder = new rialto.widget.TabFolder(this);

    	this.widget = folder;
    	return folder;
    },

    postCreateWidget: function() {
    	var folder = this.getWidget();
   		folder.activeTab(this.noActiveTab - 1);
		this.widget.divExt.tabIndex = this.getTabIndex();
		this.addKeyEventListener(this.widget.divExt);
    },

	handleKeyEvent: function(event) {
		if (event.keyCode == 39) {
			var pos = this.widget.indActiveTab + 1;
			if (pos < this.widget.arrTabItem.length) {
				this.widget.activeTab(pos);
			}
			return false;
			
		} else if (event.keyCode == 37) {
			var pos = this.widget.indActiveTab - 1;
			if (pos >= 0) {
				this.widget.activeTab(pos);
			}
			return false;
		}
		return true;
	},

	setFocus: function() {
		this.widget.divExt.focus();
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.TabItemWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
    	var folder = parent.getWidget();
		var item = folder.addTabItem(this.title);

		if (folder.orientation == "t" || folder.orientation == "b") {
			item.contenuOnglet.style.width = folder.width - 2;
			item.contenuOnglet.style.height = folder.height - folder.$tabSize - 2;
		} else if (folder.orientation == "l" || folder.orientation == "r") {
			item.contenuOnglet.style.width = folder.width - folder.$tabSize - 2;
			item.contenuOnglet.style.height = folder.height - 2;
		}
		
		var self = this;
		item.onEnableTab = function() { self.notifyEvent("onEnableTab"); };

		this.widget = item;
		return item;
    },

	setParent: function(parent){
		this.parent = parent;
	}

});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.TextWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

    createWidget: function(parent) {
    	var text = new rialto.widget.Text(
    		this.name,
    		this.top,
    		this.left,
    		this.width,
    		this.datatype,
    		null, /* parent */
    		this);

		var self = this;
		text.onblur = function() { self.notifyEvent("onblur"); };
		text.onfocus = function() { self.notifyEvent("onfocus"); };

		this.widget = text;
		return text;
    },

	getControlElement: function(){
		return this.widget.getHtmlExt().firstChild;
	},

	getValue: function() {
		switch (this.widget.datatype) {
		case "DT":
		case "DA":
		case "D":
			return this.getDateTimeValue();
		default:
			return this.widget.getValue();
		}
	},
	
	getDateTimeValue: function(){
		var value = this.widget.getValue();
		  if (value.length!=11){
		     return "";
		  }
		  if (this.datatype=="D"){
		     return this.month() + "/" + this.day() + "/" + this.year();
		  }
		  if (this.datatype=="DA"){
		     return this.year() + "-" + this.month() + "-" + this.day();
		  }
		  if (this.datatype=="DT"){
		     var datePart =  this.year() + "-" + this.month() + "-" + this.day();
		     var now = new Date();
		     var hours = now.getHours();
		     var strHours = hours<10? "0"+hours:""+hours;
		     var minutes = now.getMinutes();
		     var strMinutes = minutes<10? "0"+minutes:""+minutes;
		     var seconds = now.getSeconds();
		     var strSeconds = seconds<10? "0"+seconds:""+seconds;
		     return datePart + "T" + strHours + ":" + strMinutes + ":" + strSeconds;
		  }
		  return "";
	},
	
    year :function(){
	  var value = this.widget.getValue();
	  if (value.length != 11){
	    return "";
	  }
	  return value.substring(0,4);
	},
	
	month : function(){
	  var value = this.widget.getValue();
	  if (value.length != 11){
	    return "";
	  }
	  return value.substring(5,7);
	},
	
	day : function(){
	  var value = this.widget.getValue();
	  if (value.length != 11){
	    return "";
	  }
	  return value.substring(8,10);
	},

	setValue: function(value) {
		this.widget.setValue(value);
	},

	clear: function() {
		this.widget.setValue("");
	},

    postCreateWidget: function() {
		this.addKeyEventListener(this.widget.divExt);
    },

	setFocus: function() {
		this.widget.divExt.childNodes[0].focus();
	}
});
/*
 * Copyright (c)  2006-2007 Maskat Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
maskat.lang.Class.declare("maskat.widget.rialto.TreeWrapper")
	.extend("maskat.widget.rialto.RialtoWidgetWrapper", {

	createWidget: function(parent) {
		var tree = new rialto.widget.Tree(this);
		if (this.rootNodeData) {
			this.createTreeNode(this.rootNodeData, tree);
			delete this.rootNodeData
		}
		var self = this;
		tree.onclick = function() { self.notifyEvent("onclick"); };
		
		this.widget = tree;
		return tree;
    },

	createTreeNode: function(nodeData, root) {
		var node = new rialto.widget.TreeNode(nodeData);
		node.DIVENTETE.tabIndex = this.getTabIndex();
		
		if (nodeData._hiddenFields) {
			node._hiddenFields = nodeData._hiddenFields;
		}
		if (root) {
			root.add(node);
		}
    	var children = nodeData.children;
    	if (children) {
	    	for (var i = 0; i < children.length; i++) {
	    		node.addNode(this.createTreeNode(children[i]));
	    	}
	    }
		return node;
    },
	
    postCreateWidget: function() {
		this.widget.divExt.tabIndex = this.getTabIndex();
		this.addKeyEventListener(this.widget.divExt);
    },

	getValue: function(key) {
		if (key && this.getters[key]) {
			return this.getters[key].call(this);
		}
		var data = [];
		this.getNodeData(this.widget.rootNode, data);
		return data;
	},

	getters: {
		selectedIndex: function() {
			if (!this.widget.currentSelNode || !this.widget.rootNode) {
				return -1;
			}
			return this.widget.getCount(this.widget.currentSelNode).nbC;
		},

		selectedIndexes: function() {
			return [this.getters.selectedIndex.call(this)];
		}
	},

	getNodeData: function(node, nodes) {
		var param = {
			"NAME": node.name,
			"TEXT": node.text,
			"PARENT": node.fatherNode ? node.fatherNode.name : null
		};
		if (node._hiddenFields) {
			for (var v in node._hiddenFields) {
				param[v] = node._hiddenFields[v];
			}
		}
		nodes.push(param);
		
		if (node.hasChild()) {
			for (var i = 0; i < node.arrChildNode.length; i++) {
				this.getNodeData(node.arrChildNode[i], nodes);
			}
		}
	},

	setValue: function(value, key) {
		if (key && this.setters[key]) {
			this.setters[key].call(this, value);
		}
		this.setNodeData(value);
	},
	
	setters: {
	},
	
	setNodeData: function(value) {
		this.rootNodeData = this.createNodeParams(value);
		if (this.rootNodeData) {
			this.createTreeNode(this.rootNodeData, this.widget);
			delete this.rootNodeData
		}
	},
	
	findParentNode: function(node, name) {
		if (node.name == name) {
			return node;
		}
		if (node.children) {
			for (var i = 0; i < node.children.length; i++) {
				var result = this.findParentNode(node.children[i], name);
				if (result) {
					return result;
				}
			}
		}
		return null;
	},
	
	createNodeParams: function(value) {
		var nodeData = {};

		if (this.widget.rootNode) {
			this.widget.rootNode.remove();
			delete this.widget.rootNode;
		}
		for (var i = 0; i < value.length; i++) {
			if (!value[i].PARENT) {
				nodeData = this.getNodeParam(value[i]);
			} else {
				var node = this.findParentNode(nodeData, value[i].PARENT);
				if (node) {
					if (!node.children) {
						node.children = [];
					}
					node.children.push(this.getNodeParam(value[i]));
				}
			}
		}
		return nodeData;
	},
	
	getNodeParam: function(value) {
		var param = {
			name: value.NAME,
			text: value.TEXT,
			icon: value.icon || "images/imTreeview/pict_synthetik_off.gif",
			icon2: value.icon2 || "",
			open: value.open || true,
			reload: value.reload || false,
			onclick: value.onclick || ""
		};
		for (var v in value) {
			if (v.charAt(0) == "_") {
				if (!param._hiddenFields) {
					param._hiddenFields = {};
				}
				param._hiddenFields[v] = value[v];
			}
		}
		return param;
	},
	
	findNodeWidget: function(node, div) {
		if (node.DIVENTETE == div) {
			return node;
		}
		for (var i = 0; i < node.arrChildNode.length; i++) {
			var result = this.findNodeWidget(node.arrChildNode[i], div);
			if (result) {
				return result;
			}
		}
		return null;
	},
   
	handleKeyEvent: function(event) {
		var manager = maskat.key.KeyEventManager.getInstance();
		var element = event.target || event.srcElement;
		var widget = this.findNodeWidget(this.widget.rootNode, element);
		if (!widget) {
			return true;
		}
		switch (event.keyCode) {
		case 13:
		case 32:
			if (widget.hasChild()) {
				widget.toggle();
			} else {
				widget.click();
			}
			break;
		case 37:
			if (widget.hasChild() && widget.open) {
				widget.toggle();
			}
			break;
		case 38:
			if (widget.isFirst()) {
				if (widget.fatherNode) {
					widget.fatherNode.DIVENTETE.focus();
				}
			} else {
				var node = widget.previous();
				while (node.hasChild() && node.open){
					node = node.last();
				}
				node.DIVENTETE.focus();
			}
			break;
		case 39:
			if (widget.hasChild() && !widget.open) {
				widget.toggle();
			}
			break;
		case 40:
			if (widget.hasChild() && widget.open) {
				widget.first().DIVENTETE.focus();
			} else {
				if (!widget.isLast()) {
					widget.next().DIVENTETE.focus();
				} else {
					var node = widget.fatherNode;
					while (node) {
						if (!node.isLast()) {
							node.next().DIVENTETE.focus();
							break;
						}
						node = node.fatherNode;
					}
				}
			}
			break;
		default:
			return true;
		}
		return false;
	},

	setFocus: function() {
		this.widget.rootNode.DIVENTETE.focus();
	}
});
