/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 * 
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 * 
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 * 
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;

/**
 * 区切り文字形式データの項目名を基準として Java プログラム要素のリストと区切り文字形式データアクセスを行うハンドラを提供します。
 * 
 * @author 杉澤 浩二
 */
public class ColumnNameMappingBeanListHandler<T> extends BeanListHandlerSupport<T> {

	/**
	 * 項目名と Java プログラム要素のフィールド名のマップを保持します。
	 */
	private LinkedHashMap<String, String> nameMap = new LinkedHashMap<String, String>();

	// ------------------------------------------------------------------------

	/**
	 * コンストラクタです。
	 * 
	 * @param type Java プログラム要素の型
	 * @throws IllegalArgumentException <code>type</code> が <code>null</code> の場合
	 */
	public ColumnNameMappingBeanListHandler(final Class<T> type) {
		super(type);
	}

	// ------------------------------------------------------------------------

	/**
	 * 指定された項目名と Java プログラム要素のフィールド名をマップへ追加します。
	 * 
	 * @param column 項目名
	 * @param field Java プログラム要素のフィールド名
	 * @return このオブジェクトへの参照
	 */
	public ColumnNameMappingBeanListHandler<T> addColumn(final String column, final String field) {
		this.nameMap.put(column, field);
		return this;
	}

	/**
	 * 項目名と Java プログラム要素のフィールド名のマップを設定します。
	 * 
	 * @param columnMapping 項目名と Java プログラム要素のフィールド名のマップ
	 * @return このオブジェクトへの参照
	 * @throws IllegalArgumentException <code>columnMapping</code> が <code>null</code> の場合
	 */
	public ColumnNameMappingBeanListHandler<T> columnMapping(final Map<String, String> columnMapping) {
		if (columnMapping == null) {
			throw new IllegalArgumentException("Column mapping must not be null");
		}
		this.nameMap = new LinkedHashMap<String, String>(columnMapping);
		return this;
	}

	// ------------------------------------------------------------------------

	private String getFieldName(final String column) {
		return nameMap.get(column);
	}

	// ------------------------------------------------------------------------

	@Override
	public List<T> load(final CsvReader reader) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		prepare();

		// ヘッダ部を処理します。
		final List<String> columnNames = reader.readValues();
		if (columnNames == null) {
			// ヘッダがない場合は例外をスローします。
			throw new IOException("No header is available");
		}

		// データ部を処理します。
		final List<T> results = new ArrayList<T>();
		final int columnCount = columnNames.size();
		List<String> values;
		while ((values = reader.readValues()) != null) {
			final T obj = createBean();
			final int len = Math.min(columnCount, values.size());
			for (int pos = 0; pos < len; pos++) {
				final String fieldName = getFieldName(columnNames, pos);
				if (fieldName == null) {
					continue;
				}
				final Field f = getField(getType(), fieldName);
				final Object o = getConverter().convert(values.get(pos), f.getType());
				if (o != null) {
					setFieldValue(obj, f, o);
				}
			}
			results.add(obj);
		}
		return results;
	}

	@Override
	public void save(final List<T> list, final CsvWriter writer) throws IOException {
		// TODO: 必要があればここに入力パラメータを検証するコードを記述します。

		prepare();

		// ヘッダ部を処理します。
		final List<String> columnNames = new ArrayList<String>();
		for (final Map.Entry<String, String> entry : nameMap.entrySet()) {
			columnNames.add(entry.getKey());
		}
		writer.writeValues(columnNames);

		// データ部を処理します。
		final int columnCount = columnNames.size();
		for (final T obj : list) {
			// 要素が null の場合は null 出力します。
			if (obj == null) {
				writer.writeValues(null);
				continue;
			}

			final String[] values = new String[columnCount];
			for (int i = 0; i < columnCount; i++) {
				final String fieldName = getFieldName(columnNames, i);
				if (fieldName == null) {
					continue;
				}
				final Field f = getField(obj.getClass(), fieldName);
				values[i] = getConverter().convert(getFieldValue(obj, f));
			}
			writer.writeValues(Arrays.asList(values));
		}
	}

	/**
	 * 項目名とフィールド名のマップが指定されていない場合、フィールド名からマップを作成して準備します。
	 */
	private void prepare() {
		if (nameMap.size() == 0) {
			for (final Field f : getType().getDeclaredFields()) {
				final String name = f.getName();
				addColumn(name, name);
			}
		}
	}

	private String getFieldName(final List<String> columnNames, final int position) {
		final String columnName = columnNames.get(position);
		if (columnName != null) {
			return getFieldName(columnName);
		}
		return null;
	}

}
