/*
 * Copyright (c) 2009 OrangeSignal.com All rights reserved.
 *
 * これは Apache ライセンス Version 2.0 (以下、このライセンスと記述) に
 * 従っています。このライセンスに準拠する場合以外、このファイルを使用
 * してはなりません。このライセンスのコピーは以下から入手できます。
 *
 * http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * 適用可能な法律がある、あるいは文書によって明記されている場合を除き、
 * このライセンスの下で配布されているソフトウェアは、明示的であるか暗黙の
 * うちであるかを問わず、「保証やあらゆる種類の条件を含んでおらず」、
 * 「あるがまま」の状態で提供されるものとします。
 * このライセンスが適用される特定の許諾と制限については、このライセンス
 * を参照してください。
 */

package jp.sf.orangesignal.csv.handlers;

import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import jp.sf.orangesignal.csv.CsvConfig;
import jp.sf.orangesignal.csv.CsvReader;
import jp.sf.orangesignal.csv.CsvWriter;

import org.junit.Before;
import org.junit.Test;

/**
 * {@link ColumnNameMapListHandler} クラスの単体テストです。
 *
 * @author 杉澤 浩二
 */
public class ColumnNameMapListHandlerTest {

	private CsvConfig cfg;

	@Before
	public void setUp() throws Exception {
		cfg = new CsvConfig(',');
		cfg.setNullString("NULL");
		cfg.setIgnoreTrailingWhitespaces(true);
		cfg.setIgnoreLeadingWhitespaces(true);
		cfg.setIgnoreEmptyLines(true);
	}

	@Test
	public void testLoad1() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("symbol,name,price,volume\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final List<Map<String, String>> list = new ColumnNameMapListHandler().load(reader);
			assertThat(list.size(), is(2));
			final Map<String, String> m1 = list.get(0);
			assertThat(m1.get("symbol"), is("AAAA"));
			assertThat(m1.get("name"), is("aaa"));
			assertThat(m1.get("price"), is("10000"));
			assertThat(m1.get("volume"), is("10"));
			final Map<String, String> m2 = list.get(1);
			assertThat(m2.get("symbol"), is("BBBB"));
			assertThat(m2.get("name"), is("bbb"));
			assertTrue(m2.containsKey("price"));
			assertNull(m2.get("price"));
			assertThat(m2.get("volume"), is("0"));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testLoadOffsetLimit() throws IOException {
		final CsvReader reader = new CsvReader(new StringReader("symbol,name,price,volume\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final List<Map<String, String>> list = new ColumnNameMapListHandler().offset(1).limit(1).load(reader);

			assertThat(list.size(), is(1));
			final Map<String, String> m2 = list.get(0);
			assertThat(m2.get("symbol"), is("BBBB"));
			assertThat(m2.get("name"), is("bbb"));
			assertTrue(m2.containsKey("price"));
			assertNull(m2.get("price"));
			assertThat(m2.get("volume"), is("0"));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testLoad2() throws IOException {
		cfg.setSkipLines(1);
		final CsvReader reader = new CsvReader(new StringReader("symbol,name,price,volume\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0"), cfg);
		try {
			final List<Map<String, String>> list = new ColumnNameMapListHandler()
				.addColumn("symbol")
				.addColumn("name")
				.addColumn("price")
				.addColumn("volume")
				.load(reader);

			assertThat(list.size(), is(2));
			final Map<String, String> m1 = list.get(0);
			assertThat(m1.get("symbol"), is("AAAA"));
			assertThat(m1.get("name"), is("aaa"));
			assertThat(m1.get("price"), is("10000"));
			assertThat(m1.get("volume"), is("10"));
			final Map<String, String> m2 = list.get(1);
			assertThat(m2.get("symbol"), is("BBBB"));
			assertThat(m2.get("name"), is("bbb"));
			assertTrue(m2.containsKey("price"));
			assertNull(m2.get("price"));
			assertThat(m2.get("volume"), is("0"));
		} finally {
			reader.close();
		}
	}

	@Test
	public void testSave1() throws IOException {
		final List<Map<String, String>> list = new ArrayList<Map<String, String>>(3);
		final Map<String, String> m1 = new LinkedHashMap<String, String>(4);
		m1.put("symbol", "AAAA");
		m1.put("name", "aaa");
		m1.put("price", "10000");
		m1.put("volume", "10");
		list.add(m1);
		final Map<String, String> m2 = new LinkedHashMap<String, String>(4);
		m2.put("symbol", "BBBB");
		m2.put("name", "bbb");
		m2.put("price", null);
		m2.put("volume", "0");
		list.add(m2);

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			new ColumnNameMapListHandler().save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("symbol,name,price,volume\r\nAAAA,aaa,10000,10\r\nBBBB,bbb,NULL,0\r\n"));
	}

	@Test
	public void testSave2() throws IOException {
		final List<Map<String, String>> list = new ArrayList<Map<String, String>>(3);
		final Map<String, String> m1 = new LinkedHashMap<String, String>(4);
		m1.put("symbol", "AAAA");
		m1.put("name", "aaa");
//		m1.put("price", "10000");
		m1.put("volume", "10");
		list.add(m1);
		final Map<String, String> m2 = new LinkedHashMap<String, String>(4);
		m2.put("symbol", "BBBB");
		m2.put("name", "bbb");
//		m2.put("price", null);
		m2.put("volume", "0");
		list.add(m2);

		final StringWriter sw = new StringWriter();
		final CsvWriter writer = new CsvWriter(sw, cfg);
		try {
			new ColumnNameMapListHandler()
				.addColumn("symbol")
				.addColumn("name")
				.addColumn("price")
				.addColumn("volume")
				.save(list, writer);
		} finally {
			writer.close();
		}
		assertThat(sw.getBuffer().toString(), is("symbol,name,price,volume\r\nAAAA,aaa,NULL,10\r\nBBBB,bbb,NULL,0\r\n"));
	}

}
