/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.pal.webparts;

import java.io.IOException;
import java.util.ResourceBundle;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.GenericPortlet;
import javax.portlet.PortletConfig;
import javax.portlet.PortletContext;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletPreferences;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * This portlet displays Web Parts, like Google Gadgets.
 * 
 * @author <a href="mailto:shinsuke@yahoo.co.jp">Shinsuke Sugaya</a>
 */
public class WebPartsPortlet extends GenericPortlet {
    /**
     * Logger for this class
     */
    private static final Log log = LogFactory.getLog(WebPartsPortlet.class);

    public static final String TITLE = "jp.sf.pal.webparts.Title";

    public static final String CODE = "jp.sf.pal.webparts.Code";

    public static final String URL = "jp.sf.pal.webparts.Url";

    public static final String VIEW_PAGE = "jp.sf.pal.webparts.ViewPage";

    public static final String EDIT_PAGE = "jp.sf.pal.webparts.EditPage";

    public static final String MESSAGE = "jp.sf.pal.webparts.Message";

    public static final String DEFAULT_VIEW_VALUE = "/WEB-INF/jsp/view.jsp";

    public static final String DEFAULT_EDIT_VALUE = "/WEB-INF/jsp/edit.jsp";

    protected String defaultTitle = "";

    protected String defaultCode = "";

    protected String defaultUrl = "";

    protected String viewPage = "";

    protected String editPage = "";

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.Portlet#init(javax.portlet.PortletConfig)
     */
    public void init(PortletConfig config) throws PortletException {
        super.init(config);

        defaultTitle = config.getInitParameter(TITLE);
        defaultCode = config.getInitParameter(CODE);
        defaultUrl = config.getInitParameter(URL);
        viewPage = config.getInitParameter(VIEW_PAGE);
        if (viewPage == null) {
            viewPage = getDefaultViewPage();
        }
        editPage = config.getInitParameter(EDIT_PAGE);
        if (editPage == null) {
            editPage = getDefaultEditPage();
        }
        if (log.isDebugEnabled()) {
            log.debug("init(PortletConfig) - defaultTitle=" + defaultTitle);
            log.debug("init(PortletConfig) - defaultCode=" + defaultCode);
            log.debug("init(PortletConfig) - defaultUrl=" + defaultUrl);
            log.debug("init(PortletConfig) - defaultViewPage=" + viewPage);
            log.debug("init(PortletConfig) - defaultEditPage=" + editPage);
        }
    }

    protected String getDefaultViewPage() {
        return DEFAULT_VIEW_VALUE;
    }

    protected String getDefaultEditPage() {
        return DEFAULT_EDIT_VALUE;
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.GenericPortlet#doView(javax.portlet.RenderRequest,
     *      javax.portlet.RenderResponse)
     */
    protected void doView(RenderRequest request, RenderResponse response)
            throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("doView(RenderRequest, RenderResponse)");
        }

        // set content info
        setViewContent(request);

        renderView(request, response, viewPage);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.GenericPortlet#doEdit(javax.portlet.RenderRequest,
     *      javax.portlet.RenderResponse)
     */
    protected void doEdit(RenderRequest request, RenderResponse response)
            throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("doEdit(RenderRequest, RenderResponse)");
        }

        // set message
        String msg = request.getParameter(MESSAGE);
        if (msg == null) {
            msg = "";
        }
        request.setAttribute(MESSAGE, msg);

        // set content info
        setEditContent(request);

        renderView(request, response, editPage);
    }

    protected void setViewContent(RenderRequest request) {
        PortletPreferences prefs = request.getPreferences();
        // set values as an attribute
        request.setAttribute(TITLE, prefs.getValue(TITLE, defaultTitle));
        request.setAttribute(CODE, prefs.getValue(CODE, defaultCode));
    }

    protected void setEditContent(RenderRequest request) {
        PortletPreferences prefs = request.getPreferences();
        // set values as an attribute
        request.setAttribute(TITLE, prefs.getValue(TITLE, defaultTitle));
        request.setAttribute(URL, prefs.getValue(URL, defaultUrl));
        request.setAttribute(CODE, prefs.getValue(CODE, defaultCode));
    }

    protected void renderView(RenderRequest request, RenderResponse response,
            String viewPage) throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log.debug("renderView(RenderRequest, RenderResponse, String)"
                    + " - viewPage=" + viewPage);
        }

        response.setContentType("text/html");

        // set title to response
        response.setTitle((String) request.getAttribute(TITLE));

        PortletContext context = getPortletContext();
        PortletRequestDispatcher rd = context.getRequestDispatcher(viewPage);
        rd.include(request, response);
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.portlet.Portlet#processAction(javax.portlet.ActionRequest,
     *      javax.portlet.ActionResponse)
     */
    public void processAction(ActionRequest request, ActionResponse response)
            throws PortletException, IOException {
        if (log.isDebugEnabled()) {
            log
                    .debug("processAction(ActionRequest, ActionResponse)"
                            + " - request.getPortletMode()="
                            + request.getPortletMode());
        }

        if (request.getPortletMode().equals(PortletMode.EDIT)) {
            ResourceBundle bundle = getPortletConfig().getResourceBundle(
                    request.getLocale());

            try {
                // store data
                storeContent(request);

                response.setRenderParameter(MESSAGE, bundle
                        .getString("WebPartsPortlet.msg.Updated"));
            } catch (Exception e) {
                log.error("Failed to update data.", e);
                response.setRenderParameter(MESSAGE, bundle
                        .getString("WebPartsPortlet.msg.FailedToUpdate"));
            }
        }

    }

    protected void storeContent(ActionRequest request) throws Exception {
        String codeValue = request.getParameter(CODE);
        if (codeValue == null) {
            codeValue = "";
        }
        String titleValue = request.getParameter(TITLE);
        if (titleValue == null) {
            titleValue = "";
        }
        String urlValue = request.getParameter(URL);
        if (urlValue == null) {
            urlValue = "";
        }
        PortletPreferences prefs = request.getPreferences();
        prefs.setValue(CODE, codeValue);
        prefs.setValue(TITLE, titleValue);
        prefs.setValue(URL, urlValue);
        prefs.store();
    }

}
