/* ====================================================================
   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for additional information regarding copyright ownership.
   The ASF licenses this file to You under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
==================================================================== */

package poi.support;

import java.text.DecimalFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class POIFormatterUtil {

	public static final String LINE_SEPARATOR = System
			.getProperty("line.separator");

	private POIFormatterUtil() {
	}

	/**
	 * printf formatter (support s,d,i,e,E,f,x,n only)
	 *
	 * @param s
	 * @param fmt
	 * @param parts
	 * @return s
	 */
	public static StringBuffer format(StringBuffer s, String fmt, Object[] parts) {
		Pattern pat = Pattern
				.compile("%[\\-+ #0]?[0-9\\.]*[hljztL]*[diuoxXeEfFgGaAcspn]");
		Matcher matcher = pat.matcher(fmt);
		int pi = 0;
		int previousEnd = 0;
		while (matcher.find()) {
			String src = matcher.group();
			s.append(fmt.substring(previousEnd, matcher.start()));
			String dest;
			char c = src.charAt(src.length() - 1);
			boolean isZeroPadding = (src.charAt(1) == '0');
			int maxLength = 0;
			int decimalLength = 0;
			switch (c) {
			case 's': {
				if (isZeroPadding) {
					throw new RuntimeException(
							"FormatFlagsConversionMismatchException: Conversion = s, Flags = 0");
				} else if (src.charAt(1) == 's') {
					// %s
					maxLength = 0;
				} else {
					// %5s : space padding
					maxLength = Integer.parseInt(src.substring(1,
							src.length() - 1));
				}
				if (null == parts) {
					dest = "null";
				} else {
					dest = getStringValue(parts[pi]);
				}
				break;
			}
			case 'd': {
				// example %d %5d %05d
				if (isZeroPadding) {
					maxLength = Integer.parseInt(src.substring(2,
							src.length() - 1));
				} else if (src.charAt(1) == 'd') {
					// %d
					maxLength = 0;
				} else {
					// %5d : space padding
					maxLength = Integer.parseInt(src.substring(1,
							src.length() - 1));
				}
				if (null == parts) {
					dest = "null";
					if (maxLength > dest.length()) {
						StringBuffer sp = new StringBuffer();
						for (int i = 0; i < maxLength - dest.length(); i++) {
							sp.append(' ');
						}
						sp.append(dest);
						dest = new String(sp);
					}
				} else {
					DecimalFormat df = new DecimalFormat("0");
					dest = df.format(getDecimalValue(parts[pi]));
				}
				break;
			}
			case 'e':
			case 'E': {
				// example %e %.5e %12.3e %012.3e %1.5e %E %.5E %1.5E %12.3E
				// %012.3E
				int dotIndex = src.indexOf('.');
				if (dotIndex < 0) {
					if (src.length() > 2) {
						maxLength = Integer.parseInt(src.substring(1,
								src.length() - 1));
					} else {
						maxLength = 0;
					}
					decimalLength = 0;
					DecimalFormat df = new DecimalFormat("0.000000E00");
					if (null == parts) {
						if (c == 'E') {
							dest = "NULL";
						} else {
							dest = "null";
						}
					} else {
						dest = df.format(getDoubleValue(parts[pi]));
					}
				} else {
					if (dotIndex > 1) {
						maxLength = Integer
								.parseInt(src.substring(1, dotIndex));
					} else {
						maxLength = 0;
					}
					decimalLength = Integer.parseInt(src.substring(
							dotIndex + 1, src.length() - 1));
					;
					StringBuffer dfmt = new StringBuffer("0.");
					for (int i = 0; i < decimalLength; i++) {
						dfmt.append('0');
					}
					dfmt.append("E00");
					if (null == parts) {
						if (c == 'E') {
							dest = "NULL";
						} else {
							dest = "null";
						}
						if (decimalLength > dest.length()) {
							StringBuffer sp = new StringBuffer();
							for (int i = 0; i < decimalLength - dest.length(); i++) {
								sp.append(' ');
							}
							sp.append(dest);
							dest = new String(sp);
						}
					} else {
						DecimalFormat df = new DecimalFormat(dfmt.toString());
						dest = df.format(getDoubleValue(parts[pi]));
					}
				}
				int eIndex = dest.indexOf('E');
				if (eIndex >= 0 && !('-' == dest.charAt(eIndex + 1))) {
					dest = dest.substring(0, eIndex) + c + "+"
							+ dest.substring(eIndex + 1);
				}
				if (c == 'e') {
					dest = dest.replace('E', 'e');
				}
				break;
			}
			case 'f': {
				// example %f %.5f %1.5f %12.3f %012.3f
				int dotIndex = src.indexOf('.');
				if (dotIndex < 0) {
					if (src.length() > 2) {
						maxLength = Integer.parseInt(src.substring(1,
								src.length() - 1));
					} else {
						maxLength = 0;
					}
					decimalLength = 0;
					DecimalFormat df = new DecimalFormat("0.000000");
					if (null == parts) {
						dest = "null";
					} else {
						dest = df.format(getDoubleValue(parts[pi]));
					}
				} else {
					if (dotIndex > 1) {
						maxLength = Integer
								.parseInt(src.substring(1, dotIndex));
					} else {
						maxLength = 0;
					}
					decimalLength = Integer.parseInt(src.substring(
							dotIndex + 1, src.length() - 1));
					;
					StringBuffer dfmt = new StringBuffer("0");
					if (decimalLength > 0) {
						dfmt.append('.');
					}
					for (int i = 0; i < decimalLength; i++) {
						dfmt.append('0');
					}
					if (null == parts) {
						dest = "null";
						if (dotIndex > dest.length()) {
							StringBuffer sp = new StringBuffer();
							for (int i = 0; i < maxLength - dest.length(); i++) {
								sp.append(' ');
							}
							sp.append(dest);
							dest = new String(sp);
						}
					} else {
						DecimalFormat df = new DecimalFormat(dfmt.toString());
						dest = df.format(getDoubleValue(parts[pi]));
					}
				}
				break;
			}
			case 'x': {
				// example %x %2x %02x %8x %08x
				StringBuffer dfmt = new StringBuffer();
				int len = 0;
				boolean hasLength = false;
				if (isZeroPadding) {
					// %02x : zero padding
					hasLength = true;
					len = Integer.parseInt(src.substring(2, src.length() - 1));
					for (int i = 0; i < len; i++) {
						dfmt.append('0');
					}
				} else if (src.charAt(1) == 'x') {
					// %x
				} else {
					// %2x : space padding
					hasLength = true;
					len = Integer.parseInt(src.substring(1, src.length() - 1));
					for (int i = 0; i < len; i++) {
						dfmt.append(' ');
					}
				}
				if (null == parts) {
					dest = "null";
					if (len > dest.length()) {
						StringBuffer sp = new StringBuffer();
						for (int i = 0; i < len - dest.length(); i++) {
							sp.append(' ');
						}
						sp.append(dest);
						dest = new String(sp);
					}
				} else {
					String hex = Long.toHexString(getDecimalValue(parts[pi])
							.longValue());
					dest = dfmt.toString() + hex;
					if (hasLength && hex.length() < len) {
						dest = dest.substring(dest.length() - len);
					} else {
						dest = hex;
					}
				}
				break;
			}
			case 'n': {
				// example %n
				// replace
				s.append(LINE_SEPARATOR);
				// next
				previousEnd = matcher.end();
				continue;
			}
			default:
				throw new IllegalArgumentException("Not supported:" + src);
			}
			// Zero Padding or Space Padding
			if (maxLength > dest.length()) {
				StringBuffer sp = new StringBuffer();
				int paddingLength = maxLength - dest.length();
				if (isZeroPadding && null != parts) {
					for (int i = 0; i < paddingLength; i++) {
						sp.append('0');
					}
				} else {
					for (int i = 0; i < paddingLength; i++) {
						sp.append(' ');
					}
				}
				s.append(sp);
			}
			// replace
			s.append(dest);
			// next
			previousEnd = matcher.end();
			pi++;
		}
		if (fmt.length() > previousEnd) {
			s.append(fmt.substring(previousEnd));
		}
		return s;
	}

	private static String getStringValue(Object value) {
		return value.toString();
	}

	private static Long getDecimalValue(Object value) {
		Long result;
		if (value instanceof Integer) {
			result = new Long(((Integer) value).intValue());
		} else if (value instanceof Short) {
			result = new Long(((Short) value).shortValue());
		} else if (value instanceof Byte) {
			result = new Long(((Byte) value).byteValue());
		} else if (value instanceof Long) {
			result = (Long) value;
		} else {
			throw new RuntimeException("not supported.");
		}
		return result;
	}

	private static Double getDoubleValue(Object value) {
		Double result;
		if (value instanceof Float) {
			result = new Double(((Float) value).floatValue());
		} else if (value instanceof Double) {
			result = (Double) value;
		} else if (value instanceof Long) {
			result = new Double(((Long) value).longValue());
		} else {
			throw new RuntimeException("not supported.");
		}
		return result;
	}
}
