#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Command line options -
# Copyright (C) 2013-2017 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
OPTIONS_ERRNO=0	# 0: no error, 1: configuration error, 2: internal error
OPTIONS_SHIFT=0

# ============= Database of options which are given at each run and not saved =============
# [Syntax of option databases]
# short_name, long_name, variable, defult_vaule, set_value
# Columns are delimited by tab characters.
options_db_onetime ()
{
	cat << eof
h	help	opt_help_mode	0	1
h	short-help	opt_help_mode	0	1
H	long-help	opt_help_mode	0	2
V	show-version	opt_show_version	no	yes
a	batch-mode	opt_batch_mode	no	yes
N	reset-targets	opt_reset_targets	no	yes
M	reset-minor-options	opt_reset_minor_options	no	yes
L	reload-conf	opt_reload_conf	no	yes
i	allow-new-targets	opt_allow_new_targets	no	yes
eof
}

# ============= Database of options which are saved and renewable at each run =============
options_db_saved_and_renewable ()
{
	cat << eof
C	apply-default-config	opt_apply_default_config	no	yes
A	non-interactive-ports-only	opt_batch_ports_only	no	yes
I	interactive-ports-only	opt_interactive_ports_only	no	yes
s	avoid-vulnerability-check	opt_avoid_vulner	no	yes
q	skip-unchanged-ports	opt_skip_unchanged	no	yes
b	include-buildtime-dependencies	opt_include_buildtime_dependencies	no	yes
B	exclude-runtime-dependencies	opt_include_runtime_dependencies	yes	no
d	keep-distfiles	opt_keep_distfiles	no	yes
k	suppress-self-upadte	opt_suppress_self_upadte	no	yes
g	suppress-pkgtools-upadte	opt_suppress_pkgtools_upadte	no	yes
c	suppress-clean-obsolete-database	opt_suppress_obsolete_db_clean	no	yes
D	suppress-entire-inspection-distinfo	opt_inspect_entire_distinfo	yes	no
n	dry-run	opt_dry_run	no	yes
X	deselect-all	opt_dialog_auto	no	none
Y	select-all	opt_dialog_auto	no	all
G	use-prebuilt-package	opt_inst_by_pkg_if_can	no	yes
l	use-legacy-package-for-missing-pkgng	opt_use_legacy_pkg_for_missing_pkgng	no	yes
f	disallow-force-continuation-for-looped-dependency	opt_disallow_force_continuation_for_looped_dependency	no	yes
eof
}

# ============= Database of options which are saved and renewable only at the initial runs of redo command (on target specification) =============
options_db_saved_and_renewable_in_redo_on_target ()
{
	cat << eof
t	target-and-dependents	opt_target_dependents	''	:
T	target-and-requirements	opt_target_requirements	''	:
O	target-only-itself	opt_target_itself	''	:
o	only-target-scope	opt_only_target_scope	no	yes
eof
}

# ============= Database of options which are saved and renewable only at the initial runs of redo command (on configuration files) =============
options_db_saved_and_renewable_in_redo_on_conf ()
{
	cat << eof
p	load-pkgtoolsconf-as-default	opt_load_pkgtoolsconf	undef	default
P	load-pkgtoolsconf-as-override	opt_load_pkgtoolsconf	undef	override
Q	ignore-pkgtoolsconf	opt_load_pkgtoolsconf	undef	no
eof
}

# ============= Database of options which are saved and not renewable until the temporary database is cleaned =============
options_db_saved_and_non_renewable ()
{
	cat << eof
eof
}

# ============= Database of all options to be saved =============
options_db_saved ()
{
	options_db_saved_and_renewable
	options_db_saved_and_renewable_in_redo_on_target
	options_db_saved_and_renewable_in_redo_on_conf
	options_db_saved_and_non_renewable
}

# ============= Database of all options =============
options_db_all ()
{
	options_db_saved
	options_db_onetime
}

# ============= Database for selected option sets =============
options_db ()
{
	if [ $# -gt 0 ]
	then
		while [ $# -gt 0 ]
		do
			case $1 in
			renewable_anytime)
				options_db_saved_and_renewable;;
			renewable_in_redo_on_target)
				options_db_saved_and_renewable_in_redo_on_target;;
			renewable_in_redo_on_conf)
				options_db_saved_and_renewable_in_redo_on_conf;;
			non_renewable)
				options_db_saved_and_non_renewable;;
			saved)
				options_db_saved;;
			onetime)
				options_db_onetime;;
			all)
				options_db_all;;
			*)
				message_echo "ERROR: Invalid argument [$1] for options_inverse_parse" >&2
				exit 1;;
			esac
			shift
		done
	else
		options_db_all
	fi | sort -u
}

# ============= Convert from a short option name to the corresponding variable name =============
options_convert_shortname_to_varname ()
{
	local shortname
	shortname=$1
	options_db_all | grep -m 1 "^${shortname}[[:space:]]" | cut -f 3 | sort -u
}

# ============= Dump all command line arguments and options =============
options_dump_args ()
{
	echo -n "`echo "$0" | sed -E 's/(.)/\\\\\\1/g'`"
	while [ $# -ge 1 ]
	do
		echo -n " `echo "$1" | sed -E 's/(.)/\\\\\\1/g'`"
		shift
	done
}

# ============= Get default option settings =============
options_default ()
{
	options_db_all | cut -f 3,4 | sort -u | sed -E 's/[[:space:]]/=/'
}

# ============= Initialize option settings with the default values =============
options_set_default ()
{
	options_default > ${TMPDIR}/default_optvals.sh
	. "${TMPDIR}/default_optvals.sh"
}

# ============= Get the argument for getopts =============
options_getopts_arg ()
{
	options_db_all | cut -f 1,5 | grep -v '^[[:space:]]' | sort -u \
		| sed -E 's/[[:space:]]:$/:/;s/[[:space:]].*$//' | tr -d '\n'
}

# ============= Get option settings =============
options_getopts_only_short ()
{
	local OPTERR getopts_option option command OPTARG nshifts val
	OPTERR=0
	getopts_option=`options_getopts_arg`
	while getopts $getopts_option option > /dev/null
	do
		val=`str_escape_replaceval "$OPTARG"`
		command=`options_db_all | grep -m 1 "^${option}[[:space:]]" \
			| cut -f 3,5 | sed -E 's/[[:space:]]/=/;s/:$/'$val'/' || :`
		[ -n "$command" ] || return 1
		echo -n "$command;OPTIND=$OPTIND;"
	done
}

# ============= Get option settings =============
options_getopts ()
{
	local OPTIND option command nshifts val erropt
	nshifts=0
	erropt=
	OPTIONS_SHIFT=0
	OPTIONS_ERRNO=2
	while [ $# -gt 0 ]
	do
		eval `( set -e; options_getopts_only_short "$@")`
		shift $(($OPTIND-1))
		nshifts=$(($nshifts+$OPTIND-1))
 		[ "x$1" = "x--" ] && break
		expr "x$1" : '^x-.*' > /dev/null || break
		[ $# -eq 0 ] && break
		erropt=$1
		option=`expr "x$1" : '^x--\([a-zA-Z0-9][a-zA-Z0-9-]*\).*'` || break
		command=`options_db_all | cut -f 2,3,5 | grep -m 1 "^${option}[[:space:]]" \
			| cut -f 2,3 | sed -E 's/[[:space:]]/=/' || :`
		[ -n "$command" ] || break
		if expr "$command" : '^[^=]*=:$' > /dev/null
		then
			expr "x$1" : '^x--[a-zA-Z0-9][a-zA-Z0-9-]*=.*' > /dev/null || break
			val=`expr "x$1" : '^x--[a-zA-Z0-9][a-zA-Z0-9-]*=\(.*\)'` || :
			eval `expr "$command" : '^\([^=]*=\).*'`\$val
		else
			[ "x$1" = "x--$option" ] || break
			eval $command
		fi
		nshifts=$(($nshifts+1))
		shift
		OPTIND=1
		erropt=
	done
	[ -z "$erropt" ] \
		|| { message_echo "ERROR: Illeagal option [$erropt]" >&2; OPTIONS_ERRNO=1; return 1; }
	OPTIONS_SHIFT=$nshifts
	OPTIONS_ERRNO=0
}

# ============= Filter option value definitions to pick up selected groups =============
options_filter ()
{
	grep `options_db "$@" | cut -f 3 | sed 's/^/-e ^/;s/$/=/'`
}

# ============= Compose an option set in a command line form from the current settings =============
options_inverse_parse ()
{
	options_db "$@" | while read short_name long_name variable defult_vaule set_value
	do
		eval val=\$$variable
		eval val_def=$defult_vaule
		eval val_set=$set_value
		if [ "x$set_value" = x: ]
		then
			[ "x$val" = "x$val_def" ] \
				|| misc_get_all_vardefs | grep -m 1 "^$variable=" \
					| sed "s/^[^=]*=/-$short_name /"
		elif [ "x$val" = "x$val_set" ]
		then
			echo -n "-$short_name"
			echo
		fi
	done | tr '\n' ' ' | sed 's/ *$//'
}

# ============= Filter option value definitions to pick up non-default ones =============
options_filter_configured ()
{
	local tmpptn
	tmpptn=${TMPDIR}/options_filter_configured:ptn
	options_db "$@" | while read short_name long_name variable defult_vaule set_value
	do
		eval val=\$$variable
		eval val_def=$defult_vaule
		[ "x$val" = "x$val_def" ] || echo "^$variable="
	done > $tmpptn
	grep -f "$tmpptn"
}

# ============= Check inclusion of invalid options intending to be renewed =============
options_chk_invalid_optvals_renewal ()
{
	local  dbgroup comment_condition invalid_options num_invalid_options term_opts delimiter
	dbgroup=$1
	comment_condition=$2
	misc_get_all_vardefs | options_filter_configured "$dbgroup" \
		> ${TMPDIR}/options_renewed_optvals:invalid_options_defs
	invalid_options=`options_inverse_parse "$dbgroup" < ${TMPDIR}/options_renewed_optvals:invalid_options_defs`
	num_invalid_options=`wc -l < ${TMPDIR}/options_renewed_optvals:invalid_options_defs`
	if [ $num_invalid_options -ne 0 ]
	then
		term_opts="Option $invalid_options is"
		[ $num_invalid_options -gt 1 ] && term_opts="Options $invalid_options are"
		delimiter=
		[ -n "$comment_condition" ] && delimiter=' '
		message_echo "ERROR: $term_opts invalid in restarted runs${delimiter}${comment_condition}." >&2
		return 1
	fi
}

# ============= Get renewed option value definitions as well as checking their validity =============
options_renewed_optvals ()
{
	local flag_shortname dbgroup flag_varname flag
	flag_shortname=$1
	dbgroup=$2
	flag_varname=`options_convert_shortname_to_varname "$flag_shortname"`
	eval "flag=\$$flag_varname"
	if [ "x$flag" = xyes ]
	then
		misc_get_all_vardefs | options_filter "$dbgroup"
	else
		options_chk_invalid_optvals_renewal "$dbgroup" "without -$flag_shortname"
	fi
}

# ============= Select new target ports specified by globs if duplicated for each glob pattern =============
options_select_new_ports_if_duplicated ()
{
	local option dstfile optargs tmp_target_ports tmp_existing_ports tmp_new_ports itemlist globs_list nlines iline title desc
	option=$1
	dstfile=$2
	shift 2
	optargs="$*"
	tmp_target_ports=${TMPDIR}/options_select_new_ports_if_duplicated:target_ports
	tmp_existing_ports=${TMPDIR}/options_select_new_ports_if_duplicated:existing_ports
	tmp_new_ports=${TMPDIR}/options_select_new_ports_if_duplicated:new_ports
	globs_list=${TMPDIR}/options_select_new_ports_if_duplicated::globs_list
	itemlist=${TMPDIR}/options_select_new_ports_if_duplicated::itemlist
	cp /dev/null "$dstfile.tmp"
	echo "$optargs" | sed -E 's/[ :]+/\
/g' | grep -v '^$' | sort -u > $globs_list
	nlines=`wc -l < $globs_list`
	iline=1
	while [ $iline -le $nlines ]
	do
		glob=`sed -n ${iline}p "$globs_list"`
		iline=$(($iline+1))
		rm -f "$tmp_target_ports"
		pkgsys_register_evaluated_globs add "$tmp_target_ports" "$glob"
		if [ `cat "$tmp_target_ports" 2> /dev/null | wc -l` -eq 0 ]
		then
			message_echo "WARNING: No matching port for target glob [$glob]." >&2
			continue
		fi
		cp /dev/null "$tmp_existing_ports"
		cp /dev/null "$tmp_new_ports"
		while read origin
		do
			origin_regexp=`str_escape_regexp "$origin"`
			if pkgsys_pkg_info_eO "$origin" \
				|| cat "${DBDIR}/installed_ports" "${DBDIR}/targets_specified_so_far" 2> /dev/null \
					| grep -q -E "$origin_regexp"
			then
				echo "$origin" >> $tmp_existing_ports
			else
				echo "$origin" >> $tmp_new_ports
			fi
		done < $tmp_target_ports
		cat "$tmp_existing_ports" >> $dstfile.tmp
		if [ $opt_allow_new_targets = yes ]
		then
			if [ `wc -l < $tmp_new_ports` -gt 1 ]
			then
				title="Uninspected ports matching a glob pattern [$glob] for -$option option"
				desc='Checked ones are newly installed.'
				while read origin
				do
					echo "$origin '' off"
					printf '%s\t""\t%s\n' "$origin" '' "$val"
				done < $tmp_new_ports > $itemlist
				misc_dialog_checklist "$title" "$desc" "$dstfile.tmp" "$itemlist"
			else
				cat "$tmp_new_ports" >> $dstfile.tmp
			fi
		else
			[ `wc -l < $tmp_new_ports` -eq 0 ] || \
				message_echo "WARNING: Ignored not-yet-installed ports for target glob [$glob]." >&2
			if [ `wc -l < $tmp_existing_ports` -eq 0 ]
			then
				message_echo "WARNING: No available matching port for target glob [$glob]." >&2
			fi
		fi
	done
	mv "$dstfile.tmp" "$dstfile"
}

# ============= Get the database type of dependency algorithm for the current option setting =============
options_get_dependency_type ()
{
	case $opt_include_buildtime_dependencies+$opt_include_runtime_dependencies in
	yes+yes)
		echo all;;
	no+yes)
		echo run;;
	yes+no)
		echo build;;
	no+no)
		echo none;;
	esac
}

# ============= Parse the database type tag of dependency algorithm and reflect it to option values =============
options_parse_dependency_type ()
{
	local deptag
	deptag=$1
	case $deptag in
	all)
		opt_include_buildtime_dependencies=yes
		opt_include_runtime_dependencies=yes
		;;
	run)
		opt_include_buildtime_dependencies=no
		opt_include_runtime_dependencies=yes
		;;
	build)
		opt_include_buildtime_dependencies=yes
		opt_include_runtime_dependencies=no
		;;
	none)
		opt_include_buildtime_dependencies=no
		opt_include_runtime_dependencies=no
		;;
	esac
}

# ============= Get the database level of dependency algorithm for the current option setting =============
options_get_dependency_level ()
{
	case $opt_only_target_scope in
	yes)
		echo direct;;
	no)
		echo full;;
	esac
}

# ============= Parse the database level tag of dependency algorithm and reflect it to option values =============
options_parse_dependency_level ()
{
	local level
	level=$1
	case $level in
	direct)
		opt_only_target_scope=yes;;
	full)
		opt_only_target_scope=no;;
	esac
}

# ============= Get the message term of dependency algorithm for the current option setting =============
options_get_dependency_msgterm ()
{
	local scope
	scope=`options_get_dependency_level`
	case $opt_include_buildtime_dependencies+$opt_include_runtime_dependencies in
	yes+yes)
		echo "$scope run- and build-time";;
	no+yes)
		echo "$scope run-time";;
	yes+no)
		echo "$scope build-time";;
	no+no)
		echo 'no';;
	esac
}

# ============= Get effective option value about importing pkgtools.conf(5) =============
options_get_effective_opt_load_pkgtoolsconf ()
{
	local effective_opt_load_pkgtoolsconf
	if [ $opt_load_pkgtoolsconf = undef ]
	then
		if [ -e "$PKGTOOLSCONF" ] && which -s portupgrade
		then
			effective_opt_load_pkgtoolsconf=default
		else
			effective_opt_load_pkgtoolsconf=no
		fi
	elif ! which -s portupgrade
	then
		message_echo "WARNING: pkgtools.conf is ignored because portupgrade is not installed." >&2
		effective_opt_load_pkgtoolsconf=no
	fi
	echo "$effective_opt_load_pkgtoolsconf"
}

