/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.engine;

import java.io.IOException;
import java.io.StringReader;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Collections;

import net.morilib.db.misc.Rational;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationCursor;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.schema.MemorySqlSchema;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.schema.SqlSchemata;
import net.morilib.db.sql.DbSqlLexer;
import net.morilib.db.sql.DbSqlParser;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.dml.SqlSetExpression;
import junit.framework.TestCase;

public class RelationsCsvTest extends TestCase {

	private static final SimpleDateFormat FM1 =
			new SimpleDateFormat("yyyy/MM/dd");

	SqlSchema sch() {
		SqlCreateTable d;

		try {
			d = SqlSchemata.guessTable("idols",
					RelationsCsvTest.class.getResourceAsStream(
							"/net/morilib/db/engine/idols2.csv"));
			return MemorySqlSchema.readCSV(d,
					RelationsCsvTest.class.getResourceAsStream(
							"/net/morilib/db/engine/idols2.csv"));
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	Object get(String t) {
		DbSqlLexer l;

		try {
			l = new DbSqlLexer(new StringReader(t));
			return new DbSqlParser().parse(l);
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	RelationCursor itr(String s) {
		Object o = get(s);
		DefaultSqlEngine e = new DefaultSqlEngine();
		Relation r;

		try {
			r = (Relation)e.visit(sch(),
					(SqlSetExpression)o,
					Collections.emptyList());
			return r.iterator();
		} catch (IOException e1) {
			throw new RuntimeException(e1);
		} catch (SQLException e1) {
			throw new RuntimeException(e1);
		}
	}

	RelationTuple eqs(RelationCursor r) {
		try {
			assertTrue(r.hasNext());
			return r.next();
		} catch (IOException e) {
			throw new RuntimeException(e);
		} catch (SQLException e) {
			throw new RuntimeException(e);
		}
	}

	void eqs(RelationTuple t, String s, Object o) {
		try {
			assertEquals(o, t.get(s));
		} catch (SQLException e) {
			throw new RuntimeException(s, e);
		}
	}

	void eqs(RelationTuple t, String s, long o) {
		eqs(t, s, Rational.valueOf(o));
	}

	void eqd(RelationTuple t, String s, String o) {
		try {
			assertEquals(o, ((Rational)t.get(s)).toBigDecimal(4).toString());
		} catch (SQLException e) {
			throw new RuntimeException(s, e);
		}
	}

	void eqt(RelationTuple t, String s, String o) {
		try {
			assertEquals(o, FM1.format((java.util.Date)t.get(s)));
		} catch (SQLException e) {
			throw new RuntimeException(s, e);
		}
	}

	public void testRel0001() {
		RelationCursor r;
		RelationTuple t;

		r = itr("SELECT 名前, B, 生年月日 FROM IDOLS");
		t = eqs(r);  eqs(t, "名前", "天海春香");
		eqs(t, "B", 83);
		eqt(t, "生年月日", "1996/04/13");
		t = eqs(r);  eqs(t, "名前", "高槻やよい");
		eqt(t, "生年月日", "1999/03/25");
		eqs(t, "B", 74);
		t = eqs(r);  eqs(t, "名前", "菊地真");
		eqt(t, "生年月日", "1996/08/29");
		eqs(t, "B", 75);
		t = eqs(r);  eqs(t, "名前", "我那覇響");
		eqs(t, "B", 83);
		eqt(t, "生年月日", "1997/10/10");
		assertFalse(r.hasNext());
	}

}
