/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.relations;

import java.io.IOException;
import java.math.MathContext;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import net.morilib.db.engine.SqlEngine;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.schema.SqlSchema;
import net.morilib.db.sqlcs.dml.SqlJoinType;
import net.morilib.db.sqlcs.dml.SqlOrderBy;
import net.morilib.db.sqlcs.dml.SqlTableColumn;

public final class Relations {

	private static class Dual extends AbstractRelation {

		private static final RelationTuple X = new RelationTuple() {

			@Override
			public Object get(String name) throws SQLException {
				throw ErrorBundle.getDefault(10009, name);
			}

			@Override
			public RelationTuple copy() {
				return new DefaultRelationTuple(toMap());
			}

			@Override
			public Map<String, Object> toMap() {
				Map<String, Object> m;

				m = new LinkedHashMap<String, Object>();
				m.put("\u001e", "");
				return m;
			}

		};

		@Override
		public RelationCursor iterator() {
			return new RelationCursor() {

				private RelationTuple v = X;

				@Override
				public boolean hasNext() {
					return v != null;
				}

				@Override
				public RelationTuple next(
						) throws IOException, SQLException {
					if(v != null) {
						v = null;
						return X;
					} else {
						throw new NoSuchElementException();
					}
				}

			};
		}

		@Override
		public List<String> getColumnNames() {
			return Collections.singletonList("\u001e");
		}

	}

	/**
	 * 
	 */
	public static final Relation DUAL = new Dual();

	/**
	 * 
	 */
	public static final MathContext MC = new MathContext(6);

	/**
	 * 
	 */
	public static final RelationTuple NULLTUPLE = new RelationTuple() {

		public Object get(String name) throws SQLException {
			throw ErrorBundle.getDefault(10009, name);
		}

		public RelationTuple copy() throws SQLException {
			return new DefaultRelationTuple(toMap());
		}

		public Map<String, Object> toMap() throws SQLException {
			return Collections.emptyMap();
		}

	};

	public static final RelationExpression
	ANY = new RelationExpression() {

		@Override
		public Object eval(SqlEngine v, SqlSchema f,
				RelationTuple t, RelationAggregate m,
				List<String> group,
				List<Object> h) throws IOException, SQLException {
			return this;
		}

		@Override
		public Object init(RelationAggregate m) throws SQLException {
			return this;
		}

		@Override
		public boolean isAggregate() {
			return false;
		}

	};

	//
	private Relations() {}

	public static Relation join(SqlEngine v, SqlSchema f,
			RelationExpression on, final Relation a, final Relation b,
			RelationAggregate m, List<String> group,
			final SqlJoinType tp,
			List<Object> h) throws IOException, SQLException {
		final Relation c = new VirtualCrossJoinRelation(a, b);
		final Collection<RelationTuple> l;
		final boolean[] fl = new boolean[1];
		RelationCursor i = c.iterator();
		RelationTuple t;

		l = new ArrayList<RelationTuple>();
		fl[0] = false;
		i.addRelationScanListener(new RelationScanListener() {

			@Override
			public void scaned(
					RelationScanEvent e) throws SQLException {
				if(tp == SqlJoinType.LEFT && !fl[0]) {
					l.add(new OuterRelationTuple(
							a,
							e.getTuple(),
							c.getColumnNames()));
				}
				fl[0] = false;
			}

		});

		while(i.hasNext()) {
			t = i.next();
			if(on.test(v, f, t, m, group, h).isTrue()) {
				l.add(t.copy());
				fl[0] = true;
			}
		}
		return new TableRelation(c.getColumnNames(), l);
	}

	public static OperatedRelation operate(SqlEngine v,
			SqlSchema f, Relation a, List<SqlTableColumn> on,
			List<String> g, List<Object> hl,
			boolean[] ag) throws IOException, SQLException {
		RelationExpression[] x;
		List<String> h, d, c;
		String[] s;

		if(on.size() == 1 && on.get(0) == SqlTableColumn.WILDCARD) {
			c = a.getColumnNames();
			d = new ArrayList<String>();
			for(int i = 0; i < c.size(); i++) {
				if(a instanceof IsolatedRelation !=
						c.get(i).indexOf('.') < 0) {
					d.add(c.get(i));
				}
			}

			s = d.toArray(new String[0]);
			x = new RelationExpression[d.size()];
			for(int i = 0; i < d.size(); i++) {
				x[i] = new RelationRefer(s[i]);
			}
			h = null;
		} else {
			x = new RelationExpression[on.size()];
			s = new String[on.size()];
			for(int i = 0; i < on.size(); i++) {
				x[i] = v.visit(on.get(i).getExpression(), hl);
				if((s[i] = on.get(i).getAs()) == null &&
						x[i] instanceof RelationRefer) {
					s[i] = ((RelationRefer)x[i]).getId();
				}
				ag[0] = ag[0] || x[i].isAggregate();
			}
			h = g != null ? g : ag[0] ? new ArrayList<String>() : null;
		}
		return new OperatedRelation(a, v, f, Arrays.asList(x),
				Arrays.asList(s), h, hl);
	}

	static int compareKey(List<String> l, RelationTuple a,
			RelationTuple b) throws SQLException {
		Object x, y;
		int c;

		for(String s : l) {
			x = a.get(s);
			y = b.get(s);
			c = RelationExpression.cmp(x, y);
			if(c != 0)  return c;
		}
		return 0;
	}

	public static Comparator<RelationTuple> getComparator(
			final Relation r, final List<SqlOrderBy> l) {
		return new Comparator<RelationTuple>() {

			@Override
			public int compare(RelationTuple a, RelationTuple b) {
				Object x, y;
				int c;

				try {
					for(SqlOrderBy o : l) {
						x = a.get(o.getName());
						y = b.get(o.getName());
						c = RelationExpression.cmp(x, y);
						c = o.isAscending() ? c : -c;
						if(c != 0)  return c;
					}
					return compareKey(r.getColumnNames(), a, b);
				} catch(SQLException e) {
					// TODO check
					throw new RuntimeException(e);
				}
			}

		};
	}

	static Object _get(RelationTuple v, Relation r,
			String n) throws SQLException {
		int c = n.indexOf('.');

		if(!(r instanceof SingleTableRelation)) {
			return v.get(n);
		} else if(((SingleTableRelation)r).getName().equals(
				n.substring(0, c))) {
			return v.get(n.substring(c + 1));
		} else {
			return null;
		}
	}

	static void _add(List<String> cols, Relation r) {
		for(String s : r.getColumnNames()) {
			if(r instanceof SingleTableRelation) {
				s = ((SingleTableRelation)r).getName() + "." + s;
			}
			cols.add(s);
		}
	}

}
