/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import net.morilib.db.map.CSVDataMapper;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;

public class MemorySqlSchema implements SqlSchema {

	private Map<String, NamedRelation> relations;
	private Map<String, SqlCreateTable> ddls;

	MemorySqlSchema() {}

	/**
	 * 
	 * @param csvs
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static MemorySqlSchema readCSVs(
			String... csvs) throws IOException, SQLException {
		MemorySqlSchema m = new MemorySqlSchema();
		CSVDataMapper p = new CSVDataMapper();
		BufferedReader b;
		SqlCreateTable d;
		NamedRelation r;

		m.ddls = new HashMap<String, SqlCreateTable>();
		m.relations = new HashMap<String, NamedRelation>();
		for(String s : csvs) {
			d = SqlSchemata.guessTable(s);
			b = new BufferedReader(new InputStreamReader(
					new FileInputStream(s)));
			b.readLine();
			r = p.read(d.getName(), null, d, b);
			m.ddls.put(d.getName(), d);
			m.relations.put(d.getName(), r);
		}
		return m;
	}

	@Override
	public NamedRelation readRelation(String name,
			String as) throws IOException, SQLException {
		String s = name.toUpperCase();

		if(!relations.containsKey(s)) {
			throw ErrorBundle.getDefault(10015, name);
		} else if(as == null) {
			return relations.get(s);
		} else {
			return new TableRenameRelation(relations.get(s), as);
		}
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> z) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		String s = name.toUpperCase();

		if(!ddls.containsKey(s)) {
			throw ErrorBundle.getDefault(10015, name);
		} else {
			return ddls.get(s);
		}
	}

	@Override
	public Collection<String> getTableNames() {
		return Collections.unmodifiableCollection(relations.keySet());
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return relations.containsKey(name);
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#fork()
	 */
	@Override
	public SqlSchema fork() {
		return this;
	}

}
