﻿// //////////////////////////////////////////////////////////////////////////
//
// utility.cs
//              Sarbo Project
//          Powerd by CACTUS SOFTWARE <http://www.cactussoft.co.jp/>
//
//                                              ver 1.0  2009/08/15
// This file is part of Sarbo.
//
// Sarbo is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// Sarbo is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Sarbo.  If not, see <http://www.gnu.org/licenses/>.
//
// //////////////////////////////////////////////////////////////////////////
using System;
using System.Collections.Generic;
using System.Text;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;

namespace DShowCAC
{
	// //////////////////////////////////////////////////////////////////////
	// Rect

	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public struct Rect
	{
		public int		left;
		public int		top;
		public int		right;
		public int		bottom;
	}

	// //////////////////////////////////////////////////////////////////////
	// PIN_DIRECTION

	/// <summary>
	/// PIN_DIRECTION
	/// </summary>
	public enum PinDirection : int
	{
		Input,
		Output,
	}

	// //////////////////////////////////////////////////////////////////////
	// FILTER_STATE

	/// <summary>
	/// FILTER_STATE
	/// </summary>
	public enum FilterState : int
	{
		Stopped,
		Paused,
		Running,
	}

	// //////////////////////////////////////////////////////////////////////
	// PinInfo

	/// <summary>
	/// wrapper class of PIN_INFO
	/// </summary>
	[StructLayout(LayoutKind.Sequential, CharSet=CharSet.Unicode), ComVisible(false)]
	public class PinInfo
	{
		public IBaseFilter		pFilter;
		public PinDirection		dir;
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst=128)]
		public string			achName;
	}

	// //////////////////////////////////////////////////////////////////////
	// FilterInfo

	/// <summary>
	/// wrapper class of FILTER_INFO
	/// </summary>
	[StructLayout(LayoutKind.Sequential, CharSet=CharSet.Unicode), ComVisible(false)]
	public class FilterInfo
	{
		[MarshalAs(UnmanagedType.ByValTStr, SizeConst=128)]
		public string			achName;
		public IFilterGraph		pGraph;
	}

	// //////////////////////////////////////////////////////////////////////
	// ReferenceTime

	/// <summary>
	/// wrapper class of REFERENCE_TIME
	/// </summary>
	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public class ReferenceTime
	{
		public long		Value;

		public ReferenceTime(long refTime)
		{
			Value = refTime;
		}
		public ReferenceTime(TimeSpan timeSpan)
		{
			Value = timeSpan.Ticks;
		}

		// //////////////////////////////////////////////////////////////////
		// Property

		public TimeSpan TimeSpan
		{
			get		{		return new TimeSpan(Value);			}
			set		{		Value = value.Ticks;				}
		}
	}

	// //////////////////////////////////////////////////////////////////////
	// ReferenceInt

	/// <summary>
	/// wrapper class of int
	/// </summary>
	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public class ReferenceInt
	{
		public int		Value;

		public ReferenceInt(int refInt)
		{
			Value = refInt;
		}
	}

	// //////////////////////////////////////////////////////////////////////
	// ReferenceStr

	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public class ReferenceStr
	{
		public IntPtr	Value;

		// //////////////////////////////////////////////////////////////////
		// Property

		public string String
		{
			get		{		return Marshal.PtrToStringUni(Value);		}
		}

		public void Free()
		{
			Marshal.FreeCoTaskMem(Value);
		}
	}

	// //////////////////////////////////////////////////////////////////////
	// ReferenceObj

	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public class ReferenceObj
	{
		public object	Value;
	}

	// //////////////////////////////////////////////////////////////////////
	// AMMediaType

	/// <summary>
	/// wrapper class of AM_MEDIA_TYPE
	/// </summary>
	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public class AMMediaType
	{
		public Guid			majortype;
		public Guid			subtype;
		[MarshalAs(UnmanagedType.Bool)]
		public bool			bFixedSizeSamples;
		[MarshalAs(UnmanagedType.Bool)]
		public bool			bTemporalCompression;
		public int			lSampleSize;
		public Guid			formattype;
		public IntPtr		pUnk;
		public int			cbFormat;
		public IntPtr		pbFormat;
	}

	// //////////////////////////////////////////////////////////////////////
	// VideoInfoHeader

	/// <summary>
	/// wrapper class of VIDEOINFOHEADER
	/// </summary>
	[StructLayout(LayoutKind.Sequential), ComVisible(false)]
	public class VideoInfoHeader
	{
		public Rect				SrcRect;
		public Rect				TagRect;
		public int				BitRate;
		public int				BitErrorRate;
		public long				AvgTimePerFrame;
		public BitmapInfoHeader	BmiHeader;
	}

	[StructLayout(LayoutKind.Sequential, Pack=2), ComVisible(false)]
	public struct BitmapInfoHeader
	{
		public int		biSize;
		public int		biWidth;
		public int		biHeight;
		public short	biPlanes;
		public short	biBitCount;
		public int		biCompression;
		public int		biSizeImage;
		public int		biXPelsPerMeter;
		public int		biYPelsPerMeter;
		public int		biClrUsed;
		public int		biClrImportant;
	}

	// //////////////////////////////////////////////////////////////////////
	// ROT

	[ComVisible(false)]
	public class ROT
	{
		public static int AddToRot(
			object			iUnkGraph,
			out int			register)
		{
			register = 0;
			int result = HResult.OK;

			IRunningObjectTable iROT = null;
			IMoniker iMoniker = null;

			try
			{
				result = GetRunningObjectTable(0, out iROT);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

				IntPtr iUnk = Marshal.GetIUnknownForObject(iUnkGraph);

				string item = string.Format(
					"FilterGraph {0:x} pid {1:x}", (int)iUnk, GetCurrentProcessId());

				Marshal.Release(iUnk);

				result = CreateItemMoniker("!", item, out iMoniker);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);
				
				register = iROT.Register(RotflagsRegistrationKeepsAlive, iUnkGraph, iMoniker);
				return result;
			}
			catch (Exception)
			{
				return result;
			}
			finally
			{
				if (iROT != null)
					Marshal.ReleaseComObject(iROT); iROT = null;
				if (iMoniker != null)
					Marshal.ReleaseComObject(iMoniker); iMoniker = null;
			}
		}

		public static bool RemoveFromRot(ref int register)
		{
			IRunningObjectTable iROT = null;

			try
			{
				int result = GetRunningObjectTable(0, out iROT);
				if (HResult.Failed(result))		Marshal.ThrowExceptionForHR(result);

				iROT.Revoke(register);
				register = 0;
				return true;
			}
			catch (Exception)
			{
				return false;
			}
			finally
			{
				if (iROT != null)
					Marshal.ReleaseComObject(iROT); iROT = null;
			}
		}

		private const int RotflagsRegistrationKeepsAlive = 1;

		[DllImport("ole32.dll")]
		static extern int GetRunningObjectTable(uint reserved, out IRunningObjectTable pprot);

		[DllImport("ole32.dll", CharSet=CharSet.Unicode)]
		static extern int CreateItemMoniker(string delim, string item, out IMoniker ppmk);

		[DllImport("kernel32.dll")]
		static extern int GetCurrentProcessId();
	}
}

// //////////////////////////////////////////////////////////////////////////
