﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "BehaviorSettingsCollection.h"

// private headers
#include "DummyWindow.h"
#include "Foundation.h"
#include "Path.h"
#include "PopupMenu.h"
#include "Util.h"


namespace bearmini
{
    ///
    ///  コレクションに [ファイル名別動作] の情報（ビヘイビア設定）を追加します。
    ///
    ///  @param[in] bml 追加したい [ファイル名別動作] の情報
    ///
    void BehaviorSettingsCollection::Add(const BehaviorSettingsForPattern& behaviorSettingsForPattern)
    {
        m_allBehaviorSettings.push_back(behaviorSettingsForPattern);
    }


    ///
    ///  path で指定されたファイルのファイル名（および拡張子）にマッチするビヘイビアを取得します。
    ///
    ///  @param[in] path   ビヘイビアを取得したいファイルのパス。
    ///
    ///  @return  path で指定されたファイルのためのビヘイビア。
    ///
    BehaviorSetting BehaviorSettingsCollection::GetBehaviorSettingFor(const std::wstring& path)
    {
        std::wstring targetFileName = make_string_to_lower(Path::GetFileName(path));

        // マッチするパターンを見つける
        FOREACH(BehaviorSettingsCollection::iterator, it, m_allBehaviorSettings)
        {
            std::wstring currPattern = make_string_to_lower(it->pattern);

            if (!wildcmp(currPattern.c_str(), targetFileName.c_str()))
            {
                continue;
            }

            // 単一のビヘイビアだったら
            if (it->behaviorSettings.size() == 1)
            {
                return it->behaviorSettings.front();
            }

            // 複数のビヘイビアがあったら
            else
            {
                return offerChoice(it->behaviorSettings);
            }
        }

        // マッチするパターンが見つからなければここにくる。
        return BehaviorSetting(L"", L"%F", false);
    }


    ///
    ///  behaviros で指定された複数の [ファイル名別動作] を、ポップアップメニューを使用してユーザに提示し、
    ///  どの動作を行うのか選択してもらう。
    ///
    ///  @param[in] behaviors ポップアップメニューによってユーザに提示したい [ファイル名別動作] のコレクション
    ///
    ///  @return    選択された [ファイル名別動作] を表す Behavior クラスのインスタンスへのポインタ。
    ///             キャンセルされたら、デフォルトコンストラクタで構築した Behavior のインスタンスを返します。
    ///
    const BehaviorSetting BehaviorSettingsCollection::offerChoice(const std::vector<BehaviorSetting>& behaviors)
    {
        // ポップアップメニューを作る
        PopupMenu pm(DummyWindow::GetHandle());

        int cmd_id = 1; // ポップアップメニューの項目が選択されたときに返ってくる値の初期値。初期値は 0 より大きい値にしないといけない
        std::vector<BehaviorSetting>::const_iterator it;
        for (it = behaviors.begin(); it != behaviors.end(); it++)
        {
            if (it->auto_exec)
            {
                pm.AppendCheckedItem(cmd_id, it->command_line);
            }
            else
            {
                pm.AppendItem(cmd_id, it->command_line);
            }
            cmd_id++;
        }

        // ポップアップメニューを表示して選択してもらう
        int command = pm.Show();

        // 選択されたら
        if (command != 0)
        {
            return behaviors.at(command-1);
        } 
        // 選択されなかったか、エラーだったら
        else
        {
            return BehaviorSetting();
        }
    }
}
