﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "SendToPowerShell.h"

// private headers
#include "Console.h"
#include "EffectiveBehavior.h"
#include "MessageBox.h"
#include "Path.h"
#include "PrevilegeChecker.h"
#include "Process.h"
#include "SettingRepository.h"
#include "ShellApi.h"
#include "Shortcut.h"
#include "SystemInfo.h"


namespace bearmini
{
    ///
    ///  「PowerShell に送る」のプログラムを実行します。
    ///  
    ///  @param args コマンドラインで指定された引数
    ///
    void SendToPowerShell::Run(const wstring_vector& args)
    {
        validateArguments(args);
        
        const std::wstring powerShellPath = getFullPathOfPowerShellExe();

        FOREACH (wstring_vector::const_iterator, path, args)
        {
            std::wstring unquotedPath = Path::Unquote(*path);
            std::wstring workDir = getWorkDirectoryFor(unquotedPath);
            
            Process process(powerShellPath, workDir);
            process.Run();
            process.WaitForConsoleAllocated();
            
            Console console(process.GetId());
            console.SetTitle(getConsoleTitle(PrevilegeChecker::IsAdminProcess(process.GetHandle())));

            if (!Path::IsDirectory(unquotedPath))
            {
                ConsoleInputArray inputs;
                arrangeConsoleInputArray(unquotedPath, &inputs);
                console.Input(inputs);
            }
        }

    }


    ///
    ///  プログラムのセットアップ（インストールもしくはアンインストール）を行います。
    ///
    void SendToPowerShell::Setup()
    {
        int result = MessageBox::Show(this, L"インストールしますか？", MessageBoxType_YesNo);
        if (result == DialogResult_Yes)
        {
            install();
            return;
        }

        result = MessageBox::Show(this, L"それではアンインストールしますか？", MessageBoxType_YesNo);
        if (result == DialogResult_Yes)
        {
            uninstall();
            return;
        }
    }


    ///
    ///  コマンドラインで渡されてきた引数が PowerShell の実行にふさわしいものかどうか、
    ///  検証します。
    ///  実行にふさわしいかどうかの検証とは、ファイル数が多すぎないか、とか
    ///  PowerShell で取り扱うことのできないネットワーク上のファイルでないか、
    ///  といったような点を調べます。
    ///  実行にふさわしくないと判断された場合、例外を送出します。
    ///
    ///  @param[in] args コマンドラインで渡されてきた引数
    ///
    void SendToPowerShell::validateArguments(const wstring_vector& args)
    {
        static const unsigned int MAX_SIMULTANEOUS_EXEC_COUNT = SettingRepository::GetMaxSimultaneous();

        if (args.size() > MAX_SIMULTANEOUS_EXEC_COUNT)
        {
            throw std::domain_error("Too many files. \r\n\r\nIn order to change the number of PowerShell can be started, please modify the value of MAX_SIMULTANEOUS in SendToPS.ini file.");
        }

    }


    ///
    ///  PowerShell.Exe のパスを取得します。
    ///
    std::wstring SendToPowerShell::getFullPathOfPowerShellExe()
    {
        // http://support.microsoft.com/kb/928439 によると、
        // Windows Vista では PowerShell は以下のディレクトリにインストールされる：
        //
        //      32 ビット版の Windows では、Windows PowerShell 1.0 は次のフォルダにインストールされます。 
        //              %windir%\System32\WindowsPowerShell\V1.0 
        //
        //      64 ビット版の Windows では、Windows PowerShell 1.0 は、次のフォルダにインストールされます。 
        //              %windir%\Syswow64\WindowsPowerShell\V1.0 
        //
        // http://support.microsoft.com/?kbid=926140 によると、
        // Windows XP SP2 および Windows Server 2003 では PowerSehll は以下のディレクトリにインストールされる：
        //
        //      x86 ベースの Windows では、Windows PowerShell 1.0 は以下のフォルダにインストールされます。 
        //              %windir%\System32\WindowsPowerShell\V1.0 
        //
        //      x64 ベースおよび Itanium ベースの Windows では、の以下の場所にインストールされます。
        //          64 ビット版の Windows PowerShell は、次のフォルダにインストールされます。 
        //              %windir%\system32\WindowsPowerShell\V1.0  
        //
        //          32 ビット版の Windows PowerShell は、次のフォルダにインストールされます。 
        //              %windir%\Syswow64\WindowsPowerShell\V1.0  
        //

        std::wstring system32Dir = SystemInfo::GetSystemDir();

        if (SystemInfo::IsPlatform32bit())
        {
            std::wstringstream wss;
            wss << system32Dir << L"\\WindowsPowerShell\\V1.0\\PowerShell.exe";
            return wss.str();
        }
        else
        {
            // if (64bit OS 上であえて 32bit 版の PowerShell を実行するなら)
            // {
            //      未実装
            // }
            // else
            // {
                    std::wstringstream wss;
                    wss << system32Dir << L"\\WindowsPowerShell\\V1.0\\PowerShell.exe";
                    return wss.str();
            // }
        }
    }


    ///
    ///  path で指定されたパスのファイルまたはディレクトリをコマンドプロンプトに送る際の
    ///  作業ディレクトリとなるディレクトリを取得します。
    ///
    std::wstring SendToPowerShell::getWorkDirectoryFor(const std::wstring& path)
    {
        if (Path::IsDirectory(path))
        {
            return path;
        }
        else
        {
            return Path::GetDirectory(path);
        }
    }


    ///
    ///  コンソールのタイトルに表示する文字列を取得します。
    ///
    ///  @return コンソールのタイトルに表示するための文字列
    ///
    std::wstring SendToPowerShell::getConsoleTitle(bool isAdminProcess)
    {
        if (isAdminProcess)
        {
            return L"管理者: Windows PowerShell";
        }
        else
        {
            return L"Windows PowerShell";
        }
    }


    ///
    ///  path で指定されたファイルのパスに応じて、プロンプトに送り込むべきキー入力の配列を取得します。
    ///
    ///  @param[in]     path                 ユーザから指定されたファイルのパス
    ///  @param[in/out] pConsoleInputArray   キー入力の配列を受け取るための ConsoleInputArray クラスのインスタンスへのポインタ
    ///
    void SendToPowerShell::arrangeConsoleInputArray(const std::wstring& path, ConsoleInputArray* pConsoleInputArray)
    {
        BehaviorSettingsCollection behaviorSettings = SettingRepository::GetBehaviorSettings();
        BehaviorSetting bs = behaviorSettings.GetBehaviorSettingFor(path);
        EffectiveBehavior eb = EffectiveBehavior::CreateFrom(bs, path);

        pConsoleInputArray->Append(eb.CommandLine());

        // いったんコマンドラインの先頭にカーソルを移動してから...
        pConsoleInputArray->AppendVirtualKeyCode(VK_LEFT, static_cast<unsigned int>(eb.CommandLine().length()));

        // カーソルを指定の位置に移動
        pConsoleInputArray->AppendVirtualKeyCode(VK_RIGHT, eb.CursorPosition());

        // 自動実行の場合は ENTER を入力する
        if (bs.auto_exec)
        {
            //pConsoleInputArray->AppendVirtualKeyCode(VK_RETURN);  <- なぜかうまくいかない
            pConsoleInputArray->Append(L"\r\n");
        }
    }


    ///
    ///  インストールを行います。
    ///
    void SendToPowerShell::install()
    {
        // ショートカットを作成（通常実行）
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"PowerShell に送る.lnk");
            const std::wstring sendToPowerShellExePath = getSendToPowerShellExeFullPath();
            const std::wstring sendToPowerShellExeDir = getSendToPowerShellExeDirectory();
            const std::wstring description = L"ファイルやディレクトリを指定して PowerShell を起動します.";

            try
            {
                Shortcut::Create(linkFilePath, sendToPowerShellExePath, description, L"", sendToPowerShellExeDir, L"");
            }
            catch (std::domain_error&)
            {
                throw std::domain_error("ショートカットの作成に失敗しました.");
            }

        }

        // ショートカットを作成（管理者として実行）
        if (SystemInfo::GetOsMajorVersion() >= SystemInfo::OsMajorVersion_Vista)
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"PowerShell に送る（管理者として実行...）.lnk");
            const std::wstring sendToPowerShellExePath = getSendToPowerShellExeFullPath();
            const std::wstring sendToPowerShellExeDir = getSendToPowerShellExeDirectory();
            const std::wstring description = L"ファイルやディレクトリを指定して PowerShell を起動します.";
            
            try
            {
                Shortcut::Create(linkFilePath, sendToPowerShellExePath, description, L"", sendToPowerShellExeDir, L"", 0, 1, true);
            }
            catch (std::domain_error&)
            {
                throw std::domain_error("ショートカットの作成に失敗しました.");
            }

        }

        MessageBox::Show(this, L"インストールが完了しました.");
    }


    ///
    ///  アンインストールを行います。
    ///
    void SendToPowerShell::uninstall()
    {
        // ショートカットファイルを削除
        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"PowerShell に送る.lnk");
            ::DeleteFileW(linkFilePath.c_str());
        }

        {
            const std::wstring linkFilePath = Path::Combine(ShellApi::GetSendToFolderPath(), L"PowerShell に送る（管理者として実行...）.lnk");
            ::DeleteFileW(linkFilePath.c_str());
        }

        MessageBox::Show(this, L"アンインストールが完了しました.\r\nご利用ありがとうございました.");
    }


    ///
    ///  このプログラム（SendToPS.exe）のフルパスを取得します。
    ///
    ///  @return    このプログラムのフルパス
    ///
    std::wstring SendToPowerShell::getSendToPowerShellExeFullPath()
    {
        wchar_t lpszModuleName[_MAX_PATH];
        ::GetModuleFileNameW(0, lpszModuleName, _MAX_PATH);
        return lpszModuleName;
    }


    ///
    ///  このプログラム（SendToPS.exe）がインストールされているディレクトリを取得します。
    ///
    ///  @return    このプログラムがインストールされているディレクトリ
    ///
    std::wstring SendToPowerShell::getSendToPowerShellExeDirectory()
    {
        return Path::GetDirectory(getSendToPowerShellExeFullPath());
    }

}
