﻿/*
  Copyright 2007 Takashi Oguma

  This file is part of SendToCMD.

  SendToCMD is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  SendToCMD is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with SendToCMD; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/

#include "SettingRepository.h"

#include "Profile.h"


namespace bearmini 
{
    /// CMD.EXE に渡すオプション
    std::wstring SettingRepository::m_strCmdExeOption;

    /// Cygwin のルートディレクトリ
    std::wstring SettingRepository::m_strCygwinRootDir;

    /// Cygwin を起動するコマンド
    std::wstring SettingRepository::m_strCygwinLauncher;

    /// Cygwin でカレントディレクトリを変更するコマンド
    std::wstring SettingRepository::m_strCygwinChangeDirCommand;

    /// 最大同時起動数
    unsigned int SettingRepository::m_nMaxSimultaneous;

    /// ファイル名別動作一覧
    BehaviorSettingsCollection SettingRepository::m_allBehaviorSettings;

    ///
    ///  設定内容を INI ファイルからロードします。
    ///
    void SettingRepository::Load()
    {
        // CMD.EXE に渡すオプションを取得
        m_strCmdExeOption = Profile::GetString(L"CMD.EXEに対するオプション", L"option", L"");

        // Cygwin を起動するコマンドを取得
        m_strCygwinLauncher = Profile::GetString(L"Cygwin関連", L"cygwin_launcher", L"");

        // Cygwin でカレントディレクトリを変更するコマンド
        m_strCygwinChangeDirCommand = Profile::GetString(L"Cygwin関連", L"change_dir_command", L"cd");

        // [ファイル名別動作] をロードする
        loadBehaviorSettings();

        // 最大同時起動数を取得
        const unsigned int default_max_simultaneous = 32;
        m_nMaxSimultaneous = Profile::GetUInt(L"その他", L"MAX_SIMULTANEOUS", default_max_simultaneous);
    }


    ///
    ///  [ファイル名別動作] をすべてロードします。
    ///
    void SettingRepository::loadBehaviorSettings()
    {
        wstring_vector matchingPatternList = Profile::GetKeyNameList(L"ファイル名別動作");
        wstring_vector::iterator pattern;
        for (pattern = matchingPatternList.begin(); pattern != matchingPatternList.end(); pattern++)
        {
            m_allBehaviorSettings.Add(getBehaviorSettingsForPattern(*pattern));
        }
    }


    ///
    ///  pattern で指定されたファイル名パターンに対応するビヘイビア設定のリストを取得します。
    ///
    BehaviorSettingsForPattern SettingRepository::getBehaviorSettingsForPattern(const std::wstring& pattern)
    {
        //  [ファイル名別動作] セクションから指定されたパターンをキーとして値（右辺値）をロードする。
        //  右辺値がセクション参照になっていたら、参照先のセクションから複数の動作候補を取得する。
        //  右辺値がセクション参照でなければ、その右辺値を単一の動作候補とする。

        BehaviorSettingsForPattern result;

        // パターンを元に右辺値をロード
        std::wstring rightHand = Profile::GetString(L"ファイル名別動作", pattern.c_str(), L"");

        // 右辺値がセクション参照なら
        if (isSectionReference(rightHand))
        {
            // 参照先のセクションからキーの一覧を取得する
            std::wstring sectionName = unbracket(rightHand);
            wstring_vector keyList = Profile::GetKeyNameList(sectionName);

            // 参照先のセクションから、動作一覧をロードする
            result.pattern = pattern;
            wstring_vector::iterator key;
            for (key = keyList.begin(); key != keyList.end(); key++)
            {
                result.behaviorSettings.push_back(getOneBehaviorSetting(sectionName, *key));
            }

        }

        // 右辺値がセクション参照でなければそのまま
        else
        {
            if (pattern.at(0) == L'!')
            {
                result.pattern = pattern.substr(1);
            }
            else
            {
                result.pattern = pattern;
            }

            BehaviorSetting b;
            b.caption = L""; // 使用されないので不要
            b.command_line = rightHand;
            if (pattern.at(0) == L'!')
            {
                b.auto_exec = true;
            }
            else
            {
                b.auto_exec = false;
            }
            result.behaviorSettings.push_back(b);
        }

        return result;
    }


    ///
    ///  指定された文字列が "セクション参照" かどうかを調べます。
    ///  [ で始まって ] で終わる文字列を "セクション参照" とみなします。
    ///
    ///  @param[in] str "セクション参照" かどうかを調べたい文字列
    ///
    ///  @return    指定された文字列が"セクション参照"の場合は true、そうでなければ false
    ///
    bool SettingRepository::isSectionReference(const std::wstring& str)
    {
        return (str.at(0) == L'[') && (str.at(str.length()-1) == L']');
    }


    ///
    ///  指定された文字列が [ と ] で囲まれていたらその [ と ] を取り除いたものを返します。
    ///  指定された文字列が [ と ] で囲まれていなかったらそのままの文字列のコピーを返します。
    ///
    ///  @param[in] str [ と ] を取り除きたい文字列
    ///
    ///  @return    [ と ] が取り除かれた文字列
    ///
    std::wstring SettingRepository::unbracket(const std::wstring& str)
    {
        if ((str.at(0) == L'[') && (str.at(str.length()-1) == L']'))
        {
            return str.substr(1, str.length()-2);
        }
        return str;
    }


    ///
    ///  sectionName で指定されたセクションを参照し、そのセクションから key で指定されたビヘイビアを取得します。
    ///
    ///  @param[in] sectionName 参照するセクション名
    ///  @param[in] key         ビヘイビアのキー（左辺値）
    ///  
    ///  @return 取得したビヘイビア
    ///
    BehaviorSetting SettingRepository::getOneBehaviorSetting(const std::wstring& sectionName, const std::wstring& key)
    {
        BehaviorSetting b;
        if (key.at(0) == L'!')
        {
            b.caption = key.substr(1);
            b.auto_exec = true;
        }
        else
        {
            b.caption = key;
            b.auto_exec = false;
        }
        b.command_line = Profile::GetString(sectionName, key, L"");

        return b;
    }

}
