/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService;

import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;

import jdbcacsess2.main.Jdbcacsess2;

/**
 * java.sql.ResultSetMetaData の主要項目を、取り扱いやすいようにインスタンス変数で管理します。
 * 
 * @author sima
 * 
 */
public class ColumnAttributeResult {

	/**
	 * ResultSetMetaData の各項目を取り出し、List化して取得します。
	 * 
	 * @param rsmd
	 *            検索結果のResultSetMetaData
	 * @return　主要項目のColumnAttributeResult(リスト
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 */
	static public List<ColumnAttributeResult> convColumnAttributeResult(ResultSetMetaData rsmd) throws SQLException,
	ClassNotFoundException {

		int cnt = rsmd.getColumnCount();

		List<ColumnAttributeResult> columnAttributeResultList = new ArrayList<ColumnAttributeResult>(cnt);

		for (int i = 1; i <= cnt; i++) {
			columnAttributeResultList.add(new ColumnAttributeResult(rsmd, i));
		}

		return columnAttributeResultList;
	}

	/**
	 * ResultSetMetaData の１カラム分の属性を管理する。
	 * 
	 * @param rsmd
	 * @param column
	 * @throws SQLException
	 * @throws ClassNotFoundException
	 */
	public ColumnAttributeResult(ResultSetMetaData rsmd, int column) throws SQLException {

		columnClassName = rsmd.getColumnClassName(column);

		columnDisplaySize = rsmd.getColumnDisplaySize(column);

		columnLabel = rsmd.getColumnLabel(column);

		columnName = rsmd.getColumnName(column);

		columnType = ConstSqlTypes.searchTypes(rsmd.getColumnType(column));

		columnTypeName = rsmd.getColumnTypeName(column);

		precision = rsmd.getPrecision(column);

		scale = rsmd.getScale(column);

		columnTypeName = rsmd.getColumnTypeName(column);

		catalogName = rsmd.getCatalogName(column);
		schemaName = rsmd.getSchemaName(column);
		tableName = rsmd.getTableName(column);

		autoIncrement = rsmd.isAutoIncrement(column);
		caseSensitive = rsmd.isCaseSensitive(column);
		currency = rsmd.isCurrency(column);
		definitelyWritable = rsmd.isDefinitelyWritable(column);
		nullable = rsmd.isNullable(column);
		readOnly = rsmd.isReadOnly(column);
		searchable = rsmd.isSearchable(column);
		writable = rsmd.isWritable(column);

		try {
			columnClass = Class.forName(columnClassName, true, rsmd.getClass().getClassLoader());
		} catch (Exception e) {
			// TODO ここで、いろいろなデータ型に対応するCLASSを決定する
			Jdbcacsess2.logger.log(Level.SEVERE,
			                       "未対応クラス:" + columnClassName + " " + columnName + "は、Objectで処理続行します",
			                       e);
			columnClass = Object.class;
		}

	}

	private Class<?> columnClass;
	private final String columnClassName;
	private final int columnDisplaySize;
	private final String columnLabel;
	private final String columnName;
	private final ConstSqlTypes columnType;
	private String columnTypeName;
	private final int precision;
	private final int scale;
	private boolean signed;

	private final String catalogName;
	private final String schemaName;
	private final String tableName;

	private final boolean autoIncrement;
	private final boolean caseSensitive;
	private final boolean currency;
	private final boolean definitelyWritable;
	private final int nullable;
	private final boolean readOnly;
	private final boolean searchable;
	private final boolean writable;
	/**
	 * Java クラスの完全指定されたクラスを返します。JDBCのgetColumnClassNameをclass化したものです。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnClassName(int)
	 * @return
	 */
	public Class<?> getColumnClass() {
		return columnClass;
	}

	/**
	 * Java クラスの完全指定された名前を返します。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnClassName(int)
	 * @return columnClassName
	 */
	public String getColumnClassName() {
		return columnClassName;
	}

	/**
	 * 指定された列の通常の最大幅を文字数で示します。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnDisplaySize(int)
	 * @return columnDisplaySize
	 */
	public int getColumnDisplaySize() {
		return columnDisplaySize;
	}

	/**
	 * 印刷や表示に使用する、指定された列の推奨タイトルを取得します。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnLabel(int)
	 * @return columnLabel
	 */
	public String getColumnLabel() {
		return columnLabel;
	}

	/**
	 * 指定された列の名前を取得します。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnName(int)
	 * @return columnName
	 */
	public String getColumnName() {
		return columnName;
	}

	/**
	 * 指定された列の SQL 型を取得します。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnType(int)
	 * @return ConstSqlTypes
	 */
	public ConstSqlTypes getColumnType() {
		return columnType;
	}

	/**
	 * 指定された列のデータベース固有の型名を取得します。
	 * 
	 * @see java.sql.ResultSetMetaData#getColumnTypeName(int)
	 * @return columnTypeName
	 */
	public String getColumnTypeName() {
		return columnTypeName;
	}

	/**
	 * 指定された列の 10 進桁数を取得します。
	 * 
	 * @see java.sql.ResultSetMetaData#getPrecision(int)
	 * @return precision
	 */
	public int getPrecision() {
		return precision;
	}

	/**
	 * 指定された列の小数点以下の桁数を取得します。
	 * 
	 * @see java.sql.ResultSetMetaData#getScale()
	 * @return scale
	 */
	public int getScale() {
		return scale;
	}

	/**
	 * 指定された列の値が符号付き数値かどうかを示します。
	 * 
	 * @see java.sql.ResultSetMetaData#isSigned(int)
	 * @return signed
	 */
	public boolean isSigned() {
		return signed;
	}

	/**
	 * @return catalogName
	 */
	public String getCatalogName() {
		return catalogName;
	}

	/**
	 * @return schemaName
	 */
	public String getSchemaName() {
		return schemaName;
	}

	/**
	 * @return tableName
	 */
	public String getTableName() {
		return tableName;
	}

	/**
	 * @return autoIncrement
	 */
	public boolean isAutoIncrement() {
		return autoIncrement;
	}

	/**
	 * @return caseSensitive
	 */
	public boolean isCaseSensitive() {
		return caseSensitive;
	}

	/**
	 * @return currency
	 */
	public boolean isCurrency() {
		return currency;
	}

	/**
	 * @return definitelyWritable
	 */
	public boolean isDefinitelyWritable() {
		return definitelyWritable;
	}

	/**
	 * @return nullable
	 */
	public int getNullable() {
		return nullable;
	}

	/**
	 * @return readOnly
	 */
	public boolean isReadOnly() {
		return readOnly;
	}

	/**
	 * @return searchable
	 */
	public boolean isSearchable() {
		return searchable;
	}

	/**
	 * @return writable
	 */
	public boolean isWritable() {
		return writable;
	}
}
