/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.sqlService.dbobject;

import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.Icon;

import jdbcacsess2.main.Jdbcacsess2;
import jdbcacsess2.sqlService.ReferenceColumnResult;
import jdbcacsess2.sqlService.dbobject.DBObjectTables.DBObjectTable;
import jdbcacsess2.sqlService.dbobject.icon.DbObjectIconImage;

/**
 * TABLECOLUMNの一覧
 * 
 * @author sima
 * 
 */
public class DBObjectTableColumns extends DBObjects {

	private final ArrayList<DBObjectTableColumn> tableColumns = new ArrayList<DBObjectTableColumn>();


	private final boolean storesLowerCaseIdentifiers;
	private final boolean storesUpperCaseQuotedIdentifiers;
	/**
	 * コンストラクタ
	 * 
	 * @param table
	 * @param dmd
	 * @throws SQLException
	 */
	public DBObjectTableColumns(DBObjectTable table, DatabaseMetaData dmd) throws SQLException {

		storesLowerCaseIdentifiers = dmd.storesLowerCaseIdentifiers();
		storesUpperCaseQuotedIdentifiers = dmd.storesUpperCaseQuotedIdentifiers();

		//
		Map<String, Short> pkmap = new HashMap<String, Short>();
		ResultSet pkrs = dmd.getPrimaryKeys(table.tableCatalog, table.tableSchema, table.getTableName());
		while (pkrs.next()) {
			pkmap.put(pkrs.getString(4), pkrs.getShort(5));
		}
		pkrs.close();

		// 外部参照キー情報を取得する
		Map<String, ReferenceColumnResult> fkmap = new HashMap<String, ReferenceColumnResult>();
		ResultSet fkrs = dmd.getImportedKeys(table.tableCatalog, table.tableSchema, table.getTableName());
		List<ReferenceColumnResult> list = ReferenceColumnResult.convReferenceColumnResult(fkrs);
		for (ReferenceColumnResult rcr : list) {
			fkmap.put(rcr.getFkcolumnName(), rcr);
		}
		fkrs.close();

		//
		ResultSet rs = dmd.getColumns(table.tableCatalog, table.tableSchema, table.getTableName(), null);
		while (rs.next()) {
			tableColumns.add(new DBObjectTableColumn(rs, pkmap, fkmap));
		}
		rs.close();

	}

	@Override
	public ArrayList<? extends DBObject> getDBObjects() {
		return tableColumns;
	}

	@Override
	public String toString() {
		return "TABLE_COLUMNS";
	}

	/**
	 * TABLECOLUMNのプロパティ
	 * 
	 * @author sima
	 * 
	 */
	public class DBObjectTableColumn extends DBObject {

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１要素
		 */
		public String tableCatalog;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第２要素
		 */
		public String tableSchema;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第３要素
		 */
		public String tableName = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第４要素
		 */
		public String columnName = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第５要素
		 */
		public int dataType;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第６要素
		 */
		public String typeName = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第７要素
		 */
		public int columnSize;

		// - BUFFER_LENGTH;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第９要素
		 */
		public int decimalDigits;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１０要素
		 */
		public int numPrecRadix;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１１要素
		 */
		public int nullable;

		private String getNullableName() {
			switch (nullable) {
			case DatabaseMetaData.columnNoNulls:
				return "NoNulls";
			case DatabaseMetaData.columnNullable:
				return "Nullable";
			case DatabaseMetaData.columnNullableUnknown:
				return "NullableUnknown";
			}
			return "?";
		}

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１２要素
		 */
		public String remarks;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１３要素
		 */
		public String columnDef;



		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１６要素
		 */
		public int charOctetLength;


		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１７要素
		 */
		public int ordinalPosition;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１８要素
		 */
		public String isNullable = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第１９要素
		 */
		public String scopeCatalog = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第２０要素
		 */
		public String scopeSchema = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第２１要素
		 * 
		 * @return
		 */
		public String scopeTable = "";

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第２２要素
		 */
		public short sourceDataType;

		/**
		 * {@link DatabaseMetaData#getColumns(String, String, String, String)}
		 * の第２３要素
		 */
		public String isAutoincrement = "";

		/**
		 * {@link DatabaseMetaData#getPrimaryKeys(String, String, String)}
		 * から取得した主キー定義有無
		 * 
		 * @return
		 */
		public boolean isPrimaryKey() {
			return (primaryKeySeqNo != null);
		}

		/**
		 * {@link DatabaseMetaData#getImportedKeys(String, String, String)}
		 * から取得した外部参照キー
		 * 
		 * @return
		 */
		public String getForginKey() {
			if (forginKey != null) {
				return forginKey.getPktableName() + "." + forginKey.getPkcolumnName();
			}
			return "";
		}

		private final StringBuilder attr;

		private final Short primaryKeySeqNo;

		private final ReferenceColumnResult forginKey;

		private final List<Property> properties = new ArrayList<Property>();

		@Override
		public List<Property> getProperties() {
			return properties;
		}
		/**
		 * コンストラクタ
		 * 
		 * @param rs
		 * @param pkmap
		 * @throws SQLException
		 */
		public DBObjectTableColumn(ResultSet rs, Map<String, Short> pkmap, Map<String, ReferenceColumnResult> fkmap)
				throws SQLException {
			int colCnt = rs.getMetaData().getColumnCount();
			int i = 0;
			try {
				if (colCnt > i) {
					tableCatalog = rs.getString(++i);
					properties.add(new Property("TABLE_CAT", tableCatalog));
				}
				if (colCnt > i) {
					tableSchema = rs.getString(++i);
					properties.add(new Property("TABLE_SCHEM", tableSchema));
				}
				if (colCnt > i) {
					tableName = rs.getString(++i);
					properties.add(new Property("TABLE_NAME", tableName));
				}
				if (colCnt > i) {
					columnName = rs.getString(++i);
					properties.add(new Property("COLUMN_NAME", columnName));
				}

				if (colCnt > i) {
					dataType = rs.getInt(++i);
					properties.add(new Property("DATA_TYPE", Integer.toString(dataType)));
				}
				if (colCnt > i) {
					typeName = rs.getString(++i);
					properties.add(new Property("TYPE_NAME", typeName));
				}
				if (colCnt > i) {
					columnSize = rs.getInt(++i);
					properties.add(new Property("COLUMN_SIZE", Integer.toString(columnSize)));
				}
				if (colCnt > i) {
					++i;
					/* BUFFER_LENGTH */

				}
				if (colCnt > i) {
					decimalDigits = rs.getInt(++i);
					properties.add(new Property("DECIMAL_DIGITS", Integer.toString(decimalDigits)));
				}
				if (colCnt > i) {
					numPrecRadix = rs.getInt(++i);
					properties.add(new Property("NUM_PREC_RADIX", Integer.toString(numPrecRadix)));
				}
				if (colCnt > i) {
					nullable = rs.getInt(++i);
					properties.add(new Property("NULLABLE", getNullableName()));
				}

				if (colCnt > i) {
					remarks = rs.getString(++i);
					properties.add(new Property("REMARKS", remarks));
				}
				if (colCnt > i) {
					columnDef = rs.getString(++i);
					properties.add(new Property("COLUMN_DEF", columnDef));
				}
				if (colCnt > i) {
					++i;
					/* sqlDataType */
					/*
					 * properties.add(new
					 * Property("SQL_DATA_TYPE",Integer.toString(sqlDataType)));
					 */
				}
				if (colCnt > i) {
					++i;
					/* sqlDatetimeSub */
					/*
					 * properties.add(new
					 * Property("SQL_DATETIME_SUB",Integer.toString
					 * (sqlDatetimeSub)));
					 */
				}

				if (colCnt > i) {
					charOctetLength = rs.getInt(++i);
					properties.add(new Property("CHAR_OCTET_LENGTH", Integer.toString(charOctetLength)));
				}
				if (colCnt > i) {
					ordinalPosition = rs.getInt(++i);
					properties.add(new Property("ORDINAL_POSITION", Integer.toString(ordinalPosition)));
				}

				if (colCnt > i) {
					isNullable = rs.getString(++i);
					properties.add(new Property("IS_NULLABLE", isNullable));
				}
				if (colCnt > i) {
					scopeCatalog = rs.getString(++i);
					properties.add(new Property("SCOPE_CATLOG", scopeCatalog));
				}
				if (colCnt > i) {
					scopeSchema = rs.getString(++i);
					properties.add(new Property("SCOPE_SCHEMA", scopeSchema));
				}
				if (colCnt > i) {
					scopeTable = rs.getString(++i);
					properties.add(new Property("SCOPE_TABLE", scopeTable));
				}

				if (colCnt > i) {
					sourceDataType = rs.getShort(++i);
					properties.add(new Property("SOURCE_DATA_TYPE", Short.toString(sourceDataType)));
				}
				if (colCnt > i) {
					isAutoincrement = rs.getString(++i);
					properties.add(new Property("IS_AUTOINCREMENT", isAutoincrement));
				}
			} catch (SQLException e) {
				Jdbcacsess2.logger.warning(e.toString());
			}

			// カラム名でプライマリキーのカラム位置番号リストを検索。
			primaryKeySeqNo = pkmap.get(columnName);
			properties.add(new Property("isPrimaryKey", Boolean.toString(isPrimaryKey())));

			// カラム名で外部参照キーのカラムリストを検索。従カラムを取得する。
			forginKey = fkmap.get(columnName);
			properties.add(new Property("forginKey", getForginKey()));

			//
			attr = editAttrString(dataType, columnSize, decimalDigits);

		}

		/*
		 * (非 Javadoc)
		 * 
		 * @seejdbcacsess2.sqlService.dbobject.DBObject#getChildren(java.sql.
		 * DatabaseMetaData)
		 */
		@Override
		public DBObjects getChildren(DatabaseMetaData dmd) throws SQLException {
			return null;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSchemaTableName()
		 */
		@Override
		public SchemaTableName getSchemaTableName() throws SQLException {
			return new SchemaTableName(storesLowerCaseIdentifiers,
			                           storesUpperCaseQuotedIdentifiers,
			                           tableSchema,
			                           tableName);
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getName()
		 */
		@Override
		public String getName() {
			return columnName;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getIconImage()
		 */
		@Override
		public Icon getIconImage() {
			return DbObjectIconImage.TABLECOLUMN.getValue();
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isStrong()
		 */
		@Override
		public boolean isStrong() {
			return isPrimaryKey();
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#isBottom()
		 */
		@Override
		public boolean isBottom() {
			return true;
		}

		/*
		 * (非 Javadoc)
		 * 
		 * @see jdbcacsess2.sqlService.dbobject.DBObject#getSummary()
		 */
		@Override
		public String getSummary() {
			return getName();
		}

		@Override
		public String getProperty1() {
			return attr.toString();
		}

		@Override
		public String getProperty2() {
			return remarks;
		}

		@Override
		public String getProperty3() {
			return getForginKey();
		}

		@Override
		public String getProperty4() {
			return "";
		}

		@Override
		public boolean isPropertyEnable() {
			return true;
		}
	}

}
