package org.apache.webdav.lib.methods;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Method;
import java.net.URL;
import org.apache.commons.httpclient.ChunkedOutputStream;
import org.apache.commons.httpclient.HttpConnection;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpState;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.HttpVersion;
import org.apache.commons.httpclient.util.EncodingUtil;


/**
 * PUT Method.
 *
 *
 * @since 1.0
 */
public abstract class HttpRequestBodyMethodBase extends HttpMethodBase {


    private static Method chunkedOutputStreamFinish;

    // ----------------------------------------------------------- Constructors


    /**
     * Default constructor.
     */
    public HttpRequestBodyMethodBase() {
        super();
    }



    /**
     * URI-setting constructor.
     *
     * @param uri the URI to request. The URI is expected
     *        to be already URL encoded.  It may be either an absolute or
     *        server relative path.
     *
     * @since 1.0
     */
    public HttpRequestBodyMethodBase(String uri) {
        super(uri);
    }


    // ------------------------------------------------------- Instance Methods


    /**
     * Request body content to be sent.
     */
    private byte[] data = null;


    /**
     * Request body content to be sent.
     */
    private File file = null;


    /**
     * Request body content to be sent.
     */
    private URL url = null;


    // --------------------------------------------------------- Public Methods


    /**
     * Set my request body content to the contents of a file.
     *
     * @since 2.0
     */
    public void setRequestBody(File file) throws IOException {
        checkNotUsed();
        this.file = file;
    }

    /**
     * Set my request body content to the resource at the specified URL.
     *
     * @since 2.0
     */
    public void setRequestBody(URL url) throws IOException {
        checkNotUsed();
        this.url = url;
    }

    /**
     * Set my request body content to the contents of a byte array.
     *
     * @since 2.0
     */
    public void setRequestBody(byte[] bodydata) {
        checkNotUsed();
        this.data = bodydata;
    }

    /**
     * Set my request body content to the contents of a string.
     *
     * @since 2.0
     */
    public void setRequestBody(String bodydata) {
        checkNotUsed();
        setRequestBody(EncodingUtil.getBytes(bodydata, getRequestCharSet()));
    }

    /**
     * Set my request body content to the contents of an input stream.
     * The contents will be buffered into
     * memory. To upload large entities, it is recommended to first buffer the
     * data into a temporary file, and then send that file.
     *
     * @since 2.0
     */
    public void setRequestBody(InputStream is)
        throws IOException {

        checkNotUsed();
        byte[] buffer = new byte[4096];
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        int nb = 0;
        while (true) {
            nb = is.read(buffer);
            if (nb == -1) {
                break;
            }
            os.write(buffer, 0, nb);
        }
        data = os.toByteArray();
    }

    /**
     * Returns true if <tt>100 Continue</tt> status code
     * is found.
     *
     * @since 2.0
     */
    public boolean readContinueCode() {
        if (getStatusLine() == null) {
            return false;
        }
        if(null != getRequestHeader("expect") &&
           getStatusLine().getStatusCode() != HttpStatus.SC_CONTINUE) {
            return false;
        }
        return true;
    }

    /**
     * Do write the request body.
     * Override the method of {@link HttpMethodBase}
     * if the method should wait until a <tt>100 Continue</tt> status code
     * is expected (@link readContinueCode)
     *
     * @since 2.0
     */
    protected boolean writeRequestBody(HttpState state, HttpConnection conn)
        throws IOException, HttpException {
        OutputStream out = conn.getRequestOutputStream();
        
        if (getParams().getVersion().equals(HttpVersion.HTTP_1_1) && getRequestContentLength() == -1) {
            out = new ChunkedOutputStream(out);
        }

        InputStream inputStream = null;
        if (file != null && file.exists()) {
            inputStream = new FileInputStream(file);
        } else if (url != null) {
            inputStream = url.openConnection().getInputStream();
        } else if(data != null){
            inputStream = new ByteArrayInputStream(data);
        } else {
            return true;
        }

        byte[] buffer = new byte[4096];
        int nb = 0;
        while (true) {
            nb = inputStream.read(buffer);
            if (nb == -1) {
                break;
            }
            out.write(buffer, 0, nb);
        }

        // httpclient 3.0 support
        if(out instanceof ChunkedOutputStream){
          try{
            if(chunkedOutputStreamFinish == null){
              chunkedOutputStreamFinish = out.getClass().getMethod("finish", null);
            }
            chunkedOutputStreamFinish.invoke(out, null);
          }catch(NoSuchMethodException nsme){
            // must be httpclient 2.0.x
          }catch(Exception e){
            throw new HttpException(e.getClass() + ": " + e.getMessage());
          }
        }
        out.flush();
        return true;
    }

    /**
     * In httpclient 3.0, setting Content-Length was moved to
     * EntityEnclosingMethod class.
     */
    protected void addRequestHeaders(HttpState state, HttpConnection conn)
    throws IOException, HttpException {
        super.addRequestHeaders(state, conn);
        if ((getRequestHeader("content-length") == null)
            && (getRequestHeader("Transfer-Encoding") == null)) {
            long len = getRequestContentLength();
            if (len >= 0) {
                addRequestHeader("Content-Length", String.valueOf(len));
            } else if ((len == -1) && getParams().getVersion().equals(HttpVersion.HTTP_1_1)) {
                addRequestHeader("Transfer-Encoding", "chunked");
            }
        }
    }
    /**
     * Override the method of {@link HttpMethodBase}
     * to return the appropriate content length.
     *
     * @since 2.0
     */
    protected int getRequestContentLength() {
        if(null != data) {
            return data.length;
        } else if(null != file && file.exists()) {
            return (int)(file.length());
        } else if(url != null) {
            return -1;
        } else {
            return 0;
        }
    }


    /**
     * return true, if the method setRequestContent has been called (with a null parameter)
     *
     * @since 2.0
     */
    protected boolean isRequestContentAlreadySet() {
        return (data != null) || (file != null) || (url != null);
    }

    /**
     *
     * @since 1.0
     */
    public void recycle() {
        super.recycle();
        data = null;
        url = null;
        file = null;
    }
}

