using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Xml;

using SystemNeo;
using SystemNeo.IO;
using SystemNeo.Text;

namespace SystemNeo.Drawing.Imaging.Jfif
{
	/// <summary>
	/// 
	/// </summary>
	public sealed class App13SegmentContent : SegmentContent
	{
		#region private static fields
		private const string idValue = "Photoshop 3.0\0";
		private static readonly CollectionFormatter blocksFormatter = new CollectionFormatter() {
			BeginBracket = "[" + Environment.NewLine + CharUtil.Tab,
			EndBracket = Environment.NewLine + "]",
			Separator = "," + Environment.NewLine + CharUtil.Tab
		};
		#endregion

		#region private fields
		private readonly IList<ResourceBlock> blockList = new List<ResourceBlock>();
		#endregion

		// public vpeB //

		/// <summary>
		/// 
		/// </summary>
		public string Id { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public ICollection<ResourceBlock> ResourceBlocks
		{
			get {
				return this.blockList;
			}
		}

		// internal RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		internal App13SegmentContent(BinaryReaderNeo reader)
		{
			this.Id = reader.ReadString(idValue.Length);
			if (this.Id != idValue) {
				string msg = string.Format("APP13IDsłBID={0} Length={1:N0}",
						new QuotedStringFormatter().ToString(this.Id), reader.Length);
				throw new BadImageFormatException(msg);
			}
			do {
				this.blockList.Add(new ResourceBlock(reader));
			} while (reader.Position < reader.Length);
		}

		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			var sw = new StringWriter();
			sw.Write("Id=");
			sw.Write(this.Id.TrimEnd('\0'));
			sw.Write(", Resources=");
			blocksFormatter.WriteTo(sw, this.blockList);
			return sw.ToString();
		}

		// ^ //

		/// <summary>
		/// 
		/// </summary>
		public sealed class JpegQuality
		{
			// public vpeB //

			/// <summary>
			/// 
			/// </summary>
			public bool HuffmanTableOptimized { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public bool Progressive { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public int Quality { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public int ScanCount { get; private set; }

			// internal RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="reader"></param>
			internal JpegQuality(BinaryReaderNeo reader)
			{
				this.Quality = (short)(reader.ReadInt16() + 4);
				this.Progressive = reader.ReadBoolean();
				this.HuffmanTableOptimized = reader.ReadBoolean();
				byte dummy1 = reader.ReadByte();
				this.ScanCount = (byte)(reader.ReadByte() + 2);
				byte dummy2 = reader.ReadByte();
			}

			// public \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				var sw = new StringWriter();
				sw.Write("{Quality=");
				sw.Write(this.Quality);
				sw.Write(", Progressive=");
				sw.Write(this.Progressive.ToString().ToLower());
				sw.Write(", HuffmanTableOptimized=");
				sw.Write(this.HuffmanTableOptimized.ToString().ToLower());
				sw.Write(", ScanCount=");
				sw.Write(this.ScanCount);
				sw.Write("}");
				return sw.ToString();
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public sealed class Resolution
		{
			#region private fields
			private readonly Fraction xResolution;
			private readonly ResolutionUnit xResolutionUnit;
			private readonly Fraction yResolution;
			private readonly ResolutionUnit yResolutionUnit;
			#endregion

			// internal RXgN^ //
			
			/// <summary>
			/// 
			/// </summary>
			/// <param name="reader"></param>
			internal Resolution(BinaryReaderNeo reader)
			{
				this.xResolution = new Fraction(reader.ReadUInt32(), 0x10000);
				this.xResolutionUnit = (ResolutionUnit)reader.ReadUInt16s(2)[0];
				this.yResolution = new Fraction(reader.ReadUInt32(), 0x10000);
				this.yResolutionUnit = (ResolutionUnit)reader.ReadUInt16s(2)[0];
			}

			// public \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				var sw = new StringWriter();
				sw.Write("{X={");
				sw.Write(this.xResolution.Reduce());
				sw.Write(", Unit=");
				sw.Write(this.xResolutionUnit);
				sw.Write("}, Y={");
				sw.Write(this.yResolution.Reduce());
				sw.Write(", Unit=");
				sw.Write(this.yResolutionUnit);
				sw.Write("}}");
				return sw.ToString();
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public enum ResolutionUnit : ushort
		{
			Inch = 1,
			Centimeter = 2
		}

		/// <summary>
		/// 
		/// </summary>
		public sealed class ResourceBlock
		{
			#region private static fields
			private const int typeLength = 4;
			private static readonly string[] types = {"8BIM", "DCSR", "PHUT"};
			private static readonly ValueFormatter valueFormatter = new ValueFormatter();
			#endregion

			#region private fields
			private readonly int valueSize;
			#endregion

			// public vpeB //

			/// <summary>
			/// 
			/// </summary>
			public string Id { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public string Name { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public ResourceTag Tag { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public object Value { get; private set; }

			// static RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			static ResourceBlock()
			{
				valueFormatter.ByteArrayFormatter.MaxCount = 20;
			}

			// internal RXgN^ //
			
			/// <summary>
			/// 
			/// </summary>
			/// <param name="reader"></param>
			internal ResourceBlock(BinaryReaderNeo reader)
			{
				this.Id = reader.ReadString(typeLength);
				if (Array.IndexOf(types, this.Id) < 0) {
					string msg = "IDsłB" + new QuotedStringFormatter().ToString(this.Id);
					throw new BadImageFormatException(msg);
				}
				this.Tag = ResourceTag.GetTag(reader.ReadUInt16());
				byte nameLength = reader.ReadByte();
				if (nameLength > 0) {
					this.Name = reader.ReadString(nameLength);
				}
				if (nameLength % 2 == 0) {
					reader.ReadByte();
				}
				this.valueSize = reader.ReadInt32();
				this.Value = this.Tag.ConvertValue(reader.ReadBytes(this.valueSize));
				reader.Seek(this.valueSize % 2);
			}

			// public \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				var sw = new StringWriter();
				sw.Write("{Id=");
				sw.Write(this.Id);
				sw.Write(", Tag=");
				sw.Write(this.Tag);
				sw.Write(", Name=");
				if (this.Name != null) {
					new QuotedStringFormatter().WriteTo(sw, this.Name);
				}
				sw.Write(", Size=");
				sw.Write(this.valueSize.ToString("N0"));
				sw.Write(", Value=");
				valueFormatter.WriteTo(sw, this.Value);
				sw.Write("}");
				return sw.ToString();
			}

			// ^ //

			/// <summary>
			/// 
			/// </summary>
			private sealed class ValueFormatter : Formatter
			{
				#region private static fields
				private const int stringMaxLength = 256;
				private static readonly CollectionFormatter iptcIimRecordsFormatter
						= new CollectionFormatter() {
							BeginBracket = "[" + Environment.NewLine + "\t\t",
							EndBracket = Environment.NewLine + "\t]",
							Separator = "," + Environment.NewLine + "\t\t"
						};
				#endregion

				#region private fields
				private const int xmlMaxLength = 1024;
				#endregion

				// protected internal \bh //

				/// <summary>
				/// 
				/// </summary>
				/// <param name="writer"></param>
				/// <param name="value"></param>
				protected override void WriteToInternal(TextWriter writer, object value)
				{
					if (value is string) {
						new QuotedStringFormatter(stringMaxLength).WriteTo(writer, value);
					} else if (value is XmlDocument) {
						writer.Write("{");
						WriteXmlDocument(writer, (XmlDocument)value);
						writer.Write("}");
					} else if (value is bool) {
						writer.Write(value.ToString().ToLower());
					} else if (value is IptcIimRecord[]) {
						if (((IptcIimRecord[])value).Length == 0) {
							writer.Write("none");
						} else {
							iptcIimRecordsFormatter.WriteTo(writer, value);
						}
					} else {
						base.WriteToInternal(writer, value);
					}
				}

				// private static \bh //

				/// <summary>
				/// 
				/// </summary>
				/// <param name="writer"></param>
				/// <param name="doc"></param>
				private static void WriteXmlDocument(TextWriter writer, XmlDocument doc)
				{
					var sw = new StringWriter();
					doc.Save(sw);
					string s = sw.ToString();
					if (s.Length > xmlMaxLength) {
						writer.Write(s.Substring(0, xmlMaxLength));
						writer.Write("...");
					} else {
						writer.Write(s);
					}
				}
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public sealed class ResourceTag
		{
			#region private static fields
			private static readonly IDictionary<int, ResourceTag> tagDic;
			#endregion

			#region private fields
			private readonly object conversionParam;
			#endregion

			// public vpeB //

			/// <summary>
			/// 
			/// </summary>
			public ushort Code { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public string Name { get; private set; }

			// static RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			static ResourceTag()
			{
				tagDic = CreateTagDictionary();
			}

			// private RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="code"></param>
			/// <param name="name"></param>
			private ResourceTag(ushort code, string name)
			{
				this.Code = code;
				this.Name = name;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="code"></param>
			/// <param name="name"></param>
			/// <param name="type"></param>
			private ResourceTag(ushort code, string name, Type type) : this(code, name)
			{
				this.conversionParam = type;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="code"></param>
			/// <param name="name"></param>
			/// <param name="encoding"></param>
			private ResourceTag(ushort code, string name, Encoding encoding) : this(code, name)
			{
				this.conversionParam = encoding;
			}

			// public \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				var sw = new StringWriter();
				sw.Write("0x");
				sw.Write(this.Code.ToString("X4"));
				if (this.Name != null) {
					sw.Write("=");
					sw.Write(this.Name);
				}
				return sw.ToString();
			}

			// internal static \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="code"></param>
			/// <returns></returns>
			internal static ResourceTag GetTag(ushort code)
			{
				ResourceTag tag;
				try {
					tag = (ResourceTag)tagDic[code];
				} catch (KeyNotFoundException) {
					tag = new ResourceTag(code, null);
					tagDic[code] = tag;
				}
				return tag;
			}

			// internal \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="bytes"></param>
			/// <returns></returns>
			internal object ConvertValue(byte[] bytes)
			{
				if (this.conversionParam is Encoding) {
					return ((Encoding)this.conversionParam).GetString(bytes);
				}
				if (this.conversionParam is Type) {
					var reader = new BinaryReaderNeo(bytes, ByteOrder.BigEndian, Encoding.ASCII);
					return ConvertValueByType(reader, (Type)this.conversionParam);
				}
				return bytes;
			}

			// private static \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="reader"></param>
			/// <param name="type"></param>
			/// <returns></returns>
			private static object ConvertValueByType(BinaryReaderNeo reader, Type type)
			{
				try {
					if (type == typeof(bool)) {
						return reader.ReadBoolean();
					}
					if (type == typeof(uint)) {
						return reader.ReadUInt32();
					}
					if (type == typeof(IptcIimRecord[])) {
						return IptcIimRecord.ReadRecords(reader.BaseStream);
					}
					if (type == typeof(JpegQuality)) {
						return new JpegQuality(reader);
					}
					if (type == typeof(Resolution)) {
						return new Resolution(reader);
					}
					if (type == typeof(VersionInfo)) {
						return new VersionInfo(reader);
					}
				} catch (EndOfStreamException) {
					reader.Position = 0;
				}
				return reader.ReadToEnd();
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			private static IDictionary<int, ResourceTag> CreateTagDictionary()
			{
				var dic = new Dictionary<int, ResourceTag>();
				dic[0x03E8] = new ResourceTag(0x03E8, "Photoshop2Info");
				dic[0x03E9] = new ResourceTag(0x03E9, "MacintoshPrintInfo");
				dic[0x03EA] = new ResourceTag(0x03EA, "XMLData", Encoding.UTF8);
				dic[0x03EB] = new ResourceTag(0x03EB, "Photoshop2ColorTable");
				dic[0x03ED] = new ResourceTag(0x03ED, "ResolutionInfo", typeof(Resolution));
				dic[0x03EE] = new ResourceTag(0x03EE, "AlphaChannelsName");
				dic[0x03EF] = new ResourceTag(0x03EF, "DisplayInfo");
				dic[0x03F0] = new ResourceTag(0x03F0, "PStringCaption");
				dic[0x03F1] = new ResourceTag(0x03F1, "BorderInformation");
				dic[0x03F2] = new ResourceTag(0x03F2, "BackgroundColor");
				dic[0x03F3] = new ResourceTag(0x03F3, "PrintFlags");
				dic[0x03F4] = new ResourceTag(0x03F4, "BW_HalftoningInfo");
				dic[0x03F5] = new ResourceTag(0x03F5, "ColorHalftoningInfo");
				dic[0x03F6] = new ResourceTag(0x03F6, "DuotoneHalftoningInfo");
				dic[0x03F7] = new ResourceTag(0x03F7, "BW_TransferFunc");
				dic[0x03F8] = new ResourceTag(0x03F8, "ColorTransferFuncs");
				dic[0x03F9] = new ResourceTag(0x03F9, "DuotoneTransferFuncs");
				dic[0x03FA] = new ResourceTag(0x03FA, "DuotoneImageInfo");
				dic[0x03FB] = new ResourceTag(0x03FB, "EffectiveBW");
				dic[0x03FC] = new ResourceTag(0x03FC, "ObsoletePhotoshopTag1");
				dic[0x03FD] = new ResourceTag(0x03FD, "EPSOptions");
				dic[0x03FE] = new ResourceTag(0x03FE, "QuickMaskInfo");
				dic[0x03FF] = new ResourceTag(0x03FF, "ObsoletePhotoshopTag2");
				dic[0x0400] = new ResourceTag(0x0400, "LayerStateInfo");
				dic[0x0401] = new ResourceTag(0x0401, "WorkingPath");
				dic[0x0402] = new ResourceTag(0x0402, "LayersGroupInfo");
				dic[0x0403] = new ResourceTag(0x0403, "ObsoletePhotoshopTag3");
				dic[0x0404] = new ResourceTag(0x0404, "IPTCData", typeof(IptcIimRecord[]));
				dic[0x0405] = new ResourceTag(0x0405, "RawImageMode");
				dic[0x0406] = new ResourceTag(0x0406, "JPEG_Quality", typeof(JpegQuality));
				dic[0x0408] = new ResourceTag(0x0408, "GridGuidesInfo");
				dic[0x0409] = new ResourceTag(0x0409, "PhotoshopBGRThumbnail");
				dic[0x040A] = new ResourceTag(0x040A, "CopyrightFlag", typeof(bool));
				dic[0x040B] = new ResourceTag(0x040B, "URL", Encoding.ASCII);
				dic[0x040C] = new ResourceTag(0x040C, "PhotoshopThumbnail");
				dic[0x040D] = new ResourceTag(0x040D, "GlobalAngle", typeof(uint));
				dic[0x040E] = new ResourceTag(0x040E, "ColorSamplersResource");
				dic[0x040F] = new ResourceTag(0x040F, "ICC_Profile");
				dic[0x0410] = new ResourceTag(0x0410, "Watermark");
				dic[0x0411] = new ResourceTag(0x0411, "ICC_Untagged", typeof(bool));
				dic[0x0412] = new ResourceTag(0x0412, "EffectsVisible");
				dic[0x0413] = new ResourceTag(0x0413, "SpotHalftone");
				dic[0x0414] = new ResourceTag(0x0414, "IDsBaseValue", typeof(uint));
				dic[0x0415] = new ResourceTag(0x0415, "UnicodeAlphaNames");
				dic[0x0416] = new ResourceTag(0x0416, "IndexedColourTableCount");
				dic[0x0417] = new ResourceTag(0x0417, "TransparentIndex");
				dic[0x0419] = new ResourceTag(0x0419, "GlobalAltitude", typeof(uint));
				dic[0x041A] = new ResourceTag(0x041A, "Slices");
				dic[0x041B] = new ResourceTag(0x041B, "WorkflowURL");
				dic[0x041C] = new ResourceTag(0x041C, "JumpToXPEP");
				dic[0x041D] = new ResourceTag(0x041D, "AlphaIdentifiers");
				dic[0x041E] = new ResourceTag(0x041E, "URL_List");
				dic[0x0421] = new ResourceTag(0x0421, "VersionInfo", typeof(VersionInfo));
				dic[0x0422] = new ResourceTag(0x0422, "ExifInfo");
				dic[0x0424] = new ResourceTag(0x0424, "XMP");
				dic[0x0BB7] = new ResourceTag(0x0BB7, "ClippingPathName");
				dic[0x0FA0] = new ResourceTag(0x0FA0, null, Encoding.ASCII);
				dic[0x0FA1] = new ResourceTag(0x0FA1, null, Encoding.ASCII);
				dic[0x2710] = new ResourceTag(0x2710, "PrintFlagsInfo");
				return dic;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public sealed class VersionInfo
		{
			#region private fields
			private readonly byte[] header;
			private readonly string productName;
			private readonly string version;
			private readonly byte[] footer;
			#endregion

			// internal RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="reader"></param>
			internal VersionInfo(BinaryReaderNeo reader)
			{
				reader = new BinaryReaderNeo(
						reader.BaseStream, ByteOrder.BigEndian, Encoding.BigEndianUnicode);
				this.header = reader.ReadBytes(5);
				int length;
				length = reader.ReadInt32();
				this.productName = reader.ReadString(length);
				length = reader.ReadInt32();
				this.version = reader.ReadString(length);
				this.footer = reader.ReadToEnd();
			}

			// public \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				var baf = new ByteArrayFormatter();
				var qsf = new QuotedStringFormatter();
				var sw = new StringWriter();
				sw.Write("{");
				baf.WriteTo(sw, this.header);
				sw.Write(", ProductName=");
				qsf.WriteTo(sw, this.productName);
				sw.Write(", Version=");
				qsf.WriteTo(sw, this.version);
				sw.Write(", ");
				baf.WriteTo(sw, this.footer);
				sw.Write("}");
				return sw.ToString();
			}
		}
	}
}
