using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Security.Permissions;

using SystemNeo;
using SystemNeo.Collections.Generic;

namespace SystemNeo.Windows
{
	/// <summary>
	/// 
	/// </summary>
	internal class FileSystemEnumerable : AbstractEnumerable<FileSystemInfo>
	{
		#region private fields
		private bool directoryOnly;
		private string path;
		private string searchPattern;
		#endregion

		// internal vpeB //

		/// <summary>
		/// 
		/// </summary>
		internal string FullPath
		{
			get {
				return Path.GetFullPath(this.path);
			}
		}

		// internal RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="path"></param>
		/// <param name="searchPattern"></param>
		/// <param name="directoryOnly"></param>
		internal FileSystemEnumerable(string path, string searchPattern, bool directoryOnly)
		{
			ArgumentUtil.AssertNull(path, "path");
			ArgumentUtil.AssertNull(searchPattern, "searchPattern");
			this.path = path;
			this.searchPattern = searchPattern;
			this.directoryOnly = directoryOnly;
		}

		// protected \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		protected override IEnumerator<FileSystemInfo> GetEnumeratorInternal()
		{
			if (this.searchPattern == string.Empty) {
				IEnumerable<FileSystemInfo> e = new FileSystemInfo[] {};
				return e.GetEnumerator();
			} else {
				return new Enumerator(this);
			}
		}

		// ^ //

		/// <summary>
		/// 
		/// </summary>
		private class Enumerator : IEnumerator<FileSystemInfo>, IDisposable
		{
			#region private fields
			private FileSystemInfo current;
			private IntPtr findHandle = IntPtr.Zero;
			private readonly FileSystemEnumerable owner;
			#endregion

			// private vpeB //

			/// <summary>
			/// 
			/// </summary>
			FileSystemInfo IEnumerator<FileSystemInfo>.Current
			{
				get {
					return this.current;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			object IEnumerator.Current
			{
				get {
					return this.current;
				}
			}

			// internal RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="owner"></param>
			internal Enumerator(FileSystemEnumerable owner)
			{
				this.owner = owner;
			}

			// public \bh //

			/// <summary>
			/// 
			/// </summary>
			public void Dispose()
			{
				if (this.findHandle != IntPtr.Zero) {
					NativeMethods.FindClose(this.findHandle);
					this.findHandle = IntPtr.Zero;
				}
				this.current = null;
			}

			// private \bh //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="findData"></param>
			/// <returns></returns>
			private bool Accept(ref WIN32_FIND_DATA findData)
			{
				if (findData.cFileName == FileUtil.CurrentDirectoryPath) {
					return false;
				}
				if (findData.cFileName == FileUtil.ParentDirectoryPath) {
					return false;
				}
				if (this.owner.directoryOnly && ! findData.IsDirectory) {
					return false;
				}
				return true;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="findData"></param>
			/// <returns></returns>
			private FileSystemInfo AsFileSystemInfo(ref WIN32_FIND_DATA findData)
			{
				string fullPath = Path.Combine(this.owner.FullPath, findData.cFileName);
				if (findData.IsDirectory) {
					return new DirectoryInfo(fullPath);
				} else {
					return new FileInfo(fullPath);
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="findData"></param>
			/// <returns></returns>
			private IntPtr FindFirst(ref WIN32_FIND_DATA findData)
			{
				string fullPath = this.owner.FullPath;
				new FileIOPermission(FileIOPermissionAccess.PathDiscovery, fullPath).Demand();
				if (! Directory.Exists(fullPath)) {
					throw new DirectoryNotFoundException();
				}
				string dir = Path.GetDirectoryName(this.owner.searchPattern);
				if (dir != null && dir != string.Empty) {
					throw new NotSupportedException();
				}
				string searchPath = Path.Combine(fullPath, this.owner.searchPattern);
				return NativeMethods.FindFirstFile(searchPath, ref findData);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			bool IEnumerator.MoveNext()
			{
				var findData = new WIN32_FIND_DATA();
				bool found;
				do {
					if (this.findHandle == IntPtr.Zero) {
						this.findHandle = this.FindFirst(ref findData);
						found = (this.findHandle != NativeMethods.INVALID_HANDLE_VALUE);
					} else {
						found = NativeMethods.FindNextFile(this.findHandle, ref findData);
					}
				} while (found && ! this.Accept(ref findData));
				if (found) {
					this.current = this.AsFileSystemInfo(ref findData);
				} else {
					this.Dispose();
				}
				return found;
			}

			/// <summary>
			/// 
			/// </summary>
			void IEnumerator.Reset()
			{
				this.Dispose();
			}
		}
	}
}
