﻿using System;
using System.Reflection;

namespace SystemNeo.Reflection
{
	/// <summary>
	/// 型に関するメソッドを提供します。
	/// </summary>
	public static class TypeUtil
	{
		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <returns></returns>
		public static T CreateInstance<T>()
		{
			return CreateInstance<T>(new object[] {});
		}

		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="param1"></param>
		/// <returns></returns>
		public static T CreateInstance<T>(object param1)
		{
			return CreateInstance<T>(new object[] {param1});
		}

		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="param1"></param>
		/// <param name="param2"></param>
		/// <returns></returns>
		public static T CreateInstance<T>(object param1, object param2)
		{
			return CreateInstance<T>(new object[] {param1, param2});
		}

		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="param1"></param>
		/// <param name="param2"></param>
		/// <param name="param3"></param>
		/// <returns></returns>
		public static T CreateInstance<T>(
				object param1, object param2, object param3)
		{
			return CreateInstance<T>(new object[] {param1, param2, param3});
		}

		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="parameters"></param>
		/// <returns></returns>
		public static T CreateInstance<T>(object[] parameters)
		{
			ArgumentUtil.AssertNull(parameters, "parameters");
			var types = new Type[parameters.Length];
			for (int i = 0; i < parameters.Length; i++) {
				if (parameters[i] == null) {
					throw new ArgumentException("null 参照を含んでいます。", "parameters");
				}
				types[i] = parameters[i].GetType();
			}
			return CreateInstance<T>(MemberUtil.AllBindingFlags, types, parameters);
		}

		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <typeparam name="T"></typeparam>
		/// <param name="paramTypes"></param>
		/// <param name="parameters"></param>
		/// <returns></returns>
		public static T CreateInstance<T>(Type[] paramTypes, object[] parameters)
		{
			return CreateInstance<T>(MemberUtil.AllBindingFlags, paramTypes, parameters);
		}

		/// <summary>
		/// 型 <typeparamref name="T"/> のインスタンスを生成します。
		/// </summary>
		/// <param name="type"></param>
		/// <param name="bindingAttr"></param>
		/// <param name="paramTypes"></param>
		/// <param name="parameters"></param>
		/// <returns></returns>
		public static T CreateInstance<T>(
				BindingFlags bindingAttr, Type[] paramTypes, object[] parameters)
		{
			ConstructorInfo constructor = GetConstructor(typeof(T), bindingAttr, paramTypes);
			if (constructor == null) {
				return default(T);
			}
			if (MemberUtil.GetAccessibility(constructor) == MemberAccessibility.Private) {
				return default(T);
			}
			return (T)constructor.Invoke(parameters);
		}

		/// <summary>
		/// 指定された型の参照可能範囲を取得します。
		/// </summary>
		/// <param name="type"></param>
		/// <returns></returns>
		public static MemberAccessibility GetAccessibility(Type type)
		{
			if (type.IsPublic || type.IsNestedPublic) {
				return MemberAccessibility.Public;
			}
			if (type.IsNestedFamily) {
				return MemberAccessibility.Family;
			}
			if (type.IsNestedFamANDAssem) {
				return MemberAccessibility.FamilyAndAssembly;
			}
			if (type.IsNestedFamORAssem) {
				return MemberAccessibility.FamilyOrAssembly;
			}
			if (type.IsNestedAssembly) {
				return MemberAccessibility.Assembly;
			}
			if (type.IsNestedPrivate) {
				return MemberAccessibility.Private;
			}
			return MemberAccessibility.Assembly;
		}

		/// <summary>
		/// 指定された型のコンストラクタを取得します。
		/// </summary>
		/// <param name="type"></param>
		/// <param name="paramTypes"></param>
		/// <returns></returns>
		public static ConstructorInfo GetConstructor(Type type, Type[] paramTypes)
		{
			return GetConstructor(type, MemberUtil.AllBindingFlags, paramTypes);
		}

		/// <summary>
		/// 指定された型のコンストラクタを取得します。
		/// </summary>
		/// <param name="type"></param>
		/// <param name="bindingAttr"></param>
		/// <param name="paramTypes"></param>
		/// <returns></returns>
		public static ConstructorInfo GetConstructor(
				Type type, BindingFlags bindingAttr, Type[] paramTypes)
		{
			return GetConstructor(type, bindingAttr, null, paramTypes);
		}

		/// <summary>
		/// 指定された型のコンストラクタを取得します。
		/// </summary>
		/// <param name="type"></param>
		/// <param name="bindingAttr"></param>
		/// <param name="binder"></param>
		/// <param name="paramTypes"></param>
		/// <returns></returns>
		public static ConstructorInfo GetConstructor(
				Type type, BindingFlags bindingAttr, Binder binder, Type[] paramTypes)
		{
			ArgumentUtil.AssertNull(type, "type");
			ArgumentUtil.AssertNull(paramTypes, "paramTypes");
			var modifiers = new ParameterModifier[paramTypes.Length];
			return type.GetConstructor(bindingAttr, binder, paramTypes, modifiers);
		}
	}
}
