using System;
using System.Collections;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Text;

using SystemNeo;
using SystemNeo.Collections;

namespace SystemNeo.Text
{
	/// <summary>
	/// C# ̃R[h͂p[TłB
	/// </summary>
	public class CSharpCodeParser : Parser
	{
		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		/// <returns></returns>
		public string EvaluateStringLiteral(string value)
		{
			IEnumerable<TextElement> elements = this.ParseGradually(value).Where(IsTokenElement);
			var sb = new StringBuilder();
			int i = 0;
			foreach (TextElement elem in elements) {
				if (i++ % 2 == 0) {
					// Ԗڂ̓eł邱
					var literal = elem as StringLiteral;
					if (literal == null) {
						throw new ParseException("ɂ̓eȊOu܂B");
					}
					sb.Append(literal.Value);
				} else {
					// Ԗڂ͘AZqi"+"jł邱
					var symbol = elem as Symbol;
					if (symbol == null || symbol.SourceText != "+") {
						throw new ParseException("ɂ '+' ȊOu܂B");
					}
				}
			}
			return sb.ToString();
		}

		// protected \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="value"></param>
		/// <param name="sr"></param>
		/// <param name="readers"></param>
		protected override void GetReader(
				string value, out StringReaderNeo sr, out Func<TextElement>[] readers)
		{
			var cssr = new CSharpStringReader(value);
			readers = new Func<TextElement>[] {
					cssr.ReadSingleLineComment,
					cssr.ReadDelimitedComment,
					cssr.ReadPreprocessorDirective,
					cssr.ReadCharacterLiteral,
					cssr.ReadRegularStringLiteral,
					cssr.ReadVerbatimStringLiteral,
					cssr.ReadHexIntegerLiteral,
					cssr.ReadNumericLiteral,
					cssr.ReadIdentifier,
					cssr.ReadSymbol,
					cssr.ReadWhiteSpace,
					cssr.ReadNewLine
			};
			sr = cssr;
		}

		// private static \bh //

		/// <summary>
		/// ʎq̂Qڈȍ~ƂĎgpł镶ǂ肵܂B
		/// </summary>
		/// <param name="c"></param>
		/// <returns></returns>
		private static bool IsIdentifierPartCharacter(char c)
		{
			switch (char.GetUnicodeCategory(c)) {
			case UnicodeCategory.DecimalDigitNumber:    // Nd
			case UnicodeCategory.ConnectorPunctuation:  // Pc
			case UnicodeCategory.NonSpacingMark:        // Mn
			case UnicodeCategory.SpacingCombiningMark:  // Mc
			case UnicodeCategory.Format:                // Cf
				return true;
			}
			return IsLetterCharacter(c);
		}

		/// <summary>
		/// ʎq̂PڂƂĎgpł镶ǂ肵܂B
		/// </summary>
		/// <param name="c"></param>
		/// <returns></returns>
		private static bool IsLetterCharacter(char c)
		{
			switch (char.GetUnicodeCategory(c)) {
			case UnicodeCategory.UppercaseLetter:       // Lu
			case UnicodeCategory.LowercaseLetter:       // Ll
			case UnicodeCategory.TitlecaseLetter:       // Lt
			case UnicodeCategory.ModifierLetter:        // Lm
			case UnicodeCategory.OtherLetter:           // Lo
			case UnicodeCategory.LetterNumber:          // Nl
				return true;
			}
			return false;
		}

		/// <summary>
		/// s\ǂ肵܂B
		/// </summary>
		/// <param name="c"></param>
		/// <returns></returns>
		private static bool IsNewLineCharacter(char c)
		{
			switch (c) {
			case CharUtil.CarriageReturn:
			case CharUtil.LineFeed:
			case '\x2028':
			case '\x2029':
				return true;
			}
			return false;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="element"></param>
		/// <returns></returns>
		private static bool IsTokenElement(TextElement element)
		{
			return ! (element is NonToken);
		}

		/// <summary>
		/// 󔒂\ǂ肵܂B
		/// </summary>
		/// <param name="c"></param>
		/// <returns></returns>
		private static bool IsWhiteSpaceCharacter(char c)
		{
			switch (c) {
			case CharUtil.Tab:
			case CharUtil.VerticalTab:
			case '\f':
				return true;
			}
			return char.GetUnicodeCategory(c) == UnicodeCategory.SpaceSeparator;
		}

		// ^ //

		/// <summary>
		/// C# ̃R[h̗vf؂o@\
		/// <see cref="SystemNeo.Text.StringReaderEx"> StringReaderEx</see> łB
		/// </summary>
		private class CSharpStringReader : StringReaderNeo
		{
			// private static tB[h //

			/// <summary>
			/// 
			/// </summary>
			private static readonly Predicate<char> AsciiDigitCharFilter = CharUtil.IsAsciiDigit;

			/// <summary>
			/// 
			/// </summary>
			private static readonly Predicate<char> AsciiHexCharFilter = CharUtil.IsAsciiHex;

			/// <summary>
			/// 
			/// </summary>
			private static readonly Predicate<char> NewLineCharFilter = IsNewLineCharacter;

			/// <summary>
			/// 
			/// </summary>
			private static readonly Predicate<char> WhiteSpaceCharFilter = IsWhiteSpaceCharacter;

			// RXgN^ //

			/// <summary>
			/// 
			/// </summary>
			/// <param name="value"></param>
			internal CSharpStringReader(string value) : base(value) {}

			// internal \bh //

			/// <summary>
			/// e̒l擾܂B
			/// </summary>
			/// <param name="sequence"></param>
			/// <returns></returns>
			internal char GetCharacter(out string sequence)
			{
				sequence = null;

				int c = this.Read();
				switch (c) {
				case -1:
					throw new ParseException("R[hr؂Ă܂B");
				case CharUtil.BackSlash:
					int cEscaped = this.Read();
					sequence = string.Empty + CharUtil.BackSlash + (char)cEscaped;

					switch (cEscaped) {
					case -1:
						throw new ParseException("'\\' ̌ɕ܂B");
					case CharUtil.SingleQuote:
					case CharUtil.DoubleQuote:
					case CharUtil.BackQuote:
						return (char)cEscaped;
					case '0':
						return CharUtil.Null;
					case 'a':
						return '\a';
					case 'b':
						return '\b';
					case 'f':
						return '\f';
					case 'n':
						return CharUtil.LineFeed;
					case 'r':
						return CharUtil.CarriageReturn;
					case 't':
						return CharUtil.Tab;
					case 'v':
						return CharUtil.VerticalTab;
					case 'x':
						char cResult = this.GetEscapedUnicodeCharacter(out sequence);
						sequence = "\\x" + sequence;
						return cResult;
					}
					throw new ParseException("'\\' ̌ɕsȕ܂BU+" + ((UInt32)cEscaped).ToString("X4"));
				default:
					if (IsNewLineCharacter((char)c)) {
						throw new ParseException("ɉs͒u܂B");
					}
					break;
				}
				sequence = string.Empty + (char)c;
				return (char)c;
			}

			/// <summary>
			/// Unicode GXP[vV[PX\Ă镶擾܂B
			/// </summary>
			/// <param name="sequence"></param>
			/// <returns></returns>
			internal char GetEscapedUnicodeCharacter(out string sequence)
			{
				sequence = this.Read(AsciiHexCharFilter, 4);
				if (sequence.Length == 0) {
					throw new ParseException("16i܂B");
				}
				return (char)uint.Parse(sequence, NumberStyles.AllowHexSpecifier);
			}

			/// <summary>
			/// ݍs̍s܂łǂݏo܂B
			/// </summary>
			/// <returns></returns>
			internal string GetLineContent()
			{
				return this.Read(NewLineCharFilter, true);
			}

			/// <summary>
			/// 10iŗp鐔̕т擾܂B
			/// </summary>
			/// <returns></returns>
			internal string GetDecimalDigits()
			{
				return this.Read(AsciiDigitCharFilter);
			}

			/// <summary>
			/// ZqȂǂ̋Lނ\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadSymbol()
			{
				int c = this.Read();
				int c2;

				switch (c) {
				case '{': case '}':
				case '[': case ']':
				case '(': case ')':
				case CharUtil.Period:
				case CharUtil.Comma:
				case CharUtil.Colon:
				case CharUtil.Semicolon:
				case CharUtil.QuestionMark:
				case CharUtil.Tilde:
					return new Symbol((char)c);
				case CharUtil.Plus:
					switch (this.Peek()) {
					case CharUtil.Plus:
					case CharUtil.Equal:
						return new Symbol(CharUtil.Plus, (char)this.Read());
					}
					return new Symbol(CharUtil.Plus);
				case CharUtil.Minus:
					switch (this.Peek()) {
					case CharUtil.Minus:
					case CharUtil.Equal:
					case '>':
						return new Symbol(CharUtil.Minus, (char)this.Read());
					}
					return new Symbol(CharUtil.Minus);
				case CharUtil.Asterisk:
				case CharUtil.Slash:
				case CharUtil.Percent:
				case CharUtil.Equal:
				case CharUtil.ExclamationMark:
				case '^':
					switch (this.Peek()) {
					case CharUtil.Equal:
						return new Symbol((char)c, (char)this.Read());
					}
					return new Symbol((char)c);
				case CharUtil.Ampersand:
				case CharUtil.Pipe:
					c2 = this.Peek();
					if (c2 == c || c2 == CharUtil.Equal) {
						return new Symbol((char)c, (char)this.Read());
					}
					return new Symbol((char)c);
				case '<':
				case '>':
					c2 = this.Peek();
					if (c2 == CharUtil.Equal) {
						return new Symbol((char)c, (char)this.Read());
					} else if (c2 == c) {
						this.Read();
						if (this.Peek() == CharUtil.Equal) {
							return new Symbol((char)c, (char)c, (char)this.Read());
						} else {
							return new Symbol((char)c, (char)c);
						}
					}
					return new Symbol((char)c);
				}
				this.Seek(-1);
				return null;
			}

			/// <summary>
			/// s\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadNewLine()
			{
				if (this.StartsWith(StringUtil.CrLf)) {
					return new NewLine(this.Read(StringUtil.CrLf.Length));
				}
				int c = this.Peek();
				if (c >= 0 && IsNewLineCharacter((char)c)) {
					return new NewLine(this.Read(1));
				}
				return null;
			}

			/// <summary>
			/// ʎq\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadIdentifier()
			{
				bool atmark = false;
				if (this.Peek() == CharUtil.Atmark) {
					atmark = true;
					this.Seek(1);
				}
				StringBuilder sbName = null;
				StringBuilder sbSequence = null;

				for (;;) {
					int c;
					string sequence;

					if (this.StartsWith("\\u") || this.StartsWith("\\U")) {
						string prefix = this.Read(2);
						c = this.GetEscapedUnicodeCharacter(out sequence);
						sequence = prefix + sequence;
					} else {
						c = this.Read();
						if (c == -1) {
							break;
						}
						sequence = ((char)c).ToString();
					}
					if (sbName == null) {
						// 擪̕
						if (c == CharUtil.Underscore || IsLetterCharacter((char)c)) {
							sbName = new StringBuilder();
							sbName.Append((char)c);
							sbSequence = new StringBuilder();
							sbSequence.Append(sequence);
						} else {
							this.Seek(atmark ? -2 : -1);
							return null;
						}
					} else {
						// QԖڈȍ~̕
						if (IsIdentifierPartCharacter((char)c)) {
							sbName.Append((char)c);
							sbSequence.Append(sequence);
						} else {
							this.Seek(-1);
							break;
						}
					}
				}
				if (sbSequence == null) {
					return null;
				}
				string source = (atmark ? "@" : string.Empty) + sbSequence.ToString();
				Element result = null;
				if (! atmark) {
					result = Keyword.GetInstance(source);
					if (result != null) {
						return result;
					}
				}
				return new Identifier(source, sbName.ToString(), atmark);
			}

			/// <summary>
			/// ܂͎e\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadNumericLiteral()
			{
				int c = this.Peek();
				if (! (CharUtil.IsAsciiDigit((char)c) || c == CharUtil.Period)) {
					return null;
				}

				string intPart = this.GetDecimalDigits();
				string frecPart = null;
				var sb = new StringBuilder(intPart);
				bool isLong = false;
				bool isUnsigned = false;

				c = this.Peek();
				switch (c) {
				case 'U':
				case 'u':
					sb.Append((char)this.Read());
					isUnsigned = true;
					switch (this.Peek()) {
					case 'L':
					case 'l':
						sb.Append((char)this.Read());
						isLong = true;
						break;
					}
					return new DecimalIntegerLiteral(sb.ToString(), intPart, isLong, isUnsigned);
				case 'L':
				case 'l':
					sb.Append((char)this.Read());
					isLong = true;
					switch (this.Peek()) {
					case 'U':
					case 'u':
						sb.Append((char)this.Read());
						isUnsigned = true;
						break;
					}
					return new DecimalIntegerLiteral(sb.ToString(), intPart, isLong, isUnsigned);
				case CharUtil.Period:
					sb.Append((char)this.Read());
					frecPart = this.GetDecimalDigits();
					if (frecPart.Length == 0) {
						if (intPart.Length == 0) {
							this.Seek(-1);
							return null;
						}
						throw new ParseException("_̌ɐ܂B");
					}
					sb.Append(frecPart);
					c = this.Peek();
					break;
				}

				string exponentPart = null;
				if (c == 'E' || c == 'e') {
					string exponentSign = null;
					sb.Append((char)this.Read());
					c = this.Peek();
					if (c == CharUtil.Plus || c == CharUtil.Minus) {
						sb.Append((char)this.Read());
						exponentSign = ((char)c).ToString();
					}
					string exponentDigits = this.GetDecimalDigits();
					if (exponentDigits.Length == 0) {
						throw new ParseException("'e' ̌Ɏw܂B");
					}
					sb.Append(exponentDigits);
					exponentPart = exponentSign + exponentDigits;
					c = this.Peek();
				}

				switch (c) {
				case 'F':
				case 'f':
				case 'D':
				case 'd':
				case 'M':
				case 'm':
					sb.Append((char)this.Read());
					break;
				}
				return new RealLiteral(sb.ToString(), intPart, frecPart, exponentPart);
			}

			/// <summary>
			/// e\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadCharacterLiteral()
			{
				if (this.Peek() != CharUtil.SingleQuote) {
					return null;
				}
				this.Read();
				string sequence;
				char c = this.GetCharacter(out sequence);
				if (sequence == CharUtil.SingleQuote.ToString()) {
					throw new ParseException("e̓e܂B");
				}
				var cl = new CharacterLiteral("'" + sequence + "'", c);
				if (this.Read() != CharUtil.SingleQuote) {
					throw new ParseException("eĂ܂B");
				}
				return cl;
			}

			/// <summary>
			/// We\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadRegularStringLiteral()
			{
				if (this.Peek() != CharUtil.DoubleQuote) {
					return null;
				}
				var sbSource = new StringBuilder();
				sbSource.Append((char)this.Read());
				var sbValue = new StringBuilder();

				for (;;) {
					switch (this.Peek()) {
					case CharUtil.DoubleQuote:
						sbSource.Append((char)this.Read());
						return new StringLiteral(sbSource.ToString(), sbValue.ToString());
					default:
						string sequence;
						char c = this.GetCharacter(out sequence);
						sbSource.Append(sequence);
						sbValue.Append(c);
						break;
					}
				}
			}

			/// <summary>
			/// vvZbT fBNeBu\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadPreprocessorDirective()
			{
				if (this.Peek() == CharUtil.Sharp) {
					return new PreprocessorDirective(this.GetLineContent());
				} else {
					return null;
				}
			}

			/// <summary>
			/// Ie\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadVerbatimStringLiteral()
			{
				const string PREFIX = "@\"";

				if (! this.StartsWith(PREFIX)) {
					return null;
				}
				var sbSource = new StringBuilder(this.Read(PREFIX.Length));
				var sbValue = new StringBuilder();
				for (;;) {
					int c = this.Read();
					switch (c) {
					case -1:
						throw new ParseException("񃊃eĂ܂B");
					case CharUtil.DoubleQuote:
						if (this.Peek() == CharUtil.DoubleQuote) {
							sbSource.Append(CharUtil.DoubleQuote);
							sbSource.Append(CharUtil.DoubleQuote);
							sbValue.Append((char)this.Read());
						} else {
							return new StringLiteral(sbSource.ToString(), sbValue.ToString());
						}
						break;
					default:
						sbSource.Append((char)c);
						sbValue.Append((char)c);
						break;
					}
				}
			}

			/// <summary>
			/// PsRg\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadSingleLineComment()
			{
				const string DOUBLE_SLASH = "//";

				if (! this.StartsWith(DOUBLE_SLASH)) {
					return null;
				}
				this.Seek(DOUBLE_SLASH.Length);
				string content = this.GetLineContent();
				return new Comment(DOUBLE_SLASH + content, content);
			}

			/// <summary>
			/// ؂LtRg\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadDelimitedComment()
			{
				const string PREFIX = "/*";
				const string SUFFIX = "*/";

				if (! this.StartsWith(PREFIX)) {
					return null;
				}
				this.Seek(PREFIX.Length);
				string content = this.FindRead(SUFFIX);
				if (this.StartsWith(SUFFIX)) {
					return new Comment(PREFIX + content + this.Read(SUFFIX.Length), content);
				} else {
					throw new ParseException("RgĂ܂B");
				}
			}

			/// <summary>
			/// 16i̐e\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadHexIntegerLiteral()
			{
				if (! (this.StartsWith("0x") || this.StartsWith("0X"))) {
					return null;
				}
				var sb = new StringBuilder();
				sb.Append(this.Read(2));

				string digits = this.Read(AsciiHexCharFilter);
				if (digits.Length == 0) {
					throw new ParseException("'0x' ̌16i܂B");
				}
				sb.Append(digits);
				bool isLong = false;
				bool isUnsigned = false;

				switch (this.Peek()) {
				case 'U':
				case 'u':
					sb.Append((char)this.Read());
					isUnsigned = true;
					switch (this.Peek()) {
					case 'L':
					case 'l':
						sb.Append((char)this.Read());
						isLong = true;
						break;
					}
					break;
				case 'L':
				case 'l':
					sb.Append((char)this.Read());
					isLong = true;
					switch (this.Peek()) {
					case 'U':
					case 'u':
						sb.Append((char)this.Read());
						isUnsigned = true;
						break;
					}
					break;
				}
				return new HexIntegerLiteral(sb.ToString(), digits, isLong, isUnsigned);
			}

			/// <summary>
			/// 󔒕\ Element 擾܂B
			/// </summary>
			/// <returns></returns>
			internal TextElement ReadWhiteSpace()
			{
				string source = this.Read(WhiteSpaceCharFilter);
				return source.Length == 0 ? null : new WhiteSpace(source);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public interface ILiteral
		{
			/// <summary>
			/// 
			/// </summary>
			object Value { get; }

			/// <summary>
			/// 
			/// </summary>
			Type ValueType { get; }
		}

		/// <summary>
		/// 
		/// </summary>
		public class Element : TextElement
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="srcText"></param>
			public Element(string srcText) : base(srcText) {}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				string className = this.GetType().FullName;
				className = System.Text.RegularExpressions.Regex.Replace(className, ".+[.+]", "");
				return className + " [" + this.SourceText + "]";
			}
		}

		/// <summary>
		/// vvZbT fBNeBu\
		/// <see cref="SystemNeo.Text.CSharpCodeParser.Element"> Element</see> łB
		/// </summary>
		public class PreprocessorDirective : Element
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public PreprocessorDirective(string source) : base(source) {}
		}

		/// <summary>
		/// g[NȊO̗vf\
		/// <see cref="SystemNeo.Text.CSharpCodeParser.Element"> Element</see> łB
		/// </summary>
		public class NonToken : Element
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public NonToken(string source) : base(source) {}
		}

		/// <summary>
		/// Rg\ <see cref="SystemNeo.Text.CSharpCodeParser.Element">Element</see> łB
		/// </summary>
		public class Comment : NonToken
		{
			/// <summary>
			/// 
			/// </summary>
			public string Content { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="content"></param>
			public Comment(string source, string content) : base(source)
			{
				this.Content = content;
			}
		}

		/// <summary>
		/// s\ <see cref="SystemNeo.Text.CSharpCodeParser.Element">Element</see> łB
		/// </summary>
		public class NewLine : NonToken
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public NewLine(string source) : base(source) {}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "NewLine";
			}
		}

		/// <summary>
		/// 󔒕\ <see cref="SystemNeo.Text.CSharpCodeParser.Element">Element</see> łB
		/// </summary>
		public class WhiteSpace : NonToken
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public WhiteSpace(string source) : base(source) {}
		}

		/// <summary>
		/// g[N\ <see cref="SystemNeo.Text.CSharpCodeParser.Element">Element</see> łB
		/// </summary>
		public class Token : Element
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public Token(string source) : base(source) {}
		}

		/// <summary>
		/// 
		/// </summary>
		public class Keyword : Token
		{
			#region private static fields
			private static readonly string[] keywords;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			static Keyword()
			{
				keywords = new string[] {"abstract", "as", "base", "bool",
										 "break", "byte", "case", "catch", "char",
										 "checked", "class", "const", "continue",
										 "decimal", "default", "delegate", "do",
										 "double", "else", "enum", "event",
										 "explicit", "extern", "finally",
										 "fixed", "float", "for", "foreach",
										 "goto", "if", "implicit", "in", "int",
										 "interface", "internal", "is", "lock",
										 "long", "namespace", "new", "null",
										 "object", "operator", "out", "override",
										 "params", "private", "protected",
										 "public", "readonly", "ref", "return",
										 "sbyte", "sealed", "short", "sizeof",
										 "stackalloc", "static", "string",
										 "struct", "switch", "this", "throw",
										 "try", "typeof", "uint", "ulong",
										 "unchecked", "unsafe", "ushort",
										 "using", "virtual", "void", "volatile",
										 "while"};
				Array.Sort(keywords);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="srcText"></param>
			protected Keyword(string srcText) : base(srcText) {}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="srcText"></param>
			/// <returns></returns>
			public static Keyword GetInstance(string srcText)
			{
				bool isTrue = (srcText == bool.TrueString.ToLower());
				bool isFalse = (srcText == bool.FalseString.ToLower());
				if (isTrue || isFalse) {
					return new BooleanLiteral(srcText, isTrue);
				} else if (Array.IndexOf(keywords, srcText) >= 0) {
					return new Keyword(srcText);
				} else {
					return null;
				}
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public abstract class KeywordLiteral : Keyword, ILiteral
		{
			/// <summary>
			/// 
			/// </summary>
			public abstract object Value { get; }

			/// <summary>
			/// 
			/// </summary>
			public abstract Type ValueType { get; }

			/// <summary>
			/// 
			/// </summary>
			/// <param name="srcText"></param>
			internal KeywordLiteral(string srcText) : base(srcText) {}
		}

		/// <summary>
		/// 
		/// </summary>
		public class BooleanLiteral : KeywordLiteral
		{
			#region private fields
			private readonly object value;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			public override object Value
			{
				get {
					return this.value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			public override Type ValueType
			{
				get {
					return typeof(bool);
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="srcText"></param>
			/// <param name="value"></param>
			internal BooleanLiteral(string srcText, bool value) : base(srcText)
			{
				this.value = value;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public class Identifier : Token
		{
			#region private fields
			private readonly string name;
			private readonly bool notKeyword;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			public string Name
			{
				get {
					return this.name;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="name"></param>
			public Identifier(string source, string name) : this(source, name, false) {}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="name"></param>
			/// <param name="notKeyword"></param>
			public Identifier(string source, string name, bool notKeyword) : base(source)
			{
				this.name = name;
				this.notKeyword = notKeyword;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public bool IsKeyword()
			{
				return ! this.notKeyword;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "Identifier [" + this.Name + "]" + (this.IsKeyword() ? "" : " NotKeyword");
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public class Symbol : Token
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public Symbol(string source) : base(source) {}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			public Symbol(params char[] source) : this(new string(source)) {}
		}

		/// <summary>
		/// e\g[NłB
		/// </summary>
		public abstract class Literal : Token, ILiteral
		{
			/// <summary>
			/// e̒l擾܂B
			/// </summary>
			public abstract object Value { get; }

			/// <summary>
			/// ě^擾܂B
			/// </summary>
			public abstract Type ValueType { get; }

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			protected Literal(string source) : base(source) {}
		}

		/// <summary>
		/// 
		/// </summary>
		public class CharacterLiteral : Literal
		{
			#region private fields
			private readonly char value;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			public override object Value
			{
				get {
					return this.value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			public override Type ValueType
			{
				get {
					return typeof(char);
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="value"></param>
			public CharacterLiteral(string source, char value) : base(source)
			{
				this.value = value;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "CharacterLiteral [U+" + ((ushort)this.value).ToString("X4") + "]";
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public class StringLiteral : Literal
		{
			#region private fields
			private readonly string value;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			public override object Value
			{
				get {
					return this.value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			public override Type ValueType
			{
				get {
					return typeof(string);
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="value"></param>
			public StringLiteral(string source, string value) : base(source)
			{
				this.value = value;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "StringLiteral [" + this.value + "]";
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public abstract class NumericLiteral : Literal
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			protected NumericLiteral(string source) : base(source) {}
		}

		/// <summary>
		/// 
		/// </summary>
		public class RealLiteral : NumericLiteral
		{
			#region private fields
			private readonly string intDigits;
			private readonly string frecDigits;
			private readonly string exponent;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			public override object Value
			{
				get {
					var sb = new StringBuilder();
					sb.Append(this.intDigits.Length == 0 ? "0" : this.intDigits);
					if (this.frecDigits.Length > 0) {
						sb.Append('.');
						sb.Append(this.frecDigits);
					}
					if (this.exponent.Length > 0) {
						sb.Append('e');
						sb.Append(this.exponent);
					}
					decimal val = decimal.Parse(sb.ToString(), NumberStyles.Float);
					Type t = this.ValueType;
					if (t == typeof(float)) {
						return (float)val;
					} else if (t == typeof(double)) {
						return (double)val;
					} else {
						return val;
					}
				}
			}

			/// <summary>
			/// 
			/// </summary>
			public override Type ValueType
			{
				get {
					switch (this.SourceText[this.SourceText.Length - 1]) {
					case 'F':
					case 'f':
						return typeof(float);
					case 'M':
					case 'm':
						return typeof(decimal);
					default:
						return typeof(double);
					}
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="intDigits"></param>
			/// <param name="frecDigits"></param>
			/// <param name="exponent"></param>
			public RealLiteral(string source,
					string intDigits, string frecDigits, string exponent) : base(source)
			{
				this.intDigits = intDigits;
				this.frecDigits = frecDigits;
				this.exponent = exponent;
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "RealLiteral ["
						+ this.intDigits + "].[" + this.frecDigits + "]e[" + this.exponent + "]";
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public abstract class IntegerLiteral : NumericLiteral
		{
			#region private fields
			private readonly object value;
			private readonly Type valueType;
			#endregion

			/// <summary>
			/// 
			/// </summary>
			public bool IsLong { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public bool Unsigned { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			public override object Value
			{
				get {
					return this.value;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			public override Type ValueType
			{
				get {
					return this.valueType;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			protected string Digits { get; private set; }

			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="digits"></param>
			/// <param name="isLong"></param>
			/// <param name="unsigned"></param>
			protected IntegerLiteral(
					string source, string digits, bool isLong, bool unsigned) : base(source)
			{
				this.Digits = digits;
				this.IsLong = isLong;
				this.Unsigned = unsigned;

				decimal val = this.GetValueAsDecimal();
				this.valueType = this.GetValueType(val);
				this.value = this.GetValue(val);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			protected abstract decimal GetValueAsDecimal();

			/// <summary>
			/// 
			/// </summary>
			/// <param name="val"></param>
			/// <returns></returns>
			private object GetValue(decimal val)
			{
				Type t = this.valueType;
				if (t == typeof(int)) {
					return (int)val;
				} else if (t == typeof(uint)) {
					return (uint)val;
				} else if (t == typeof(long)) {
					return (long)val;
				} else {
					return (ulong)val;
				}
			}

			/// <summary>
			/// 
			/// </summary>
			/// <param name="val"></param>
			/// <returns></returns>
			private Type GetValueType(decimal val)
			{
				if (this.IsLong) {
					if (this.Unsigned || val > long.MaxValue) {
						return typeof(ulong);
					} else {
						return typeof(long);
					}
				} else {
					if (this.Unsigned) {
						if (val <= uint.MaxValue) {
							return typeof(uint);
						} else {
							return typeof(ulong);
						}
					} else if (val <= int.MaxValue) {
						return typeof(int);
					} else if (val <= uint.MaxValue) {
						return typeof(uint);
					} else if (val <= long.MaxValue) {
						return typeof(long);
					} else {
						return typeof(ulong);
					}
				}
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public class DecimalIntegerLiteral : IntegerLiteral
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="digits"></param>
			/// <param name="isLong"></param>
			/// <param name="unsigned"></param>
			public DecimalIntegerLiteral(
					string source, string digits, bool isLong, bool unsigned)
					: base(source, digits, isLong, unsigned) {}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "DecimalIntegerLiteral [" + this.Digits + "]"
					+ (this.Unsigned ? " unsigned" : string.Empty)
					+ (this.IsLong ? " long" : string.Empty);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			protected override decimal GetValueAsDecimal()
			{
				return decimal.Parse(this.Digits);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public class HexIntegerLiteral : IntegerLiteral
		{
			/// <summary>
			/// 
			/// </summary>
			/// <param name="source"></param>
			/// <param name="digits"></param>
			/// <param name="isLong"></param>
			/// <param name="unsigned"></param>
			public HexIntegerLiteral(
					string source, string digits, bool isLong, bool unsigned)
					: base(source, digits, isLong, unsigned) {}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			public override string ToString()
			{
				return "HexIntegerLiteral [0x" + this.Digits + "]"
						+ (this.Unsigned ? " unsigned" : string.Empty)
						+ (this.IsLong ? " long" : string.Empty);
			}

			/// <summary>
			/// 
			/// </summary>
			/// <returns></returns>
			protected override decimal GetValueAsDecimal()
			{
				return decimal.Parse(this.Digits, NumberStyles.AllowHexSpecifier);
			}
		}
	}
}
