﻿using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;

using NUnit.Framework;

using SystemNeo.Text.RegularExpressions;

namespace SystemNeoTest.Text.RegularExpressions.RegexUtilTest
{
	[TestFixture]
	public class Format
	{
		[Test]
		public void Normal()
		{
			// 準備
			Match m = Regex.Match("2012/3/12", @"(?<year>\d+)/(?<month>\d+)/(?<day>\d+)");
			Assert.AreEqual("2012", m.Groups["year" ].Value);
			Assert.AreEqual("3",    m.Groups["month"].Value);
			Assert.AreEqual("12",   m.Groups["day"  ].Value);

			// 実行
			string result = RegexUtil.Format(m, "{year}-{month}-{day}");

			// 検証
			Assert.AreEqual("2012-3-12", result);
		}
	}

	[TestFixture]
	public class Get
	{
		[Test]
		public void Normal()
		{
			// 準備
			Match m = Regex.Match("2012/3/12", @"(?<year>\d+)/(?<month>\d+)/(?<day>\d+)");

			// 実行
			string result = RegexUtil.Get(m, "year");

			// 検証
			Assert.AreEqual("2012", result);
		}
	}

	[TestFixture]
	public class GetAsInt
	{
		[Test]
		public void Normal()
		{
			// 準備
			Match m = Regex.Match("2012/3/12", @"(?<year>\d+)/(?<month>\d+)/(?<day>\d+)");

			// 実行
			int result = RegexUtil.GetAsInt(m, "year");

			// 検証
			Assert.AreEqual(2012, result);
		}
	}

	[TestFixture]
	public class GetGroupNamesInPattern
	{
		[Test]
		public void Normal()
		{
			// 実行
			IEnumerable<string> result = RegexUtil.GetGroupNamesInPattern(@"(?<year>\d+)/(?<month>\d+)/(?<day>\d+)");

			// 検証
			Assert.AreEqual(new string[] {"year", "month", "day"}, result);
		}
	}

	[TestFixture]
	public class Matches_Regex_IEnumerableString
	{
		[Test]
		public void Normal()
		{
			// 準備
			string[] inputs = {"Date:", "2012", "/3/", "12", "(Mon)"};

			// 実行
			IEnumerable<Match> result = RegexUtil.Matches(new Regex(@"\d+"), inputs);

			// 検証
			string[] expected = {"2012", "3", "12"};
			int i = 0;
			foreach (Match m in result) {
				Assert.AreEqual(expected[i++], m.Value);
			}
			Assert.AreEqual(expected.Length, i);
		}
	}

	[TestFixture]
	public class Matches_Regex_FuncMatchT
	{
		[Test]
		public void Normal()
		{
			// 準備
			Func<Match, long> func = (m) => long.Parse(m.Value);

			// 実行
			IEnumerable<long> result = RegexUtil.Matches(new Regex(@"\d+"), "2012/3/12", func);

			// 検証
			Assert.AreEqual(new long[] {2012, 3, 12}, result);
		}
	}

	[TestFixture]
	public class MatchValues_Regex_String_Int32
	{
		[Test]
		public void Normal()
		{
			// 検証
			Regex re = new Regex(@"(\d+)/(\d+)/(\d+)");
			string input = "2012/4/12～2012/10/11";

			// 実行
			IEnumerable<string> result = RegexUtil.MatchValues(re, input, 2);

			// 検証
			Assert.AreEqual(new string[] {"4", "10"}, result);
		}
	}

	[TestFixture]
	public class MatchValues_Regex_String_String
	{
		[Test]
		public void Normal()
		{
			// 検証
			Regex re = new Regex(@"(?<year>\d+)/(?<month>\d+)/(?<day>\d+)");
			string input = "2012/4/12～2012/10/11";

			// 実行
			IEnumerable<string> result = RegexUtil.MatchValues(re, input, "day");

			// 検証
			Assert.AreEqual(new string[] {"12", "11"}, result);
		}
	}

	[TestFixture]
	public class ToDictionary
	{
		[Test]
		public void Normal()
		{
			// 準備
			Match m = Regex.Match("2012/3/12", @"(?<year>\d+)/(?<month>\d+)/(?<day>\d+)");
			Assert.AreEqual("2012", m.Groups["year" ].Value);
			Assert.AreEqual("3",    m.Groups["month"].Value);
			Assert.AreEqual("12",   m.Groups["day"  ].Value);

			// 実行
			GroupDictionary result = RegexUtil.ToDictionary(m);

			// 検証
			var expected = new Dictionary<string, string> {{"year", "2012"}, {"month", "3"}, {"day", "12"}};
			foreach (var pair in expected) {
			    Assert.AreEqual(pair.Value, result[pair.Key]);
			}
		}
	}
}
