unit ColorSettingFrame;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms, 
  Dialogs, StdCtrls, ExtCtrls, Contnrs, Menus;

type
  TfrmColorSetting = class(TFrame)
    lstElement: TListBox;
    cbxColor: TColorBox;
    lblTheme: TLabel;
    btnSave: TButton;
    mnThemes: TPopupMenu;
    btnTheme: TButton;
    procedure lstElementDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure lstElementClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
    procedure btnThemeMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure cbxColorChange(Sender: TObject);
  private
    FCurrentTheme: TStringList;
    FThemeList: TStringList;
    function Token(const Str: String; const Delimiter: char;
      const Index: integer): String;
    function SelectedColor: TColor;
    function ColorIndex(Index: integer): TColor;
    function GetColor(Key: String): TColor;
    procedure SetColor(Key: String; const Value: TColor);
    procedure mnThemeItemClick(Sender: TObject);
  protected
    procedure Loaded; override;
    procedure UpdateThemeList;
  public
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
    property Color[Key: String]: TColor read GetColor write SetColor;
  end;

implementation

{$R *.dfm}

{ TfrmColorSetting }

procedure TfrmColorSetting.lstElementDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var ItemName: String;
begin
  ItemName := Token(lstElement.Items[Index], '|', 0);
  with lstElement.Canvas do begin
    TextRect(Rect, Rect.Left+lstElement.ItemHeight, Rect.Top+2, ItemName);
    Pen.Color := clBlack;
    Brush.Color := ColorIndex(Index); 
    Rectangle(Rect.Left+2, Rect.Top+2, Rect.Left+lstElement.ItemHeight-4, Rect.Bottom-2);
  end;
end;

function TfrmColorSetting.Token(const Str: String; const Delimiter: char;
  const Index: integer): String;
var i, c, len: integer;
begin
  i := 1;
  c := 0;
  len := length(Str);
  Result := '';
  while i <= len do begin
    if Str[i] = Delimiter then begin
      Inc(c);
      if c > Index then Break;
    end else if c = Index then Result := Result + Str[i];
    Inc(i);
  end;
end;

constructor TfrmColorSetting.Create(Owner: TComponent);
begin
  inherited;
  FCurrentTheme := TStringList.Create;
  FThemeList := TStringList.Create;
  UpdateThemeList;
end;

destructor TfrmColorSetting.Destroy;
begin
  inherited;
  FCurrentTheme.Free;
  FThemeList.Free;
end;

procedure TfrmColorSetting.Loaded;
begin
  inherited;
end;

procedure TfrmColorSetting.lstElementClick(Sender: TObject);
begin
  if lstElement.ItemIndex = -1 then begin
    cbxColor.Enabled := false;
  end else begin
    cbxColor.Enabled := true;
    cbxColor.Selected := SelectedColor;
    cbxColor.Invalidate;
  end;
end;

function TfrmColorSetting.SelectedColor: TColor;
begin
  Result := ColorIndex(lstElement.ItemIndex);
end;

function TfrmColorSetting.ColorIndex(Index: integer): TColor;
var Key: String;
begin
  Key := Token(lstElement.Items[Index], '|', 1);
  if FCurrentTheme.Values[Key] <> '' then
    Result := StringToColor(FCurrentTheme.Values[Key])
  else
    Result := StringToColor(Token(lstElement.Items[Index], '|', 2));
end;

function TfrmColorSetting.GetColor(Key: String): TColor;
var i: integer;
begin
  Result := clBlack;
  if FCurrentTheme.Values[Key] <> '' then
    Result := StringToColor(FCurrentTheme.Values[Key])
  else begin
    for i := 0 to lstElement.Items.Count-1 do
      if Token(lstElement.Items[i], '|', 1) = Key then begin
        Result := StringToColor(Token(lstElement.Items[i], '|', 2));
        Exit;
      end;
  end;
end;

procedure TfrmColorSetting.SetColor(Key: String; const Value: TColor);
begin
  FCurrentTheme.Values[Key] := ColorToString(Value);
end;

procedure TfrmColorSetting.btnSaveClick(Sender: TObject);
var Title: String;
    Path: String;
begin
  Title := 'Untitled';
  if not InputQuery('^Cg', 'e[}̖O', Title) then Exit;
  Path := ExtractFilePath(Application.ExeName) + '\themes\';
  if FileExists(Path + Title + '.txt') then begin
    if MessageDlg('"' + Title + '"㏑܂?', mtConfirmation, mbOkCancel, 0) = mrCancel then Exit;
  end;
  // Vۑꍇ́AVe[}̃CX^X쐬
  FCurrentTheme.SaveToFile(Path + Title + '.txt');
  UpdateThemeList;
end;

procedure TfrmColorSetting.UpdateThemeList;
var i, tag: integer;
    F: TSearchRec;
    Item: TMenuItem;
    path: String;
begin
  mnThemes.Items.Clear;
  FThemeList.Clear;
  tag := 0;
  path := ExtractFilePath(Application.ExeName);
  i := FindFirst(path + 'themes\*.txt', 0, F);
  if i = 0 then begin
    repeat
      Item := TMenuItem.Create(mnThemes);
      Item.Caption := ChangeFileExt(ExtractFileName(F.Name), '');
      Item.Tag := tag;
      FThemeList.Add(ChangeFileExt(ExtractFileName(F.Name), ''));
      Item.OnClick := mnThemeItemClick;
      mnThemes.Items.Add(Item);
      Inc(tag);
      i := FindNext(F);
    until i <> 0;
  end;
  FindClose(F);
end;

procedure TfrmColorSetting.mnThemeItemClick(Sender: TObject);
var Title, path: String;
begin
  Title := FThemeList[(Sender as TMenuItem).Tag];
  path := ExtractFilePath(Application.ExeName);
  FCurrentTheme.LoadFromFile(path + 'themes\' + Title + '.txt');
  lstElement.Invalidate;
  lstElement.ItemIndex := -1;
  lstElementClick(self);
end;

procedure TfrmColorSetting.btnThemeMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var Pos: TPoint;
begin
  if Button <> mbLeft then Exit;
  if mnThemes.Items.Count = 0 then Exit;
  Pos := btnTheme.ClientToScreen(Point(0, btnTheme.Height));
  mnThemes.Popup(Pos.X, Pos.Y);
end;

procedure TfrmColorSetting.cbxColorChange(Sender: TObject);
begin
  if lstElement.ItemIndex < 0 then Exit;
  lstElement.Invalidate;
  FCurrentTheme.Values[Token(lstElement.Items[lstElement.ItemIndex], '|', 1)] := ColorToString(cbxColor.Selected);
end;

end.
