{*******************************************************}
{       IdSLPP20 - Indy Client for SLPP Connection      }
{                                                       }
{       Copyright (c) 2002-2003 naruto/CANO-Lab         }
{*******************************************************}

unit IdSLPP20;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  IdTCPClient, IdCoderMIME, IdGlobal, IdException;

const
  SLPP_PORT = 9871;
  SLPP_HOST = 'bottle.mikage.to';

type
  TIdSLPPEventType = (etConnectOk, etScript, etMemberCount, etChannelCount,
                    etChannelList, etUnicast, etForceBroadcast, etCloseChannel,
                    etForceBroadcastInformation, etBroadcastInformation);
  TIdSLPPEvent = procedure (Sender: TObject;
    EventType: TIdSlppEventType; const Param: String) of Object;

  TIdSLPP20 = class;

  TIdSLPP20ReadThread = class(TThread)
  private
  protected
    FClient: TIdSLPP20;
    FRecvData: TStringList;
    FEvent: TIdSLPPEventType; // SLPP Command
    FParam: String;           // SLPP Command Parameter
    FReceivedLog: TStringList;
    function Parse: boolean;
    procedure Execute; override;
  public
    constructor Create(AClient: TIdSLPP20); reintroduce;
    property  Client: TIdSLPP20 read FClient;
  end;

  TIdSLPP20 = class(TIdTCPClient)
  private
    FSLPPThread: TIdSLPP20ReadThread;
    FDebugMode: boolean;
    FProxyMode: boolean;
    FProxyUser: String;
    FProxyPass: String;
    FLUID: String;
    FOnSlppEvent: TIdSlppEvent;
    FOnConnect: TNotifyEvent;
    FOnDisconnect: TNotifyEvent;
    FLastReadTime: Int64;
    FTimeout: Integer;
    FOnConnectFailed: TNotifyEvent;
    procedure SetDebugMode(const Value: boolean);
    procedure SetLUID(const Value: String);
    procedure SetOnSlppEvent(const Value: TIdSlppEvent);
    procedure SetProxyMode(const Value: boolean);
    procedure SetProxyUser(const Value: String);
    procedure SetProxyPass(const Value: String);
    procedure SetOnConnect(const Value: TNotifyEvent);
    procedure SetOnDisconnect(const Value: TNotifyEvent);
    function GetLastReadTimeInterval: integer;
    procedure SetLastReadTime(const Value: Int64);
    procedure SetOnConnectFailed(const Value: TNotifyEvent);
  public
    constructor Create(AOwner: TComponent); override;
    procedure ConnectServer(const ATimeout: Integer = IdTimeoutDefault);
    procedure Disconnect; override;
    procedure DoOnSlppEvent;
    procedure DoOnConnect;
    procedure DoOnConnectFailed;
    procedure DoOnDisconnect;
    property SLPP20ReadThread: TIdSLPP20ReadThread read FSLPPThread;
    property LastReadTime: Int64 read FLastReadTime write SetLastReadTime;
    property LastReadTimeInterval: integer read GetLastReadTimeInterval;
  published
    property LUID: String read FLUID write SetLUID;
    property Port default SLPP_PORT;
    property DebugMode: boolean read FDebugMode write SetDebugMode;
    property ProxyMode: boolean read FProxyMode write SetProxyMode;
    property ProxyUser: String read FProxyUser write SetProxyUser;
    property ProxyPass: String read FProxyPass write SetProxyPass;
    property OnConnect: TNotifyEvent read FOnConnect write SetOnConnect;
    property OnConnectFailed: TNotifyEvent read FOnConnectFailed write SetOnConnectFailed;
    property OnDisconnect: TNotifyEvent read FOnDisconnect write SetOnDisconnect;
    property OnSLPPEvent: TIdSlppEvent read FOnSlppEvent write SetOnSlppEvent;
    {ProxyMode = truêƂ́AHost, PortɃvLV}
  end;

  EIdSlppError = class(EIdException);
  EIdSlppClientConnectError = class(EIdSlppError);

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Indy Clients', [TIdSLPP20]);
end;

{ TIdSLPP20 }

// Do NOT call TIdSLPP20.Connect outside of this unit.
procedure TIdSLPP20.ConnectServer;
begin
  try
    FTimeout := ATimeout;
    FSLPPThread := TIdSLPP20ReadThread.Create(self);
  except on E: EIdSocketError do
    raise EIdSlppClientConnectError.Create('Connection Failed');
  end;
end;

constructor TIdSLPP20.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Port := SLPP_PORT;
  Host := SLPP_HOST;
  ProxyUser := '';
  ProxyPass := '';
end;

procedure TIdSLPP20.Disconnect;
begin
  inherited Disconnect;
  if Assigned(FSLPPThread) then begin
    // The thread will be destroyed eventually
    FSLPPThread.Terminate;
    FSLPPThread := nil;
  end;
  DoOnDisconnect;
end;

procedure TIdSLPP20.DoOnConnect;
begin
  if Assigned(FOnConnect) then
    FOnConnect(self);
end;

procedure TIdSLPP20.DoOnConnectFailed;
begin
  if Assigned(FOnConnectFailed) then
    FOnConnectFailed(self);
end;

procedure TIdSLPP20.DoOnSlppEvent;
begin
  try
    FOnSlppEvent(self, FSLPPThread.FEvent, FSLPPThread.FParam);
  except
    on E: Exception do
      ShowMessage(Format('Exception occurred in OnSlppEvent: %s'#13#10 +
        'Event type: %d'#13#10'%s',
        [E.Message, Ord(FSLPPThread.FEvent), FSlppThread.Fparam]));
  end;
end;

function TIdSLPP20.GetLastReadTimeInterval: integer;
begin
  Result := 0;
  if Connected then Result := GetTickCount - FLastReadTime;
end;

procedure TIdSLPP20.SetDebugMode(const Value: boolean);
begin
  FDebugMode := Value;
end;

procedure TIdSLPP20.SetLastReadTime(const Value: Int64);
begin
  FLastReadTime := Value;
end;

procedure TIdSLPP20.SetLUID(const Value: String);
begin
  FLUID := Value;
end;

procedure TIdSLPP20.SetOnConnect(const Value: TNotifyEvent);
begin
  FOnConnect := Value;
end;

procedure TIdSLPP20.SetOnConnectFailed(const Value: TNotifyEvent);
begin
  FOnConnectFailed := Value;
end;

procedure TIdSLPP20.SetOnDisconnect(const Value: TNotifyEvent);
begin
  FOnDisconnect := Value;
end;

procedure TIdSLPP20.SetOnSlppEvent(const Value: TIdSlppEvent);
begin
  FOnSlppEvent := Value;
end;

procedure TIdSLPP20.SetProxyMode(const Value: boolean);
begin
  FProxyMode := Value;
end;

procedure TIdSLPP20.SetProxyUser(const Value: String);
begin
  FProxyUser := Value;
end;

procedure TIdSLPP20.SetProxyPass(const Value: String);
begin
  FProxyPass := Value;
end;

procedure TIdSLPP20.DoOnDisconnect;
begin
  if Assigned(FOnDisconnect) then
    FOnDisconnect(Self);
end;

{ TIdSLPP20ReadThread }

constructor TIdSLPP20ReadThread.Create(AClient: TIdSLPP20);
begin
  inherited Create(true);
  FClient := AClient;
  FreeOnTerminate := true;
  Resume;
end;

procedure TIdSLPP20ReadThread.Execute;
var Line: String;
    EncodedPassword, PlainPassword: String;
    Base64Encoder: TIdEncoderMIME;
begin
  try
    FClient.Connect(FClient.FTimeout);
    if Assigned(FClient.OnConnect) then begin
      Synchronize(FClient.DoOnConnect);
    end;
  except
    Synchronize(FClient.DoOnConnectFailed);
    Exit;
  end;

  EncodedPassword := '';
  if FClient.ProxyUser <> '' then begin
    if FClient.ProxyPass <> '' then begin
      PlainPassword := FClient.ProxyUser + ':' + FClient.ProxyPass;
      Base64Encoder := TIdEncoderMIME.Create(nil);
      try
        EncodedPassword := Base64Encoder.Encode(PlainPassword);
      finally
        Base64Encoder.Free;
      end;
    end;
  end;

  FRecvData := TStringList.Create;
  FReceivedLog := TStringList.Create;
  if FClient.ProxyMode then begin
    FClient.Writeln('POST http://bottle.mikage.to:9871/ HTTP/1.0');
    FClient.Writeln('Content-Length: ' + IntToStr(Length(FClient.LUID)));
    FClient.Writeln('Connection: close');

    if EncodedPassword <> '' then begin
      FClient.Writeln('Proxy-Authorization: Basic ' + EncodedPassword);
    end;

    FClient.Writeln;
    FClient.Writeln(FClient.LUID);
  end else begin
    FClient.WriteLn('POST / HTTP/1.0');
    FClient.WriteLn;
    FClient.WriteLn(FClient.LUID);
  end;
  while not Terminated do begin
    try
      FClient.CheckForDisconnect;
      Line := FClient.ReadLn(EOL);
      if FClient.DebugMode then begin
        FReceivedLog.Add(Line);
        FReceivedLog.SaveToFile(ExtractFilePath(Application.ExeName)+'slpp20_debug.log');
      end;
      if not FClient.ReadLnTimedOut then FClient.LastReadTime := getTickCount; 
      if Length(Line) = 0 then begin  // sĂ
        if FRecvData.Count > 0 then begin  // ͂Ăꍇ
          FClient.CheckForDisconnect; //ؒf̒r[ȃf[^Mh
          if Parse and not Terminated then
            Synchronize(FClient.DoOnSlppEvent);
          FRecvData.Clear;
        end;
      end else begin
        FRecvData.Add(Line);
      end;
    except
      on EIdException do begin
        Synchronize(self.Terminate);
      end;
    end;
  end;

  if FClient.Connected then
    FClient.Disconnect;
  FreeAndNil(FReceivedLog);
  FreeAndNil(FRecvData);
end;

function TIdSLPP20ReadThread.Parse: boolean;
var
  command: String;
begin
  //ŒR}hs+1͂Ȃƃ_ - 2ȏ
  if FRecvData.Count <= 1 then begin
    Result := false;
    Exit;
  end;

  command := FRecvData[0];
  FRecvData.Delete(0);
  FParam := FRecvData.Text;

  //ɂG[`FbNĂ݂
  if Length(FParam) = 0 then begin
    Result := false;
    Exit;
  end;

  Result := true;
  if command = 'broadcastMessage' then begin
    FEvent := etScript;
  end else if command = 'allUsers' then begin
    FEvent := etMemberCount;
  end else if command = 'channelUsers' then begin
    FEvent := etChannelCount;
  end else if command = 'channelList' then begin
    FEvent := etChannelList;
  end else if (command = 'HTTP/1.0 200 OK') or (command = 'HTTP/1.1 200 OK') then begin
    FEvent := etConnectOk;
  end else if command = 'forceBroadcastMessage' then begin
    FEvent := etForceBroadcast;
  end else if command = 'forceBroadcastInformation' then begin
    FEvent := etForceBroadcastInformation;
  end else if command = 'BroadcastInformation' then begin
    FEvent := etBroadcastInformation;
  end else if command = 'closeChannel' then begin
    FEvent := etCloseChannel;
  end else if command = 'unicastMessage' then begin
    FEvent := etUnicast;
  end else Result := false;
end;

end.
