{********************************************************}
{ TSsParser Component - Parser for Sakura Script         }
{                                                        }
{       Copyright (c) 2001-2003 naruto/CANO-Lab          }
{                 (c) 2001-2005 WinBottle Project        }
{********************************************************}

unit SsParser;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Contnrs;

type
  // Kind of the Markup
  // mtTag:  Tag. (begins with \)
  // mtMeta: Meta expression. (begins with %)
  // mtTagErr: Seems to be a markup error
  // mtStr: Other normal talk string
  TSsMarkUpType = (mtTag, mtMeta, mtTagErr, mtStr);

  // Intercepting the parsing
  TSsParseEvent = procedure (Sender: TObject; const Script: string;
    var Len: integer; var MarkType: TSsMarkUpType; var Extra: string) of object;

  // Exception class
  ESsParserError = class(Exception);

  // Internally used private class
  // Do not use this class outside this unit!
  TSsMarkUp = class(TPersistent)
  private
    FPos: Integer;
    FExtra: string;
    FMarkUpType: TSsMarkUpType;
    FStr: string;
  public
    procedure Assign(Source: TPersistent); override;
    constructor Create(Pos: Integer; MarkUpType: TSsMarkUpType; Str: string;
      Extra: string = '');
  published
    property Pos: Integer read FPos; //This markup is Pos'th character in InputString
    property MarkUpType: TSsMarkUpType read FMarkUpType;
    property Str: string read FStr;
    property Extra: string read FExtra write FExtra;
  end;

   // TSsParseResult class
  TSsParseResult = class(TObject)
  private
    FInputString: string;
    FList: TObjectList;
    function GetCount: Integer;
    function GetExtra(Index: Integer): string;
    function GetMarkUpType(Index: Integer): TSsMarkUpType;
    function GetPosition(Index: Integer): Integer;
    function GetStr(Index: Integer): string;
    procedure SetExtra(Index: Integer; const Value: string);
    function GetItem(Index: Integer): TSsMarkUp;
    procedure Clear;
    procedure Add(Pos: Integer; MarkUpType: TSsMarkUpType; Str: string;
      Extra: string = '');
  protected
    property Item[Index: Integer]: TSsMarkUp read GetItem;
  public
    constructor Create(const InputString: string);
    destructor Destroy; override;
    function MarkUpAt(const Pos: Integer): Integer;
    property Count: Integer read GetCount;
    property MarkUpType[Index: Integer]: TSsMarkUpType read GetMarkUpType;
    property Str[Index: Integer]: string read GetStr; default;
    property Extra[Index: Integer]: string read GetExtra write SetExtra;
    property Position[Index: Integer]: Integer read GetPosition;
    property InputString: string read FInputString; // Cannot Write
  end;

  // Notes to user:
  // This class has been modified so that it can handle multiple parsing
  // results, but this class is NOT thread-safe.
  // If you want to parse scripts within more than one thread,
  // separate parser instance should be prepared for each thread.
  TSsParser = class(TComponent)
  private
    FTagPattern: TStrings;  //SakuraScript tag pattern string
    FMetaPattern: TStrings; //SakuraScript meta expression pattern string
    FBuffList: TSsParseResult; //To implement deprecated properties
    FLeaveEscape: boolean;
    FEscapeInvalidMeta: boolean;
    FOnParse: TSsParseEvent;
    FPatternStudied: Boolean; // Whether pattern definition is optimized
    FTagPatternTable: TStrings;
    FTagPatternPointer: array[Char] of Integer;
    FMetaPatternTable: TStrings;
    FMetaPatternPointer: array[Char] of Integer;
    procedure SetInputString(const Value: string);
    function GetCount: integer;
    function GetExtra(Index: integer): string;
    function GetMarkUpType(Index: integer): TSsMarkUpType;
    function GetStr(Index: integer): string;
    procedure SetExtra(Index: integer; const Value: string);
    procedure SetMetaPattern(const Value: TStrings);
    procedure SetTagPattern(const Value: TStrings);
    procedure SetOnParse(const Value: TSsParseEvent);
    function GetFirstChar(const Str: string): string;
    function GetPosition(Index: integer): integer;
    function GetInputString: string;
  protected
    function ChopFirstChar(var Str: string): string;
    procedure BeginParse(ResultList: TSsParseResult);
    procedure Study;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function MatchP(PStr, PPattern: PChar): integer;
    function Match(Str, Pattern: string): integer;
    function GetParam(Tag: string; const Index: integer): string;
    function EscapeParam(const Param: string): string;
    function MarkUpAt(const Pos: integer): integer;

    function Parse(const Script: string): TSsParseResult;

    { These properties are deprecated. }
    property Count: integer read GetCount;
    property MarkUpType[Index: integer]: TSsMarkUpType read GetMarkUpType;
    property Str[Index: integer]: string read GetStr; default;
    property Extra[Index: integer]: string read GetExtra write SetExtra;
    property Position[Index: integer]: integer read GetPosition;

    property InputString: string read GetInputString write SetInputString;
  published
    // Script parsing patterns.
    property TagPattern: TStrings read FTagPattern write SetTagPattern;
    property MetaPattern: TStrings read FMetaPattern write SetMetaPattern;
    // Whether to leave escape sequence "\\" and "\%" in mtStr elements
    property LeaveEscape: boolean read FLeaveEscape write FLeaveEscape
      default true;
    // Replace '%' to '\%' if sequence follwing the '%' could not
    // be parsed as a meta expression 
    property EscapeInvalidMeta: boolean read FEscapeInvalidMeta
      write FEscapeInvalidMeta default false;
    // Component users can intercept and handle part of the parsing using this event.
    property OnParse: TSsParseEvent read FOnParse write SetOnParse;
  end;

procedure Register;

implementation

resourcestring
  CElementTypeError = 'Gg^CvԂĂ';
  CTagPatternSyntaxError = 'TagPattern\L~X %d s';
  CMarkupIndexOutofRange = 'CfbNX͈͂Ă܂';

{
  // English error message
  CElementTypeError = 'Returned element type is invalid in OnParse.';
  CTagPatternSyntaxError = 'TagPattern syntax error at line %d.';
  CMarkupIndexOutofRange = 'Markup index if out of range.';
}


procedure Register;
begin
  RegisterComponents('Samples', [TSsParser]);
end;

{ TSsParser }

procedure TSsParser.BeginParse(ResultList: TSsParseResult);
var Str, Talk, T, Ex: string;
    i, Le: integer;
    IsErr: boolean;
    Mt: TSsMarkUpType;
    PHead, PStr, PTalk, PTail, Buf: PChar;
begin
  // Pattern study must be done the first time you parse.
  if not FPatternStudied then
    Study;
  ResultList.Clear;
  Str := ResultList.InputString; // The string to be parsed from now
  // This is to avoid access violation if `Str` is terminated
  // with DBCS leadbyte. (Such string is invalid from the beginning of course) 
  Str := Str + #0#0;
  if Length(Str) = 0 then
    Exit;
  PHead := PChar(Str); // const
  PTail := PChar(Str);
  PStr := PChar(Str);
  // Length(Talk) <= Length(Str) * 2
  // Because a script like '%%%' is converted to '\%\%\%' if EscapeInvalidMeta is on
  SetLength(Talk, Length(Str) * 2 + 2);
  PTalk := PChar(Talk);
  while PStr^ <> #0 do begin
    if PStr^ = '\' then begin
      Inc(PStr);
      if PStr^ = '\' then
      begin
        // Escaped sequence "\\"
        if FLeaveEscape then
        begin
          PTalk^ := '\';
          Inc(PTalk);
          PTalk^ := '\';
          Inc(PTalk);
        end else
        begin
          PTalk^ := '\';
          Inc(PTalk);
        end;
        Inc(PStr);
        Continue;
      end else if PStr^ = '%' then
      begin
        // Escaped sequence "\%"
        if FLeaveEscape then
        begin
          PTalk^ := '\';
          Inc(PTalk);
          PTalk^ := '%';
          Inc(PTalk);
        end else
        begin
          PTalk^ := '%';
          Inc(PTalk);
        end;
        Inc(PStr);
        Continue;
      end else
      begin
        Dec(PStr);
        // might be a tag
        // Generate OnParser event
        Le := 0;
        Ex := '';
        IsErr := false;
        if Assigned(FOnParse) then begin
          FOnParse(Self, string(PStr), Le, Mt, Ex);
          if Le > 0 then begin
            if (Mt <> mtTag) and (Mt <> mtTagErr) then begin
              raise ESsParserError.Create(CElementTypeError);
              Exit;
            end;
            if Mt = mtTagErr then IsErr := true;
          end;
        end;
        if Le <= 0 then
        begin
          Inc(PStr);
          i := FTagPatternPointer[PStr^];
          if i > -1 then
            while (i < FTagPatternTable.Count) do begin
              T := FTagPatternTable[i];
              IsErr := T[1] = '!';
              Buf := PChar(T);
              Inc(Buf);
              Le := MatchP(PStr, Buf);
              if Le > 0 then
              begin
                Inc(Le);
                Dec(PStr);
                Break;
              end;
              if T[2] <> PStr^ then
                Break;
              Inc(i);
            end;
        end;
        if PChar(Talk) <> PTalk then begin
          PTalk^ := #0;
          ResultList.Add(PTail-PHead+1, mtStr, string(PChar(Talk)));
          PTalk := PChar(Talk);
        end;
        if Le > 0 then begin
          // Parsed as a correct tag
          T := Copy(string(PStr), 1, Le);
          if IsErr then
            ResultList.Add(PStr-PHead+1, mtTagErr, T, Ex)
          else
            ResultList.Add(PStr-PHead+1, mtTag, T, Ex);
          Inc(PStr, Le);
        end else begin
          // Parsing failed; The character '\' and the next one character is
          // marked as a tag error.
          Inc(PStr); // Skip '\'
          if PStr^ in LeadBytes then
          begin
            T := '\' + Copy(string(PStr), 1, 2);
            ResultList.Add(PStr-PChar(Str), mtTagErr, T);
            Inc(PStr, 2);
          end else
          begin
            T := '\' + PStr^;
            ResultList.Add(PStr-PChar(Str), mtTagErr, T);
            Inc(PStr);
          end;
        end;
        PTail := PStr;
      end;
    end else if (PStr^ = '%') then begin
      Le := 0; Ex := '';
      if Assigned(FOnParse) then begin
        FOnParse(Self, string(PStr), Le, Mt, Ex);
        if Le > 0 then begin
          if Mt <> mtMeta then begin
            raise ESsParserError.Create(CElementTypeError);
            Exit;
          end;
          Dec(Le);
        end;
      end;
      Inc(PStr); // Skip '%'
      if Le <= 0 then
      begin
        i := FMetaPatternPointer[PStr^];
        if i > -1 then
          while i < FMetaPatternTable.Count do
          begin
            Le := MatchP(PStr, PChar(FMetaPatternTable[i]));
            if Le > 0 then
              Break;
            if FMetaPatternTable[i][1] <> PStr^ then
              Break;
            Inc(i);
          end;
      end;
      if Le > 0 then // Matched as valid meta string
      begin
        if PChar(Talk) <> PTalk then
        begin
          PTalk^ := #0;
          ResultList.Add(PTail-PHead+1, mtStr, string(PChar(Talk)));
          PTalk := PChar(Talk);
        end;
        T := Copy(string(PStr), 1, Le);
        ResultList.Add(PStr-PChar(Str), mtMeta, '%' + T, Ex);
        Inc(PStr, Le);
        PTail := PStr;
      end else
      begin
        // in case this %??? sequence seems NOT to be a meta expression
        if FEscapeInvalidMeta then
        begin
          PTalk^ := '\';
          Inc(PTalk);
          PTalk^ := '%';
          Inc(PTalk);
        end else
        begin
          PTalk^ := '%';
          Inc(PTalk);
        end;
        Continue;
      end;
    end else begin
      if PStr^ in LeadBytes then
      begin
        PTalk^ := PStr^;
        Inc(PStr); Inc(PTalk);
        PTalk^ := PStr^;
        Inc(PStr); Inc(PTalk);
      end else
      begin
        PTalk^ := PStr^;
        Inc(PStr); Inc(PTalk);
      end;
    end;
  end; // of while
  if PTalk <> PChar(Talk) then
  begin
    PTalk^ := #0;
    ResultList.Add(PTail-PHead+1, mtStr, string(PChar(Talk)));
  end;
end;

function TSsParser.ChopFirstChar(var Str: string): string;
begin
  Result := GetFirstChar(Str);
  Delete(Str, 1, Length(Result));
end;

constructor TSsParser.Create(AOwner: TComponent);
begin
  inherited;
  FTagPattern  := TStringList.Create;
  FMetaPattern := TStringList.Create;
  FTagPatternTable  := TStringList.Create;
  FMetaPatternTable := TStringList.Create;
  FLeaveEscape := true;
end;

destructor TSsParser.Destroy;
begin
  inherited;
  FTagPattern.Free;
  FMetaPattern.Free;
  FTagPatternTable.Free;
  FMetaPatternTable.Free;
  if FBuffList <> nil then
    FBuffList.Free;
end;

function TSsParser.EscapeParam(const Param: string): string;
begin
  //StringReplace supports DBCS
  Result := StringReplace(Param, '\', '\\', [rfReplaceAll]);
  Result := StringReplace(Param, ']', '\]', [rfReplaceAll]);
end;

function TSsParser.GetCount: integer;
begin
  if FBuffList <> nil then
    Result := FBuffList.Count
  else
    Result := 0;
end;

function TSsParser.GetExtra(Index: integer): string;
begin
  if (Index >= 0) and (Index < FBuffList.Count) then
    Result := FBuffList.Extra[Index]
  else
    Result := '';
end;

function TSsParser.GetFirstChar(const Str: string): string;
begin
  // Get the first character of the given string. Supports DBCS
  if Length(Str) <= 1 then begin
    Result := Str;
  end else begin
    if Str[1] in LeadBytes then begin
      Result := Str[1] + Str[2];
    end else begin
      Result := Str[1];
    end;
  end;
end;

function TSsParser.GetInputString: string;
begin
  if FBuffList <> nil then
    Result := FBuffList.InputString
  else
    Result := '';
end;

function TSsParser.GetMarkUpType(Index: integer): TSsMarkUpType;
begin
  if (Index >= 0) and (Index <= FBuffList.Count-1) then
    Result := FBuffList.MarkUpType[Index]
  else
    raise ESsParserError.Create(CMarkupIndexOutOfRange);
end;

function TSsParser.GetParam(Tag: string;
  const Index: integer): string;
var ParamCount: integer;
    First, Param: string;
    Escape, Inside: boolean;
begin
  if Index <= 0 then Exit;
  Inside := false;
  ParamCount := 0;
  Escape := false;
  repeat
    First := ChopFirstChar(Tag);
    if Inside then begin
      if Escape then begin
        if First = '\' then Param := Param + '\'
        else if First = ']' then Param := Param + ']'
        else Param := Param + '\' + First;
        Escape := false;
      end else if First = '\' then
        Escape := true
      else if First = ']' then
        Inside := false
      else begin
        Escape := false;
        Param := Param + First;
      end;
    end else if First = '[' then begin
      Inside := true;
      Escape := false;
      Param := '';
      Inc(ParamCount);
    end;
  until (First = '') or ((ParamCount = Index) and not Inside);
  if ((ParamCount = Index) and not Inside) then
    Result := Param
  else
    Result := '';
end;

function TSsParser.GetPosition(Index: integer): integer;
begin
  if (Index >= 0) and (Index < FBuffList.Count) then
    Result := FBuffList.Position[Index]
  else
    Result := 0;
end;

function TSsParser.GetStr(Index: integer): string;
begin
  if (Index >= 0) and (Index < FBuffList.Count) then
    Result := FBUffList[Index]
  else
    Result := '';
end;

function TSsParser.MarkUpAt(const Pos: integer): Integer;
begin
  Result := FBuffList.MarkUpAt(Pos);
end;

function TSsParser.Match(Str, Pattern: string): integer;
begin
  if (Length(Str) = 0) or (Length(Pattern) = 0) then
    Result := 0
  else
    Result := MatchP(@Str[1], @Pattern[1]);
end;

function TSsParser.MatchP(PStr, PPattern: PChar): integer;
var Matched, F, Escape: boolean;
begin
  Matched := true;
  Result := 0;
  while Matched and (PPattern^ <> #0) do begin
    if PPattern^ = '%' then
    begin
      if PStr^ = #0 then
      begin
        Matched := false;
        Break;
      end;
      Inc(PPattern);
      case PPattern^ of
        '%':
          begin
            if PStr^ = '%' then
            begin
              Inc(Result);
              Inc(PStr);
              Inc(PPattern);
            end else
            begin
              Matched := false;
              Break;
            end;
          end;
        'd':
          begin
            if PStr^ in ['0' .. '9'] then
            begin
              Inc(Result);
              Inc(PStr);
              Inc(PPattern);
            end else
              Matched := false;
          end;
        'D':
          begin
            if PStr^ in ['0' .. '9'] then
            begin
              while PStr^ in ['0' .. '9'] do
              begin
                Inc(Result);
                Inc(PStr);
              end;
              Inc(PPattern);
            end else
              Matched := false;
          end;
        'b': //String enclosed by '[' and ']'. The content may be an empty string.
          begin
            if PStr^ <> '[' then
            begin
              Matched := false;
            end else
            begin
              F := false;
              Escape := false; //After escape character
              Inc(PStr);   // '['
              Inc(Result); // '['
              repeat
                if Escape then
                begin
                  Escape := false;
                end else
                begin
                  if PStr^ = '\' then Escape := true;
                  if PStr^ = ']' then F := true;
                end;
                if PStr^ in LeadBytes then
                begin
                  Inc(Result, 2);
                  Inc(PStr, 2);
                end else
                begin
                  Inc(Result);
                  Inc(PStr);
                end;
              until (PStr^ = #0) or F;
              if not F then
                Matched := false;
            end;
            Inc(PPattern);
          end;
        'c': // String which can be the argument content enclosed by '[' and ']'
          begin
            Inc(PPattern);
            if not (PStr^ = ']') then
            begin
              Escape := false;
              repeat
                if Escape then
                  Escape := false
                else if PStr^ = ']' then
                  Break
                else
                  if PStr^ = '\' then Escape := true;
                if PStr^ in LeadBytes then
                begin
                  Inc(Result, 2);
                  Inc(PStr, 2);
                end else
                begin
                  Inc(Result);
                  Inc(PStr);
                end;
              until (PStr^ = #0);
            end else
              Matched := false;
          end;
        'm':
          begin
            if not (PStr^ in LeadBytes) then
            begin
              Inc(PPattern);
              Inc(PStr);
              Inc(Result);
            end else Matched := false;
          end;
        'M':
          begin
            if (PStr^ in LeadBytes) then
            begin
              Inc(PPattern);
              Inc(PStr, 2);
              Inc(Result, 2);
            end else Matched := false;
          end;
        '.':
          if (PStr^ in LeadBytes) then
          begin
            Inc(PPattern);
            Inc(PStr, 2);
            Inc(Result, 2);
          end else
          begin
            Inc(PPattern);
            Inc(PStr);
            Inc(Result);
          end;
        else
          if PStr^ = '%' then
          begin
            Inc(PStr);
            Inc(Result);
          end else
          begin
            Matched := false;
          end;
      end // of case
    end else
    begin
      if PStr^ <> PPattern^ then
        Matched := false
      else
      begin
        Inc(Result);
        Inc(PStr);
        Inc(PPattern);
      end;
    end;
  end; //of while
  if not Matched then Result := 0;
end;

function TSsParser.Parse(const Script: string): TSsParseResult;
begin
  Result := TSsParseResult.Create(Script);
  BeginParse(Result);
end;

procedure TSsParser.SetExtra(Index: integer; const Value: string);
begin
  FBuffList.Extra[Index] := Value;
end;

procedure TSsParser.SetInputString(const Value: string);
begin
  // Warning! This property is deprecated
  // Use Parse() method instead.
  if FBuffList <> nil then
  begin
    FBuffList.FInputString := Value; // Access to private method outside the class!
    FBuffList.Clear
  end else
    FBuffList := TSsParseResult.Create(Value);
  BeginParse(FBuffList);
end;

procedure TSsParser.SetMetaPattern(const Value: TStrings);
begin
  FMetaPattern.Assign(Value);
  FPatternStudied := false;
end;

procedure TSsParser.SetOnParse(const Value: TSsParseEvent);
begin
  FOnParse := Value;
end;

procedure TSsParser.SetTagPattern(const Value: TStrings);
begin
  FTagPattern.Assign(Value);
  FPatternStudied := false;
end;

procedure TSsParser.Study;
var
  i, index, count: Integer;
  c: Char;
begin
  // Pattern studying is done for reducing meaningless pattern matching.
  FTagPatternTable.Clear;
  for c := Low(Char) to High(Char) do
  begin
    count := 0;
    FTagPatternPointer[c] := -1;
    for i := 0 to FTagPattern.Count-1 do
    begin
      if Length(FTagPattern[i]) < 2 then
        Continue; // invalid pattern
      if not (FTagPattern[i][1] in ['\', '!']) then
        raise ESsParserError.CreateFmt(CTagPatternSyntaxError, [i+1]);
      if FTagPattern[i][2] = c then
      begin
        Inc(count);
        index := FTagPatternTable.Add(FTagPattern[i]);
        if count = 1 then
          FTagPatternPointer[c] := index;
      end;
    end;
  end;
  FMetaPatternTable.Clear;
  for c := Low(Char) to High(Char) do
  begin
    count := 0;
    FMetaPatternPointer[c] := -1;
    for i := 0 to FMetaPattern.Count-1 do
    begin
      if Length(FMetaPattern[i]) < 1 then
        Continue; // invalid pattern
      if FMetaPattern[i][1] = c then
      begin
        Inc(count);
        index := FMetaPatternTable.Add(FMetaPattern[i]);
        if count = 1 then
          FMetaPatternPointer[c] := index;
      end;
    end;
  end;
  FPatternStudied := true;
end;

{ TSsMarkUp }

procedure TSsMarkUp.Assign(Source: TPersistent);
var
  Src: TSsMarkUp;
begin
  if Source is TSsMarkUp then
  begin
    Src := Source as TSsMarkUp;
    FPos := Src.FPos;
    FMarkUpType := Src.FMarkUpType;
    FStr := Src.FStr;
    FExtra := Src.FExtra;
  end else
    inherited;
end;

constructor TSsMarkUp.Create(Pos: Integer;
  MarkUpType: TSsMarkUpType; Str: string; Extra: string = '');
begin
  FPos := Pos;
  FMarkUpType := MarkUpType;
  FStr := Str;
  FExtra := Extra;
end;

{ TSsParseResult }

procedure TSsParseResult.Add(Pos: Integer;
  MarkUpType: TSsMarkUpType; Str: string; Extra: string);
begin
  FList.Add(TSsMarkUp.Create(Pos, MarkUpType, Str, Extra));
end;

procedure TSsParseResult.Clear;
begin
  FList.Clear;
end;

constructor TSsParseResult.Create(const InputString: string);
begin
  FList := TObjectList.Create(true);
  FInputString := InputString;
end;

destructor TSsParseResult.Destroy;
begin
  FList.Free;
  inherited;
end;

function TSsParseResult.GetCount: Integer;
begin
  Result := FList.Count;
end;

function TSsParseResult.GetExtra(Index: integer): string;
begin
  try
    Result := Item[Index].Extra;
  except
    on EListError do
    begin
      raise ESsParserError.Create(CMarkupIndexOutofRange);
    end;
  end;
end;

function TSsParseResult.GetItem(Index: Integer): TSsMarkUp;
begin
  Result := FList[Index] as TSsMarkUp;
end;

function TSsParseResult.GetMarkUpType(Index: integer): TSsMarkUpType;
begin
  try
    Result := Item[Index].MarkUpType;
  except
    on EListError do
    begin
      raise ESsParserError.Create(CMarkupIndexOutofRange);
    end;
  end;
end;

function TSsParseResult.GetPosition(Index: integer): integer;
begin
  try
    Result := Item[Index].Pos;
  except
    on EListError do
    begin
      raise ESsParserError.Create(CMarkupIndexOutofRange);
    end;
  end;
end;

function TSsParseResult.GetStr(Index: integer): string;
begin
  try
    Result := Item[Index].Str;
  except
    on EListError do
    begin
      raise ESsParserError.Create(CMarkupIndexOutofRange);
    end;
  end;
end;

function TSsParseResult.MarkUpAt(const Pos: integer): integer;
var i, lo, hi: integer;
begin
  if FList.Count = 0 then
    Result := -1
  else begin
    lo := 0;
    hi := FList.Count-2;
    i := (hi-lo) div 2 + lo;
    while (hi > lo) do
    begin
      i := (hi-lo) div 2 + lo;
      if (Item[i].Pos >= Pos) and
         (Item[i+1].Pos < Pos) then
      begin
         Result := i;
         Exit;
      end else if Item[i].Pos > Pos then
        hi := i
      else
        lo := i;
    end;
    Result := i;
  end;
end;

procedure TSsParseResult.SetExtra(Index: Integer; const Value: string);
begin
  if (Index >= 0) and (Index < FList.Count) then
    Item[Index].Extra := Value
  else
    raise ESsParserError.Create(CMarkupIndexOutOfRange);
end;

end.
