{*************************************************************}
{ TSsPlayTime Component - Estimates SakuraScript playing time }
{                                                             }
{       Copyright (c) 2001-2003 naruto/CANO-Lab               }
{                 (c) 2001-2005 WinBottle Project             }
{*************************************************************}

unit SsPlayTime;

interface

uses
  Windows, SysUtils, Classes, SsParser;

type
  TSsPlayTimeException = class(Exception);
  TSsPlayTimeInitException = class(TSsPlayTimeException);

  TSsPlayTimeSpecialChar = class(TCollectionItem)
  private
    FWait: integer;
    FChar: String;
    procedure SetChar(const Value: String);
    procedure SetWait(const Value: integer);
  protected
    function GetDisplayName: String; override;
  public
    procedure Assign(Source: TPersistent); override;
    function DisplayChar: String;
  published
    property Char: String read FChar write SetChar;
    property Wait: integer read FWait write SetWait;
  end;

  TSsPlayTimeSpecialChars = class(TCollection)
  end;

  TSsPlayTimeParams = class(TComponent)
  private
    FCostWait: integer;
    FCostDBChar: integer;
    FCostChar: integer;
    FCostConst: integer;
    FCostHiResWait: integer;
    FCostSurface: integer;
    FCostQuickChar: integer;
    FProfileName: String;
    FSpecialChars: TSsPlayTimeSpecialChars;
    procedure SetCostChar(const Value: integer);
    procedure SetCostConst(const Value: integer);
    procedure SetCostDBChar(const Value: integer);
    procedure SetCostWait(const Value: integer);
    procedure SetCostHiResWait(const Value: integer);
    procedure SetCostSurface(const Value: integer);
    procedure SetCostQuickChar(const Value: integer);
    procedure SetProfileName(const Value: String);
    procedure SetSpecialChars(const Value: TSsPlayTimeSpecialChars);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
  published
    property ProfileName: String read FProfileName write SetProfileName;
    property CostConst: integer read FCostConst write SetCostConst default 0;
    property CostWait: integer read FCostWait write SetCostWait default 50;
    property CostHiResWait: integer read FCostHiResWait write SetCostHiResWait default 1;
    property CostSurface: integer read FCostSurface write SetCostSurface default 5;
    property CostChar: integer read FCostChar write SetCostChar default 50;
    property CostDBChar: integer read FCostDBChar write SetCostDBChar default 50;
    property CostQuickChar: integer read FCostQuickChar write SetCostQuickChar default 0;
    property SpecialChars: TSsPlayTimeSpecialChars read FSpecialChars write SetSpecialChars;
  end;

  TSsPlayTimeCount = record
    Wait: integer;
    HiResWait: integer;
    Surface: integer;
    Char: integer;
    DBChar: integer;
    QuickChar: integer;
    Specials: integer;
  end;

  TSsPlayTime = class(TComponent)
  private
    FPlayTimeParams: TSsPlayTimeParams;
    FSsParser: TSsParser;
    FCounts: TSsPlayTimeCount;
    procedure SetPlayTimeParams(const Value: TSsPlayTimeParams);
    procedure SetSsParser(const Value: TSsParser);
  protected
    procedure CountElements;
    procedure CountCharacterType(const Str: String; out SB, DB,
      SPNum, SPWait: integer);
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    function PlayTime(const Script: String): integer;
    property Counts: TSsPlayTimeCount read FCounts;
  published
    property PlayTimeParams: TSsPlayTimeParams read FPlayTimeParams write SetPlayTimeParams;
    property SsParser: TSsParser read FSsParser write SetSsParser;
  end;

procedure Register;

implementation

const
  CDBWhiteSpace = #129 + #64; // Shift_JIS DB White Space;

procedure Register;
begin
  RegisterComponents('Samples', [TSsPlayTime, TSsPlayTimeParams]);
end;

{ TSsPlayTimeParams }

procedure TSsPlayTimeParams.Assign(Source: TPersistent);
var
  Src: TSsPlayTimeParams;
begin
  if not (Source is TSsPlayTimeParams) then
    inherited
  else
  begin
    Src := Source as TSsPlayTimeParams;
    FCostConst        := Src.FCostConst;
    FCostWait         := Src.FCostWait;
    FCostHiResWait    := Src.FCostHiResWait;
    FCostSurface      := Src.FCostSurface;
    FCostChar         := Src.FCostChar;
    FCostDBChar       := Src.FCostDBChar;
    FCostQuickChar    := Src.FCostQuickChar;
    FProfileName      := Src.FProfileName;
    FSpecialChars.Assign(Src.SpecialChars);
  end;
end;

constructor TSsPlayTimeParams.Create(AOwner: TComponent);
begin
  inherited;
  FCostConst        := 0;
  FCostWait         := 50;
  FCostHiResWait    := 1;
  FCostSurface      := 5;
  FCostChar         := 50;
  FCostDBChar       := 50;
  FCostQuickChar    := 0;
  FSpecialChars := TSsPlayTimeSpecialChars.Create(TSsPlayTimeSpecialChar);
end;

destructor TSsPlayTimeParams.Destroy;
begin
  FSpecialChars.Free;
  inherited;
end;

procedure TSsPlayTimeParams.SetCostChar(const Value: integer);
begin
  FCostChar := Value;
end;

procedure TSsPlayTimeParams.SetCostConst(const Value: integer);
begin
  FCostConst := Value;
end;

procedure TSsPlayTimeParams.SetCostDBChar(const Value: integer);
begin
  FCostDBChar := Value;
end;

procedure TSsPlayTimeParams.SetCostHiResWait(const Value: integer);
begin
  FCostHiResWait := Value;
end;

procedure TSsPlayTimeParams.SetCostQuickChar(const Value: integer);
begin
  FCostQuickChar := Value;
end;

procedure TSsPlayTimeParams.SetCostSurface(const Value: integer);
begin
  FCostSurface := Value;
end;

procedure TSsPlayTimeParams.SetCostWait(const Value: integer);
begin
  FCostWait := Value;
end;

procedure TSsPlayTimeParams.SetProfileName(const Value: String);
begin
  FProfileName := Value;
end;

procedure TSsPlayTimeParams.SetSpecialChars(
  const Value: TSsPlayTimeSpecialChars);
begin
  FSpecialChars.Assign(Value);
end;

{ TSsPlayTime }

procedure TSsPlayTime.CountCharacterType(const Str: String; out SB,
  DB, SPNum, SPWait: integer);
var
  i, j, lnStr: integer;
  InLeadByte, Flag: boolean;
  AChar: TSsPlayTimeSpecialChar;
begin
  SB := 0;
  DB := 0;
  SPNum := 0;
  SPWait := 0;
  InLeadByte := false;
  lnStr := Length(Str);
  for i := 1 to lnStr do
  begin
    if InLeadByte then
    begin
      Inc(DB);
      InLeadByte := false;
    end
    else
    begin
      Flag := false;
      for j := 0 to PlayTimeParams.SpecialChars.Count-1 do
      begin
        AChar := PlayTimeParams.SpecialChars.Items[j] as TSsPlayTimeSpecialChar;
        if (Length(AChar.Char) = 2) and (lnStr-i > 0) then
        begin
          if Str[i] + Str[i+1] = AChar.Char then
          begin
            Inc(SPWait, AChar.Wait);
            Inc(SPNum);
            Dec(DB);
            Flag := true;
            InLeadByte := true;
            Break;
          end;
        end else
        begin
          if Str[i] = AChar.Char then
          begin
            Inc(SPWait, AChar.Wait);
            Inc(SPNum);
            Flag := true;
            Break;
          end;
        end;
      end;
      if not Flag then
      begin
        if Str[i] in LeadBytes then
          InLeadByte := true
        else
          Inc(SB);
      end;
    end;
  end;
end;

procedure TSsPlayTime.CountElements;
var
  i, SB, DB, SPNum, SPWait: integer;
  Mark: String;
  InQuick: boolean;
begin
  ZeroMemory(@FCounts, sizeof(FCounts));
  InQuick := false;
  with SsParser do
  begin
    for i := 0 to SsParser.Count-1 do
    begin
      Mark := Str[i];
      case MarkUpType[i] of
        mtTag:
          begin
            if Mark = '\_q' then
              InQuick := not InQuick
            else if Match(Mark, '\w%d') = 3 then
              FCounts.Wait := FCounts.Wait + Ord(Mark[3]) - Ord('0')
            else if Match(Mark, '\_w[%D]') > 0 then
              Inc(FCounts.HiResWait, StrToInt(GetParam(Mark, 0)))
            else if Match(Mark, '\s%d') = 3 then
              Inc(FCounts.Surface)
            else if Match(Mark, '\s%b') > 0 then
              Inc(FCounts.Surface)
          end;
        mtMeta, mtStr:
          begin
            if InQuick then
            begin
              CountCharacterType(Mark, SB, DB, SPNum, SPWait);
              Inc(FCounts.QuickChar, SB + DB + SPNum);
            end else
            begin
              CountCharacterType(Mark, SB, DB, SPNum, SPWait);
              Inc(FCounts.Char, SB);
              Inc(FCounts.DBChar, DB);
              Inc(FCounts.Specials, SPWait);
            end;
          end;
        // Ignore all tag errors
      end;
    end;
  end;
end;

procedure TSsPlayTime.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if Operation = opRemove then
  begin
    if AComponent = FPlayTimeParams then
      FPlayTimeParams := nil;
    if AComponent = FSsParser then
      FSsParser := nil;
  end;
end;

function TSsPlayTime.PlayTime(const Script: String): integer;
begin
  Result := 0;
  if FSsParser = nil then
    raise TSsPlayTimeInitException.Create('SsParser is not set');
  if FPlayTimeParams = nil then
    raise TSsPlayTimeInitException.Create('PlayTimeParams is not set');
  SsParser.InputString := Script;
  CountElements;
  with PlayTimeParams do
    Result := CostConst + CostWait * FCounts.Wait + CostSurface * FCounts.Surface +
      CostHiResWait * FCounts.HiResWait + CostChar * FCounts.Char +
      CostDBChar * FCounts.DBChar + FCounts.Specials +
      CostQuickChar * FCounts.QuickChar;
end;

procedure TSsPlayTime.SetPlayTimeParams(const Value: TSsPlayTimeParams);
begin
  FPlayTimeParams := Value;
  if Value <> nil then
    Value.FreeNotification(self);
end;

procedure TSsPlayTime.SetSsParser(const Value: TSsParser);
begin
  FSsParser := Value;
  if Value <> nil then
    Value.FreeNotification(self);
end;

{ TSsPlayTimeSpecialChar }

procedure TSsPlayTimeSpecialChar.Assign(Source: TPersistent);
begin
  if not(Source is TSsPlayTimeSpecialChar) then
    inherited
  else
  begin
    Self.FChar := (Source as TSsPlayTimeSpecialChar).FChar;
    Self.FWait := (Source as TSsPlayTimeSpecialChar).FWait;
  end;
end;

function TSsPlayTimeSpecialChar.DisplayChar: String;
begin
  if FChar = ' ' then
    Result := '(SP)'
  else if FChar = CDBWhiteSpace then
    Result := '(DB SP)'
  else
    Result := FChar;
end;

function TSsPlayTimeSpecialChar.GetDisplayName: String;
begin
  Result := Format('"%s" = %d', [DisplayChar, FWait]);
end;

procedure TSsPlayTimeSpecialChar.SetChar(const Value: String);
begin
  if Value = '' then
    FChar := Value
  else
  begin
    if (Value[1] in LeadBytes) then
    begin
      if Length(Value) = 2 then
        FChar := Value;
    end else if Length(Value) = 1 then
      FChar := Value;
  end;
end;

procedure TSsPlayTimeSpecialChar.SetWait(const Value: integer);
begin
  FWait := Value;
end;

end.
