package net.waltzstudio.montecarlo.service;

import java.util.Map;
import javax.annotation.Resource;
import org.aopalliance.intercept.MethodInvocation;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.Execute;
import net.waltzstudio.base.framework.util.BaseUtil;
import net.waltzstudio.montecarlo.dao.account.AccountDao;
import net.waltzstudio.montecarlo.dao.label.LabelDao;
import net.waltzstudio.montecarlo.dao.project.ProjectDao;
import net.waltzstudio.montecarlo.dto.RequestInfoDto;
import net.waltzstudio.montecarlo.dto.SessionInfoDto;
import net.waltzstudio.montecarlo.entity.account.Account;
import net.waltzstudio.montecarlo.entity.project.Project;
import net.waltzstudio.montecarlo.form.LoginForm;
import net.waltzstudio.montecarlo.object.Global;
import net.waltzstudio.montecarlo.types.AdminFlg;

/**
 * ログインService
 */
public class LoginService {
	/** セッション情報 */
	@Resource
	protected SessionInfoDto sessionInfoDto;
	/** リクエスト情報 */
	@Resource
	protected RequestInfoDto requestInfoDto;
	/** リクエスト */
	@Resource
	protected Map<String, Object> requestScope;
	/** アカウントDao */
	@Resource
	protected AccountDao accountDao;
	/** プロジェクトDao */
	@Resource
	protected ProjectDao projectDao;
	/** ラベルDao */
	@Resource
	protected LabelDao labelDao;

	/**
	 * ログイン処理
	 *
	 * @return ログイン失敗⇒false / ログイン成功⇒true
	 */
	public boolean login(LoginForm form) {
		String login = form.id;
		String password = form.password;
		if (StringUtil.isEmpty(login) || StringUtil.isEmpty(password)) {
			return false;
		}
		password = BaseUtil.getDigest(password, "MD5");
		Account account = accountDao.findByLoginPassword(login, password);
		if (account == null) {
			return false;
		}
		sessionInfoDto.id = account.id;
		sessionInfoDto.name = account.name;
		sessionInfoDto.isAdmin = (AdminFlg.ADMIN == account.adminFlg);
		return true;
	}

	/**
	 * ログイン判定
	 *
	 * @param invocation
	 * @return ログイン済⇒true / 未ログイン⇒false
	 */
	public boolean isLoggedIn(MethodInvocation invocation) {
		if (invocation.getMethod().isAnnotationPresent(Execute.class) && sessionInfoDto.id == null) {
			return false;
		}
		return true;
	}

	/**
	 * 管理者ログイン判定
	 *
	 * @param invocation
	 * @return 管理者ログイン済⇒true / 管理者未ログイン⇒false
	 */
	public boolean isAdminLoggedIn(MethodInvocation invocation) {
		if (isLoggedIn(invocation) && sessionInfoDto.isAdmin) {
			return true;
		}
		return false;
	}

	/**
	 * プロジェクト識別子チェック
	 */
	public boolean checkProjectCode() {
		String code = BaseUtil.cast(requestScope.get(Global.RequestInfo.PROJECT_CODE));
		if (StringUtil.isEmpty(code)) {
			return true;
		}
		Project project = projectDao.findByCodeAndAccountId(code, sessionInfoDto.id);
		if (project != null) {
			requestInfoDto.projectCode = project.code;
			requestInfoDto.projectId = project.id;
			requestInfoDto.projectName = project.name;
			return true;
		}
		return false;
	}

	/**
	 * リクエスト情報設定
	 */
	public void setRequestInfo() {
		requestInfoDto.projects = projectDao.findByAccountId(sessionInfoDto.id);
		requestInfoDto.labels = labelDao.findByAccountId(sessionInfoDto.id);
	}

}
