package net.waltzstudio.montecarlo.service.batch;

import static net.waltzstudio.base.framework.util.SyntaxSugar.*;
import javax.annotation.Resource;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.container.hotdeploy.HotdeployUtil;
import org.seasar.framework.log.Logger;
import net.waltzstudio.montecarlo.object.Batch;
import net.waltzstudio.montecarlo.object.Global;

/**
 * バッチ管理Service
 */
public class BatchControlService {
	/** logger */
	private Logger logger = Logger.getLogger(BatchControlService.class);
	/** ロック取得用オブジェクト */
	private static final Object LOCK = new Object();

	/** バッチ情報 */
	@Resource
	protected Global global;

	/**
	 * バッチ起動
	 *
	 * @param batchId バッチID
	 * @return 起動成功⇒true / 起動失敗⇒false
	 */
	public boolean start(String batchId) {
		Batch batch = null;
		synchronized (LOCK) {
			batch = global.batches.get(batchId);
			if (batch != null) {
				logger.info(S(batch.getBatchName(), "は既に実行中です。"));
				return false;
			}
			batch = getInstance(batchId);
			if (batch == null) {
				return false;
			}
			global.batches.put(batchId, batch);
		}
		batch.setDaemon(true);
		batch.start();
		return true;
	}

	/**
	 * バッチ停止
	 *
	 * @param batchId バッチID
	 */
	public void stop(String batchId) {
		synchronized (LOCK) {
			Batch batch = global.batches.get(batchId);
			if (batch != null) {
				logger.info(S(batch.getBatchName(), "の停止要求を受け付けました。"));
				batch.interrupt();
				global.batches.remove(batchId);
			} else {
				logger.info(S(batchId, "のバッチは実行されていません。"));
			}
		}
	}

	/**
	 * バッチ削除
	 *
	 * @param batchId バッチID
	 */
	public void remove(String batchId) {
		synchronized (LOCK) {
			global.batches.remove(batchId);
		}
	}

	/**
	 * バッチインスタンス取得
	 *
	 * @param batchId バッチID
	 * @return バッチインスタンス
	 */
	public Batch get(String batchId) {
		return global.batches.get(batchId);
	}

	/**
	 * バッチインスタンス生成
	 *
	 * @param batchId バッチID
	 * @return バッチインスタンス
	 */
	protected Batch getInstance(String batchId) {
		if (HotdeployUtil.isHotdeploy()) {
			return null;
		}
		Batch instance = null;
		if (Global.BatchInfo.SENDMAIL_DAEMON.equals(batchId)) {
			instance = SingletonS2Container.getComponent(SendmailDaemonService.class);
		} else if (Global.BatchInfo.SENDMAIL_BATCH.equals(batchId)) {
			instance = SingletonS2Container.getComponent(SendmailBatchService.class);
		}
		return instance;
	}

}
