/*
 * $Id: FormSetFactory.java 293498 2005-10-04 02:59:43Z niallp $
 * $Rev: 293498 $
 * $Date: 2005-10-04 03:59:43 +0100 (Tue, 04 Oct 2005) $
 *
 * ====================================================================
 * Copyright 2005 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.commons.validator;

import org.xml.sax.Attributes;
import org.apache.commons.digester.AbstractObjectCreationFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Factory class used by Digester to create FormSet's.
 *
 * @since Validator 1.2
 */
public class FormSetFactory extends AbstractObjectCreationFactory {

    /** Logging */
    private static final Log log = LogFactory.getLog(ValidatorResources.class);

    /**
     * <p>Create or retrieve a <code>FormSet</code> for the specified
     *    attributes.</p>
     *
     * @param attributes The sax attributes for the formset element.
     * @return The FormSet for a locale.
     */
    public Object createObject(Attributes attributes) throws Exception {

        ValidatorResources resources = (ValidatorResources)digester.peek(0);

        String language = attributes.getValue("language");
        String country  = attributes.getValue("country");
        String variant  = attributes.getValue("variant");

        return createFormSet(resources, language, country, variant);

    }

    /**
     * <p>Create or retrieve a <code>FormSet</code> based on the language, country
     *    and variant.</p>
     *
     * @param resources The validator resources.
     * @param language The locale's language.
     * @param country The locale's country.
     * @param variant The locale's language variant.
     * @return The FormSet for a locale.
     * @since Validator 1.2
     */
    private FormSet createFormSet(ValidatorResources resources,
                                  String language,
                                  String country,
                                  String variant) throws Exception {

        // Retrieve existing FormSet for the language/country/variant
        FormSet formSet = resources.getFormSet(language, country, variant);
        if (formSet != null) {
            if (log.isDebugEnabled()) {
                log.debug("FormSet[" + formSet.displayKey() + "] found - merging.");
            }
            return formSet;
        }

        // Create a new FormSet for the language/country/variant
        formSet = new FormSet();
        formSet.setLanguage(language);
        formSet.setCountry(country);
        formSet.setVariant(variant);

        // Add the FormSet to the validator resources
        resources.addFormSet(formSet);

        if (log.isDebugEnabled()) {
            log.debug("FormSet[" + formSet.displayKey() + "] created.");
        }

        return formSet;

    }

}
