
import java.util.HashMap;
import java.util.Map;
import sdl4gcj.SDLSystem;
import sdl4gcj.video.*;
import sdl4gcj.event.*;
import sdl4gcj.input.*;

class Component implements Rect
{
	static final Color DEFAULT_BACKGROUND = new Color(0x00, 0x00, 0x33);
	static final Color DEFAULT_FOREGROUND = Color.WHITE;
	
	private int x = 0;
	private int y = 0;
	private int w = 100;
	private int h = 100;
	private Color background = DEFAULT_BACKGROUND;
	private Color foreground = DEFAULT_FOREGROUND;

	public Color getBackground() { return background; }
	public void setBackground(Color background) { this.background = background; }

	public Color getForeground() { return foreground; }
	public void setForeground(Color foreground) { this.foreground = foreground; }

	public int getW() { return w; }
	public void setW(int w) { this.w = w; }

	public int getH() { return h; }
	public void setH(int h) { this.h = h; }

	public int getX() { return x; }
	public void setX(int x) { this.x = x; }

	public int getY() { return y; }
	public void setY(int y) { this.y = y; }

	public void paint(Surface surface)
	{
		surface.setClipRect(this.getX(), this.getY(), 
			this.getW(), this.getH());
		int bgPixel = surface.mapRGB(this.getBackground());
		surface.fillRect(this.getX(), this.getY(), 
			this.getW(), this.getH(), bgPixel);
	}
}

class ButtonView extends Component
{
	static final int MAX_BUTTONS = 20;
	Joystick joystick;
	Surface buttonImage;
	int buttonIndex;

	ButtonView(Joystick joystick, int buttonIndex, Surface buttonImage)
	{
		this.joystick = joystick;
		this.buttonIndex = buttonIndex;
		this.buttonImage = buttonImage;

		this.setW(buttonImage.getW() / MAX_BUTTONS);
		this.setH(buttonImage.getH() / 2);
		this.setX(buttonIndex * this.getW());
		this.setY(Screen.getVideoSurface().getH() - this.getH());
	}

	public void paint(Surface surface)
	{
		super.paint(surface);
		int srcX = this.getX();
		int srcY = this.joystick.getButton(this.buttonIndex) * this.getH();
		surface.blitSurface(this.buttonImage, 
			srcX, srcY, this.getW(), this.getH(), 
			this.getX(), this.getY());
	}
}

class TwoAxesView extends Component
{
	static final int CURSOR_SIZE = 6;
	static final int DEFAULT_WIDTH = 100;
	static final int DEFAULT_HEIGHT = 100;

	private Joystick joystick;
	private int xAxisIndex, yAxisIndex;
	private int xAxisMax = 1;
	private int yAxisMax = 1;

	TwoAxesView(Joystick joystick, int xAxisIndex, int yAxisIndex)
	{
		this.joystick = joystick;
		this.xAxisIndex = xAxisIndex;
		this.yAxisIndex = yAxisIndex;

		this.setW(DEFAULT_WIDTH);
		this.setH(DEFAULT_HEIGHT);
	}

	public void paint(Surface surface)
	{
		super.paint(surface);

		int movableW = this.getW() - CURSOR_SIZE;
		int movableH = this.getH() - CURSOR_SIZE;

		int xAxisValue = this.joystick.getAxis(this.xAxisIndex);
		if (Math.abs(xAxisValue) > xAxisMax) 
			xAxisMax = Math.abs(xAxisValue);

		int yAxisValue = this.joystick.getAxis(this.yAxisIndex);
		if (Math.abs(yAxisValue) > yAxisMax) 
			yAxisMax = Math.abs(yAxisValue);

		int cursorX = this.getX() + (int)((xAxisValue + xAxisMax) / (double)xAxisMax / 2 * movableW);
		int cursorY = this.getY() + (int)((yAxisValue + yAxisMax) / (double)yAxisMax / 2 * movableH);

		surface.fillRect(cursorX, cursorY, CURSOR_SIZE, CURSOR_SIZE,
			surface.mapRGB(this.getForeground()));
	}
}

class AxisView extends Component
{
	static final int CURSOR_SIZE = 5;
	static final int DEFAULT_WIDTH = 10;
	static final int DEFAULT_HEIGHT = 100;

	private Joystick joystick;
	private int axisIndex;
	private int axisMax = 1;

	AxisView(Joystick joystick, int axisIndex)
	{
		this.joystick = joystick;
		this.axisIndex = axisIndex;

		this.setX(0);
		this.setY(0);
		this.setW(DEFAULT_WIDTH);
		this.setH(DEFAULT_HEIGHT);
	}

	public void paint(Surface surface)
	{
		super.paint(surface);

		int axisValue = this.joystick.getAxis(this.axisIndex);
		if (Math.abs(axisValue) > axisMax) 
			axisMax = Math.abs(axisValue);

		int movableH = this.getH() - CURSOR_SIZE;
		int cursorY = this.getY() + (int)((axisValue + axisMax) / (double)axisMax / 2 * movableH);

		surface.fillRect(this.getX(), cursorY, this.getW(), CURSOR_SIZE,
			surface.mapRGB(this.getForeground()));
	}
}

class HatView extends Component
{
	static final int CURSOR_SIZE = 6;
	static final int DEFAULT_WIDTH = 100;
	static final int DEFAULT_HEIGHT = 100;

	private Joystick joystick;
	private int hatIndex;

	HatView(Joystick joystick, int hatIndex)
	{
		this.joystick = joystick;
		this.hatIndex = hatIndex;

		this.setW(DEFAULT_WIDTH);
		this.setH(DEFAULT_HEIGHT);
	}

	public void paint(Surface surface)
	{
		super.paint(surface);

		int movableW = this.getW() - CURSOR_SIZE;
		int movableH = this.getH() - CURSOR_SIZE;

		int hatValue = this.joystick.getHat(this.hatIndex);

		int cursorX = this.getX();
		if ((hatValue & Joystick.SDL_HAT_RIGHT) != 0)
			cursorX += movableW;
		else if ((hatValue & Joystick.SDL_HAT_LEFT) == 0)
			cursorX += movableW/2;;

		int cursorY = this.getY();
		if ((hatValue & Joystick.SDL_HAT_DOWN) != 0)
			cursorY += movableH;
		else if ((hatValue & Joystick.SDL_HAT_UP) == 0) 
			cursorY += movableH/2;

		surface.fillRect(cursorX, cursorY, 
			CURSOR_SIZE, CURSOR_SIZE,
			surface.mapRGB(this.getForeground()));
	}
}

class JoystickView 
{
	static final String BUTTON_IMAGE_PATH = "images/JoyButtons.bmp";
	static final Color BACKGROUND_COLOR = new Color(0, 0, 0);

	Screen screen;
	Joystick joystick;
	Surface buttonImage;
	ButtonView[] buttons;
	TwoAxesView twoAxes;
	AxisView[] axis;
	HatView[] hats;

	public Joystick getJoystick() { return joystick; }
	public void setJoystick(Joystick joystick) { this.joystick = joystick; }

	public JoystickView()
	{
	}

	public void setUp()
	{
		this.screen = Screen.getVideoSurface();

		this.buttonImage = Surface.loadBMP(BUTTON_IMAGE_PATH);
		this.buttonImage.displayFormat();

		Joystick joystick = this.getJoystick();
		joystick.open();
		this.buttons = new ButtonView[joystick.getNumButtons()];
		for (int i = buttons.length-1;i >= 0;i--)
		{
			this.buttons[i] = new ButtonView(joystick, i, this.buttonImage);
		}

		int x = 10;
		int y = 10;

		int axisIndex = 0;
		if (joystick.getNumAxes() >= 2)
		{
			this.twoAxes = new TwoAxesView(joystick, axisIndex++, axisIndex++);
			twoAxes.setX(x);
			twoAxes.setY(y);
			x += twoAxes.getW() + 10;
		}

		this.axis = new AxisView[joystick.getNumAxes() - axisIndex];
		for (int i = axis.length-1;i >= 0;i--)
		{
			AxisView axisView = new AxisView(joystick, axisIndex++);
			axisView.setX(x);
			axisView.setY(y);
			x += axisView.getW() + 10;
			this.axis[i] = axisView;
		}

		x = 10;
		y += 120;
		this.hats = new HatView[joystick.getNumHats()];
		for (int i = this.hats.length-1;i >= 0;i--)
		{
			HatView hatView = new HatView(joystick, i);
			System.out.println(hatView);
			hatView.setX(x);
			hatView.setY(y);
			x += hatView.getW() + 10;
			this.hats[i] = hatView;
		}

		this.screen.setClipRect();
		this.screen.fillRect(this.screen.mapRGB(BACKGROUND_COLOR));
		this.paint();
	}

	public void tearDown()
	{
		this.joystick.close();
		this.joystick = null;
		this.screen = null;
		this.buttonImage = null;
		this.twoAxes = null;
		this.axis = null;
		this.hats = null;
		this.buttons = null;
	}

	void paint()
	{
		this.twoAxes.paint(this.screen);
		for (int i = axis.length-1;i >= 0;i--)
			this.axis[i].paint(this.screen);
		for (int i = hats.length-1;i >= 0;i--)
			this.hats[i].paint(this.screen);
		for (int i = buttons.length-1;i >= 0;i--)
			this.buttons[i].paint(this.screen);
		this.screen.flip();
	}
}


interface State
{
	String getName();
	void setContext(Context context);
	Context getContext();
	void processEvent(EventManager e);
	void setUp();
	void tearDown();
}

class SimpleState implements State
{
	public String getName()
	{
		return this.getClass().getName();
	}

	public void setUp()
	{
	}

	public void tearDown()
	{
	}

	private Context context;
	public void setContext(Context context)
	{
		this.context = context;
	}
	public Context getContext()
	{
		return this.context;
	}

	public void processEvent(EventManager event)
	{
	}
}


class Context 
{
	private Map stateTable = new HashMap();
	private State currentState;

	public void addState(State state)
	{
		stateTable.put(state.getName(), state);
		state.setContext(this);
	}

	public void removeState(State state)
	{
		stateTable.remove(state.getName());
		state.setContext(null);
	}

	public void changeState(String name)
	{
		State newState = (State)stateTable.get(name);
		if (newState != null)
		{
			if (this.currentState != null)
			{
				this.currentState.tearDown();
			}
			this.currentState = newState;
			this.currentState.setUp();
		}
	}

	public boolean done;
	public void mainLoop()
	{
		EventManager event = new EventManager();
		done = false;
		while (!done)
		{
			event.waitEvent();
			this.currentState.processEvent(event);
		}
	}
}

class SelectJoystickState extends SimpleState
{
	private Screen screen;
	private Joystick[] joysticks;
	private JoystickMenuItem[] joystickMenuItems;
	private JoystickView joystickView;
	private int selectedIndex = 0;

	public SelectJoystickState(JoystickView joystickView)
	{
		this.joystickView = joystickView;
	}

	public void setUp()
	{
		this.screen = Screen.getVideoSurface();
		this.screen.setClipRect();
		this.screen.fillRect(this.screen.mapRGB(Color.BLACK));
		this.screen.flip();
		this.joysticks = Joystick.getAllJoysticks();
		this.joystickMenuItems = new JoystickMenuItem[joysticks.length];
		this.selectedIndex = 0;
		for (int i = 0;i < joysticks.length;i++)
		{
			this.joystickMenuItems[i] = new JoystickMenuItem(joysticks[i]);
			this.joystickMenuItems[i].setW(620);
			this.joystickMenuItems[i].setH(70);
			this.joystickMenuItems[i].setX(10);
			this.joystickMenuItems[i].setY(10 + i * (this.joystickMenuItems[i].getH()+10));
		}
		this.paint();
	}

	public void tearDown()
	{
		this.joystickView.setJoystick(joysticks[selectedIndex]);
		this.screen = null;
		this.joysticks = null;
		this.joystickMenuItems = null;
		this.selectedIndex = 0;
	}

	public void processEvent(EventManager event)
	{
		switch (event.type)
		{
			case SDLEvent.SDL_KEYDOWN :
				switch (event.keydown.sym)
				{
					case Keyboard.SDLK_ESCAPE:
					case Keyboard.SDLK_q:
						this.getContext().done = true;
						break;
					case Keyboard.SDLK_UP:
						this.selectedIndex = Math.abs(this.selectedIndex - 1) % this.joysticks.length;
						this.paint();
						break;
					case Keyboard.SDLK_DOWN:
						this.selectedIndex = (this.selectedIndex + 1) % this.joysticks.length;
						this.paint();
						break;
					case Keyboard.SDLK_RETURN:
					case Keyboard.SDLK_SPACE:
						this.getContext().changeState("CheckJoystickState");
						break;
					default:
						break;
				}
				break;
			case SDLEvent.SDL_QUIT:
				this.getContext().done = true;
				break;
			default:
				break;
		}
	}

	protected void paint()
	{
		for (int i = this.joystickMenuItems.length-1;i >= 0;i--)
		{
			this.joystickMenuItems[i].paint(this.screen);
		}
		this.screen.flip();
	}

	class JoystickMenuItem extends Component
	{
		static final Color selectedColor = new Color(0xff, 0xff, 0xff);
		static final ImageFont font = new ImageFont("images/font-8x16.bmp", 8, 16);
//		static final ImageFont font = new ImageFont();

		private Joystick joystick; 
		private String name; 
		private StringBuffer info = new StringBuffer(); 

		public JoystickMenuItem(Joystick joystick)
		{
			this.joystick = joystick;
			this.joystick.open();
			this.name = this.joystick.getName();
			info.append(this.joystick.getNumAxes());
			info.append(" Axes, ");
			info.append(this.joystick.getNumBalls());
			info.append(" Balls, ");
			info.append(this.joystick.getNumHats());
			info.append(" Hats, ");
			info.append(this.joystick.getNumButtons());
			info.append(" Buttons.");
			this.joystick.close();
		}

		public void paint(Surface surface)
		{
			super.paint(surface);
			if (this.joystick.getIndex() == selectedIndex)
			{
				surface.fillRect(this.getX(), this.getY(), 
					this.getW(), this.getH(),
					surface.mapRGB(selectedColor));

				surface.fillRect(this.getX()+2, this.getY()+2, 
					this.getW()-4, this.getH()-4,
					surface.mapRGB(this.getBackground()));
			}

			int destX, destY;
			destX = this.getX()+4;
			destY = this.getY()+4;
			font.draw(surface, name, destX, destY);

			destY += font.getHeight();
			font.draw(surface, info, destX, destY);
		}
	}
}


class CheckJoystickState extends SimpleState
{
	JoystickView joystickView;

	public CheckJoystickState(JoystickView joystickView)
	{
		this.joystickView = joystickView;
	}

	public void setUp()
	{
		this.joystickView.setUp();
	}

	public void tearDown()
	{
		this.joystickView.tearDown();
	}

	public void processEvent(EventManager event)
	{
		switch (event.type)
		{
			case SDLEvent.SDL_JOYAXISMOTION :
			case SDLEvent.SDL_JOYBALLMOTION :
			case SDLEvent.SDL_JOYHATMOTION :
			case SDLEvent.SDL_JOYBUTTONDOWN :
			case SDLEvent.SDL_JOYBUTTONUP :
				System.out.println(event.lastEvent);
				this.joystickView.paint();
				break;
			case SDLEvent.SDL_KEYDOWN :
				if ((event.keydown.sym == Keyboard.SDLK_ESCAPE) ||
					(event.keydown.sym == Keyboard.SDLK_q))
					this.getContext().changeState("SelectJoystickState");
				break;
			case SDLEvent.SDL_QUIT:
				this.getContext().changeState("SelectJoystickState");
				break;
			default:
				break;
		}

	}

}

public class JoystickTest
{
	static final int WAIT_TIME = 30;
	public static void main(String[] args)
	{
		if (SDLSystem.init(SDLSystem.SDL_INIT_VIDEO|SDLSystem.SDL_INIT_JOYSTICK) != 0)
		{
			System.out.println("Init() fail");
			System.exit(0);
		}

		if (Joystick.getNumJoysticks() <= 0)
		{
			System.out.println("No Joystick Found");
			System.exit(0);
		}
		

		try
		{
			Screen screen = Screen.setVideoMode(640, 480, 32, Surface.SDL_SWSURFACE);

			JoystickView joystickView = new JoystickView();

			SelectJoystickState selectJoystickState;
			selectJoystickState = new SelectJoystickState(joystickView);

			CheckJoystickState checkJoystickState;
			checkJoystickState = new CheckJoystickState(joystickView);

			Context context = new Context();
			context.addState(selectJoystickState);
			context.addState(checkJoystickState);

			context.changeState(selectJoystickState.getName());
			context.mainLoop();

		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		finally
		{
			SDLSystem.quit();
		}
	}
}
