/************************************************************
Copyright (C) 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/
package sdl4gcj.input;

import sdl4gcj.SDLException;

public class Joystick implements JoystickConstants
{
	private static Joystick[] joysticks = null;

	// class methods
	private static SDLException createNotOpenedException(int index, String name)
	{
		return new SDLException("Joystick " + index + "(" + name + ") is not opened.");
	}

	private static Joystick[] getJoysticks() 
	{
		if (joysticks == null)
		{
			joysticks = new Joystick[Joystick.getNumJoysticks()];
		}
		else if (Joystick.getNumJoysticks() > joysticks.length)
		{
			Joystick[] newArray = new Joystick[Joystick.getNumJoysticks()];
			for (int i = 0;i < joysticks.length;i++)
				newArray[i] = joysticks[i];
			joysticks = newArray;
		}
		return joysticks; 
	}

	/**
	* Gets count available joysticks.
	* SDL_NumJoysticks() wrapper.
	**/
	public static native int getNumJoysticks();

	/**
	* Gets joystick at index.
	**/
	public static Joystick getJoystick(int index)
	{
		Joystick joystick;

		if (index < 0 || index >= Joystick.getNumJoysticks())
			throw new SDLException("No such joystick : " + index);

		Joystick[] sticks = getJoysticks();
		if (sticks[index] == null) 
			sticks[index] = new Joystick(index);
		joystick = sticks[index];

		return joystick;
	}

	/**
	* Gets first joystick.
	**/
	public static Joystick getJoystick()
	{
		return Joystick.getJoystick(0);
	}

	public static Joystick[] getAllJoysticks()
	{
		int num = Joystick.getNumJoysticks();
		Joystick[] joysticks = new Joystick[num];

		for (int i = num-1;i >= 0;i--)
		{
			joysticks[i] = Joystick.getJoystick(i);
		}

		return joysticks;
	}

	/**
	* Gets joystick event polling mode.
	* SDL_JoystickEventState() wrapper.
	**/
	public static native boolean getEventState();

	/**
	* Sets joystick event polling mode.
	* SDL_JoystickEventState() wrapper.
	**/
	public static native void setEventState(boolean state);

	/**
	* Updates the state of all joysticks.
	* Updates  the  state(position,  buttons,  etc.) of all open
	* joysticks. If  joystick  events  have  been  enabled  with
	* Joystick.setEventState(true) then  this is called automatically
	* in the event loop.
	* SDL_JoystickUpdate() wrapper.
	**/
	public static native void update();

	// constructor
	private Joystick(int joystickIndex)
	{
		this.index = joystickIndex;
	}

	// instance method
	/**
	* Gets joystick index number.
	**/
	public int getIndex()
	{
		return this.index;
	}

	/**
	* Returns <code>true</code>, if this joystick is opened. 
	**/
	public native boolean isOpened();

	/**
	* Open this joystick devece.
	* SDL_JoystickOpen() wrapper.
	**/
	public native void open();

	/**
	* Close this joystick devece.
	* SDL_JoystickClose() wrapper.
	**/
	public native void close();

	/**
	* Get joystick name.
	* SDL_JoystickName() wrapper.
	**/
	public native String getName();

	/**
	* Get number of axes.
	* SDL_JoystickNumAxes() wrapper.
	**/
	public native int getNumAxes();

	/**
	* Get number of joystick balls.
	* SDL_JoystickNumBalls() wrapper.
	**/
	public native int getNumBalls();

	/**
	* Get number of joystick hats.
	* SDL_JoystickNumHats() wrapper.
	**/
	public native int getNumHats();

	/**
	* Get number of joystick buttons.
	* SDL_JoystickNumButtons() wrapper.
	**/
	public native int getNumButtons();

	/**
	* Get the current state of an axis.
	* SDL_JoystickGetAxis() wrapper.
	**/
	public native int getAxis(int axisIndex);

	/**
	* Get relative trackball motion.
	* It's buggy... Don't use.
	* SDL_JoystickGetBall() wrapper.
	**/
	public native int getBallDX(int ballIndex);

	/**
	* Get relative trackball motion.
	* It's buggy... Don't use.
	* SDL_JoystickGetBall() wrapper.
	**/
	public native int getBallDY(int ballIndex);

	/**
	* Get the current state of an axis.
	* SDL_JoystickGetHat() wrapper.
	* @return The current state is returned as a int which is 
	*         OR'd combination of one or more of the following
	* <ul>
	* <li>JoystickConstants.SDL_HAT_CENTERED</li>
	* <li> JoystickConstants.SDL_HAT_UP</li>
	* <li> JoystickConstants.SDL_HAT_RIGHT</li>
	* <li> JoystickConstants.SDL_HAT_DOWN</li>
	* <li> JoystickConstants.SDL_HAT_LEFT</li>
	* <li> JoystickConstants.SDL_HAT_RIGHTUP</li>
	* <li> JoystickConstants.SDL_HAT_RIGHTDOWN</li>
	* <li> JoystickConstants.SDL_HAT_LEFTUP</li>
	* <li> JoystickConstants.SDL_HAT_LEFTDOWN</li>
	* </ul>
	**/
	public native int getHat(int hatIndex);

	/**
	* Get  the  current state of a given button on this joystick.
	* SDL_JoystickGetButton() wrapper.
	**/
	public native int getButton(int butotnIndex);

	public String toString()
	{
		return super.toString() + "\n" +
		"getName() : " + getName() + "\n" +
		"getIndex() : " + getIndex() + "\n" ;
	}

	protected void finalize() throws Throwable
	{
		this.close();
		super.finalize();
	}

	private int index;
	private gnu.gcj.RawData implementation;
}
