/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/


#include<stdio.h>
#include<stdlib.h>

#include"zlib.h"
#include"SDL_archive.h"


#define	GZ_BUF	1024
#define	UN_BUF	65536


enum
{
	HEADER_SUCCESS,
	HEADER_BROKEN,
	HEADER_NO_MAGIC
};


enum
{
	FILE_GZIP,
	FILE_UNCOMPRESS
};


typedef struct gz_filter
{
	SDL_Archive *inputArchive;
	
	int openNumber;
	
	/* for uncompress */
	int isStreamFinish;
	long fileSeek;
	long originalSize;
	
	int headSize;
	int memSeek;
	int flush;
	char compressData[GZ_BUF];
	char uncompressData[UN_BUF];
	long restDataSize;
	z_stream z;
} GZ_FILTER;


static void Uncompressed_SetMethod(SDL_Archive *archive);

static long Uncompressed_Size(SDL_Archive *archive);
static long Uncompressed_Tell(SDL_Archive *archive);
static int Uncompressed_GetChar(SDL_Archive *archive);
static int Uncompressed_Seek(SDL_Archive *archive, const long offset, const int whence);
static int Uncompressed_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Uncompressed_EOF(SDL_Archive *archive);
static int Uncompressed_Close(SDL_Archive *archive);


static void GzArc_SetMethod(SDL_Archive *archive);

static long GzArc_Size(SDL_Archive *archive);
static long GzArc_Tell(SDL_Archive *archive);
static int GzArc_GetChar(SDL_Archive *archive);
static int GzArc_Seek(SDL_Archive *archive, const long offset, const int whence);
static int GzArc_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int GzArc_EOF(SDL_Archive *archive);
static int GzArc_Close(SDL_Archive *archive);


static int Common_Open(SDL_Archive *archive, const char *file);
static int Common_NumOpen(SDL_Archive *archive, const int num);
static int Common_NameToIndex(SDL_Archive *archive, const char *name);
static int Common_FileNumber(SDL_Archive *archive);
static void Common_Finish(SDL_Archive *archive);


SDL_Archive *Archive_FromGZFilter(
	ArchiverList *archiverList,
	const char *file)
{
	SDL_Archive *archive;
	SDL_Archive *inputArchive;
	GZ_FILTER *gzFilter;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	inputArchive = Archive_FromArchiverList(archiverList, file);
	if (inputArchive == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	
	gzFilter = (GZ_FILTER*)malloc(sizeof(GZ_FILTER));
	if (gzFilter == NULL) {
		free(archive);
		return NULL;
	}
	archive->data = gzFilter;
	gzFilter->inputArchive = inputArchive;
	
	archive->open = Common_Open;
	archive->numopen = Common_NumOpen;
	archive->name2index = Common_NameToIndex;
	archive->filenum = Common_FileNumber;
	archive->finish = Common_Finish;
	
	archive->get_char = NULL;
	archive->read = NULL;
	archive->seek = NULL;
	archive->size = NULL;
	archive->tell = NULL;
	archive->eof = NULL;
	archive->close = NULL;
	
	gzFilter->openNumber = -1;
	
	return archive;
}




void Uncompressed_SetMethod(SDL_Archive *archive)
{
	archive->get_char = Uncompressed_GetChar;
	archive->read = Uncompressed_Read;
	archive->seek = Uncompressed_Seek;
	archive->size = Uncompressed_Size;
	archive->tell = Uncompressed_Tell;
	archive->eof = Uncompressed_EOF;
	archive->close = Uncompressed_Close;
}


long Uncompressed_Size(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_Size(gzFilter->inputArchive);
}


long Uncompressed_Tell(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_Tell(gzFilter->inputArchive);
}


int Uncompressed_GetChar(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_GetChar(gzFilter->inputArchive);
}


int Uncompressed_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_Seek(gzFilter->inputArchive, offset, whence);
}


int Uncompressed_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_Read(gzFilter->inputArchive, mem, size, maxnum);
}


int Uncompressed_EOF(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_EOF(gzFilter->inputArchive);
}


int Uncompressed_Close(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	if (gzFilter->openNumber != -1) {
		Archive_Close(gzFilter->inputArchive);
		gzFilter->openNumber = -1;
	}
	return ARCHIVE_SUCCESS;
}








static void GzArc_InCheck(GZ_FILTER *gzt);
static void GzArc_OutCheck(GZ_FILTER *gzt);
static int GzArc_IsEOF(GZ_FILTER *gzt);
static int GzArc_ReadAttended(GZ_FILTER *gzt);


void GzArc_SetMethod(SDL_Archive *archive)
{
	archive->get_char = GzArc_GetChar;
	archive->read = GzArc_Read;
	archive->seek = GzArc_Seek;
	archive->size = GzArc_Size;
	archive->tell = GzArc_Tell;
	archive->eof = GzArc_EOF;
	archive->close = GzArc_Close;
}


static int GzArc_ReadAttended(GZ_FILTER *gzFilter)
{
	int i;
	int method;
	int flags;
	int templen;
	long count;
	char check[2];
	unsigned char originalFileCheck[4];
	const char gzMagic[2] = {'\x1f', '\x8b'};
	
	count = 0;
	
	count += 2;
	if (gzFilter->restDataSize < count) {
		return HEADER_NO_MAGIC;
	}
	check[0] = Archive_GetChar(gzFilter->inputArchive);
	check[1] = Archive_GetChar(gzFilter->inputArchive);
	if (check[0] != gzMagic[0] || check[1] != gzMagic[1]) {
		return HEADER_NO_MAGIC;
	}
	
	count++;
	if (gzFilter->restDataSize < count) {
		return HEADER_BROKEN;
	}
	method = Archive_GetChar(gzFilter->inputArchive);
	
	count++;
	if (gzFilter->restDataSize < count) {
		return HEADER_BROKEN;
	}
	flags = Archive_GetChar(gzFilter->inputArchive);
	
	if (method != Z_DEFLATED || (flags & 0xE0) != 0) {
		return HEADER_BROKEN;
	}
	
	count += 6;
	if (gzFilter->restDataSize < count) {
		return HEADER_BROKEN;
	}
	Archive_GetChar(gzFilter->inputArchive);
	Archive_GetChar(gzFilter->inputArchive);
	Archive_GetChar(gzFilter->inputArchive);
	Archive_GetChar(gzFilter->inputArchive);
	Archive_GetChar(gzFilter->inputArchive);
	Archive_GetChar(gzFilter->inputArchive);
	
	if (gzFilter->restDataSize < count) {
		return HEADER_BROKEN;
	}
	if (flags & 0x04) { /* skip the extra field */
		templen = Archive_GetChar(gzFilter->inputArchive);
		templen += (Archive_GetChar(gzFilter->inputArchive) << 8);
		count += templen;
		if (gzFilter->restDataSize < count) {
			return HEADER_BROKEN;
		}
		for (i = 0; i < count; i++) {
			Archive_GetChar(gzFilter->inputArchive);
		}
	}
	
	if (flags & 0x08) { /* skip the original file name */
		do {
			count += 1;
			if (gzFilter->restDataSize < count) {
				return HEADER_BROKEN;
			}
		} while (Archive_GetChar(gzFilter->inputArchive) != 0);
	}
	
	if (flags & 0x10) { /* skip the .gz file comment */
		do {
			count += 1;
			if (gzFilter->restDataSize < count) {
				return HEADER_BROKEN;
			}
		} while (Archive_GetChar(gzFilter->inputArchive) != 0);
	}
	
	if (flags & 0x02) { /* skip the header crc */
		count += 2;
		if (gzFilter->restDataSize < count) {
			return HEADER_BROKEN;
		}
	}
	
	gzFilter->headSize = count;
	
	Archive_Seek(gzFilter->inputArchive, gzFilter->restDataSize - count - 4, SEEK_CUR);
	Archive_Read(gzFilter->inputArchive, originalFileCheck, 4, 1);
	gzFilter->originalSize = 0;
	gzFilter->originalSize += originalFileCheck[0];
	gzFilter->originalSize += originalFileCheck[1] * 256;
	gzFilter->originalSize += originalFileCheck[2] * 256 * 256;
	gzFilter->originalSize += originalFileCheck[3] * 256 * 256 * 256;
	
	return HEADER_SUCCESS;
}


int Common_Open(SDL_Archive *archive, const char *file)
{
	return Common_NumOpen(archive, Common_NameToIndex(archive, file));
}


int Common_NumOpen(SDL_Archive *archive, const int num)
{
	int ret;
	int flag;
	int readSize;
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	if (gzFilter->openNumber != -1) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	ret = Archive_NumOpen(gzFilter->inputArchive, num);
	if (ret != ARCHIVE_SUCCESS) {
		return ret;
	}
	
	gzFilter->restDataSize = Archive_Size(gzFilter->inputArchive);
	
	ret = GzArc_ReadAttended(gzFilter);
	if (ret == HEADER_BROKEN) {
		Archive_Close(gzFilter->inputArchive);
		return ARCHIVE_ERROR_BROKEN_DATA;
	}
	if (ret == HEADER_NO_MAGIC) {
		Archive_Seek(gzFilter->inputArchive, 0, SEEK_SET);
		gzFilter->originalSize = Archive_Size(gzFilter->inputArchive);
		gzFilter->fileSeek = 0L;
		gzFilter->isStreamFinish = 0;
		gzFilter->openNumber = num;
		Uncompressed_SetMethod(archive);
		return ARCHIVE_SUCCESS;
	}
	
	Archive_Seek(gzFilter->inputArchive, 0, SEEK_SET);
	
	readSize = GZ_BUF;
	gzFilter->flush = Z_NO_FLUSH;
	if (gzFilter->restDataSize < GZ_BUF) {
		readSize = (int)(gzFilter->restDataSize);
	}
	Archive_Read(gzFilter->inputArchive, gzFilter->compressData, readSize, 1);
	gzFilter->restDataSize -= readSize;
	if (gzFilter->headSize == -1) {
		Archive_Close(gzFilter->inputArchive);
		return ARCHIVE_ERROR_BROKEN_DATA;
	}
	
	gzFilter->z.zalloc = Z_NULL;
	gzFilter->z.zfree = Z_NULL;
	gzFilter->z.opaque = Z_NULL;
	
	gzFilter->z.next_in = Z_NULL;
	gzFilter->z.avail_in = 0;
	flag = inflateInit2(&gzFilter->z, -MAX_WBITS);
	if (flag != Z_OK) {
		Archive_Close(gzFilter->inputArchive);
		return ARCHIVE_ERROR_BROKEN_DATA;
	}
	
	gzFilter->z.next_in = (char*)(gzFilter->compressData + gzFilter->headSize);
	gzFilter->z.avail_in = readSize - gzFilter->headSize;
	gzFilter->z.next_out = (char*)gzFilter->uncompressData;
	gzFilter->z.avail_out = UN_BUF;
	
	gzFilter->fileSeek = 0L;
	gzFilter->memSeek = 0L;
	gzFilter->isStreamFinish = 0;
	
	flag = inflate(&gzFilter->z, gzFilter->flush);
	if (flag == Z_OK) {
		gzFilter->openNumber = num;
		GzArc_SetMethod(archive);
		return ARCHIVE_SUCCESS;
	}
	if (flag == Z_STREAM_END) {
		gzFilter->openNumber = num;
		gzFilter->isStreamFinish = 1;
		GzArc_SetMethod(archive);
		return ARCHIVE_SUCCESS;
	}
	
	return ARCHIVE_ERROR_UNKNOWN;
}


long GzArc_Size(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return gzFilter->originalSize;
}


long GzArc_Tell(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return gzFilter->fileSeek;
}


static void GzArc_InCheck(GZ_FILTER *gzFilter)
{
	int readSize;
	
	if (!Archive_EOF(gzFilter->inputArchive) && gzFilter->z.avail_in == 0) {
		readSize = GZ_BUF;
		if (gzFilter->restDataSize < GZ_BUF) {
			gzFilter->flush = Z_FINISH;
			readSize = (int)gzFilter->restDataSize;
		}
		Archive_Read(gzFilter->inputArchive, gzFilter->compressData, readSize, 1);
		gzFilter->restDataSize -= readSize;
		gzFilter->z.next_in = gzFilter->compressData;
		gzFilter->z.avail_in = readSize;
	}
}


static void GzArc_OutCheck(GZ_FILTER *gzFilter)
{
	int status;
	
	if (gzFilter->isStreamFinish == 0 && gzFilter->memSeek >= UN_BUF - (int)gzFilter->z.avail_out) {
		gzFilter->z.next_out = gzFilter->uncompressData;
		gzFilter->z.avail_out = UN_BUF;
		gzFilter->memSeek = 0;
		status = inflate(&gzFilter->z, gzFilter->flush);
		if (status == Z_STREAM_END) {
			gzFilter->isStreamFinish = 1;
		}
	}
}


int GzArc_GetCharWithoutSeek(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	GzArc_InCheck(gzFilter);
	GzArc_OutCheck(gzFilter);
	if (GzArc_IsEOF(gzFilter)) {
		return EOF;
	}
	
	return gzFilter->uncompressData[gzFilter->memSeek++];
}


int GzArc_GetChar(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	GzArc_InCheck(gzFilter);
	GzArc_OutCheck(gzFilter);
	if (GzArc_IsEOF(gzFilter)) {
		return EOF;
	}
	gzFilter->fileSeek++;
	
	return gzFilter->uncompressData[gzFilter->memSeek++];
}


int GzArc_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	int num;
	int check;
	long position;
	long i;
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	switch (whence) {
	case SEEK_SET:
		position = offset;
		break;
	case SEEK_CUR:
		position = gzFilter->fileSeek + offset;
		break;
	case SEEK_END:
		position = gzFilter->originalSize + offset;
		break;
	default:
		return ARCHIVE_ERROR_WHENCE;
	}
	
	if (position > gzFilter->originalSize) {
		position = gzFilter->originalSize;
	}
	if (position < 0) {
		position = 0;
	}
	
	if (position > gzFilter->fileSeek) {
		for (i = gzFilter->fileSeek; i < position; i++) {
			GzArc_GetChar(archive);
		}
	} else if (position < gzFilter->fileSeek) {
		num = gzFilter->openNumber;
		GzArc_Close(archive);
		check = Common_NumOpen(archive, num);
		if (check != ARCHIVE_SUCCESS) {
			return check;
		}
		for (i = 0; i < position; i++) {
			GzArc_GetChar(archive);
		}
	}
	
	
	return ARCHIVE_SUCCESS;
}


int GzArc_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	int i;
	int count;
	long position;
	
	char *memc;
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	memc = (char*)mem;
	
	position = 0L;
	count = 0;
	while (count < maxnum && gzFilter->fileSeek + position < gzFilter->originalSize) {
		for (i = 0; i < size; i++) {
			memc[count * size + i] = GzArc_GetCharWithoutSeek(archive);
		}
		position += size;
		count++;
	}
	
	gzFilter->fileSeek += size *count;
	
	return count;
}


static int GzArc_IsEOF(GZ_FILTER *gzFilter)
{
	if (gzFilter->isStreamFinish == 1 && gzFilter->memSeek >= UN_BUF - (int)gzFilter->z.avail_out) {
		return 1;
	} else {
		if (gzFilter->memSeek > UN_BUF) {
			return 1;
		} else {
			return 0;
		}
	}
}


int GzArc_EOF(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	GzArc_InCheck(gzFilter);
	GzArc_OutCheck(gzFilter);
	
	return GzArc_IsEOF(gzFilter);
}


int GzArc_Close(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	if (gzFilter->openNumber != -1) {
		inflateEnd(&gzFilter->z);
		Archive_Close(gzFilter->inputArchive);
		gzFilter->openNumber = -1;
	}
	return ARCHIVE_SUCCESS;
}


int Common_NameToIndex(SDL_Archive *archive, const char *name)
{
	int ret;
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	ret = Archive_NameToIndex(gzFilter->inputArchive, name);
	
	return ret;
}


int Common_FileNumber(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	return Archive_FileNum(gzFilter->inputArchive);
}


void Common_Finish(SDL_Archive *archive)
{
	GZ_FILTER *gzFilter;
	
	gzFilter = (GZ_FILTER*)archive->data;
	
	if (gzFilter->openNumber != -1) {
		Archive_Close(archive);
	}
	
	if (gzFilter != NULL) {
		Archive_Close(gzFilter->inputArchive);
		free(gzFilter);
	}
	
	Archive_FreeMainContext(archive);
}
