/*
    SDL_archive
    Copyright (C) 2004  Kazunori Itoyanagi

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Kazunori Itoyanagi
    itkz@users.sourceforge.jp
*/

#include<math.h>
#include<stdlib.h>
#include<stdio.h>
#include<string.h>

#include"SDL_archive.h"


#define	TAR_BLOCK	512

typedef struct _tar_header
{
  char name[100];
  char mode[8];
  char uid[8];
  char gid[8];
  char size[12];
  char mtime[12];
  char chksum[8];
  char typeflag;
  char linkname[100];
  char magic[6];
  char version[2];
  char uname[32];
  char gname[32];
  char devmajor[8];
  char devminor[8];
  char prefix[167];
} TAR_HEADER;


typedef struct _tar_chain {
	long seek;
	long size;
	TAR_HEADER tar;
} TAR_CHAIN;


typedef struct _tar_data {
	int filenum;
	int allocatedchain;
	TAR_CHAIN *chain;
	
	FILE *fp;
	
	int openNumber;
	long seekInArchive;
	long nowSize;
	long nowSeek;
} TAR_DATA;


static TAR_DATA *Tar_Create(const char *file);
static void Tar_Finish(SDL_Archive *archive);
static int Tar_NameToIndex(SDL_Archive *archive, const char *filename);
static int Tar_Open(SDL_Archive *archive, const char *filename);
static int Tar_NumOpen(SDL_Archive *archive, const int num);
static int Tar_FileNumber(SDL_Archive *archive);
static int Tar_GetChar(SDL_Archive *archive);
static long Tar_Size(SDL_Archive *archive);
static long Tar_Tell(SDL_Archive *archive);
static int Tar_EOF(SDL_Archive *archive);
static int Tar_Close(SDL_Archive *archive);
static int Tar_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum);
static int Tar_Seek(SDL_Archive *archive, const long offset, const int whence);


int Archive_IsTar(const char *file)
{
	int ret;
	FILE *fp;
	TAR_HEADER temp_header;
	
	fp = fopen(file, "rb");
	if (fp == NULL) {
		return 0;
	}
	
	ret = fread(&temp_header, TAR_BLOCK, 1, fp);
	if (ret != 1 || strcmp(temp_header.magic, "ustar") != 0) {
		fclose(fp);
		return 0;
	}
	fclose(fp);
	
	return 1;
}


SDL_Archive *Archive_FromTar(const char *file)
{
	SDL_Archive *archive;
	TAR_DATA *tar;
	
	archive = Archive_Alloc();
	if (archive == NULL) {
		return NULL;
	}
	
	tar = Tar_Create(file);
	if (tar == NULL) {
		Archive_FreeMainContext(archive);
		return NULL;
	}
	tar->openNumber = -1;
	archive->data = tar;
	
	archive->open = Tar_Open;
	archive->numopen = Tar_NumOpen;
	archive->name2index = Tar_NameToIndex;
	archive->filenum = Tar_FileNumber;
	archive->finish = Tar_Finish;
	
	archive->get_char = Tar_GetChar;
	archive->read = Tar_Read;
	archive->seek = Tar_Seek;
	archive->size = Tar_Size;
	archive->tell = Tar_Tell;
	archive->eof = Tar_EOF;
	archive->close = Tar_Close;
	
	return archive;
}


TAR_DATA *Tar_Create(const char *file)
{
	void *tempptr;
	int i;
	int blocknum;
	char sizetemp[2];
	long fileseek;
	long filesize;
	TAR_DATA *tar;
	
	tar = malloc(sizeof(TAR_DATA));
	if (tar == NULL) {
		return NULL;
	}
	
	tar->fp = fopen(file, "rb");
	if (tar->fp == NULL) {
		free(tar);
		return NULL;
	}
	
	fseek(tar->fp, 0L, SEEK_END);
	filesize = ftell(tar->fp);
	rewind(tar->fp);
	
	tar->filenum = 0;
	fileseek = 0L;
	
	tar->chain = malloc(sizeof(TAR_CHAIN));
	if (tar->chain == NULL) {
		fclose(tar->fp);
		free(tar);
		return NULL;
	}
	tar->allocatedchain = 1;
	while (filesize - fileseek >= TAR_BLOCK) {
		if (tar->filenum + 1 > tar->allocatedchain) {
			tar->allocatedchain = tar->allocatedchain * 2;
			tempptr = realloc(tar->chain,
				sizeof(TAR_CHAIN) * tar->allocatedchain);
			if (tempptr == NULL) {
				break;
			}
			tar->chain = (TAR_CHAIN *)tempptr;
		}
		fread(&tar->chain[tar->filenum].tar, TAR_BLOCK, 1, tar->fp);
		blocknum = 1;
		if (strlen(tar->chain[tar->filenum].tar.name) == 0) {
			fileseek += TAR_BLOCK;
			continue;
		}
		tar->chain[tar->filenum].size = 0L;
		tar->chain[tar->filenum].seek = fileseek + TAR_BLOCK;
		/* convert octadecimal to decimal */
		for (i = 0; i < 12; i++) {
			sizetemp[0] = tar->chain[tar->filenum].tar.size[i - 1];
			sizetemp[1] = '\0';
			tar->chain[tar->filenum].size +=
				(long)(atoi(sizetemp) * pow(8, 11 - i));
		}
		blocknum += tar->chain[tar->filenum].size / TAR_BLOCK;
		/* odd */
		if (tar->chain[tar->filenum].size % TAR_BLOCK != 0) {
			blocknum++;
		}
		fileseek += blocknum * TAR_BLOCK;
		fseek(tar->fp, fileseek, SEEK_SET);
		tar->filenum++;
	}
	
	return tar;
}


void Tar_Finish(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	fclose(tar->fp);
	free(tar->chain);
	free(tar);
}


int Tar_NameToIndex(SDL_Archive *archive, const char *filename)
{
	int i;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	for (i = 0; i < tar->filenum; i++) {
		if (strcmp(tar->chain[i].tar.name, filename) == 0) {
			return i;
		}
	}
	
	return -1;
}


int Tar_Open(SDL_Archive *archive, const char *filename)
{
	return Tar_NumOpen(archive, Tar_NameToIndex(archive, filename));
}


int Tar_NumOpen(SDL_Archive *archive, const int num)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	if (num < 0 || tar->filenum <= num) {
		return ARCHIVE_ERROR_NO_EXIST_FILE;
	}
	
	if (tar->openNumber != -1) {
		return ARCHIVE_ERROR_ALREADY_OPEN;
	}
	
	fseek(tar->fp, tar->chain[num].seek, SEEK_SET);
	tar->openNumber = num;
	tar->nowSeek = 0L;
	tar->nowSize = tar->chain[num].size;
	tar->seekInArchive = tar->chain[num].seek;
	
	return ARCHIVE_SUCCESS;
}


int Tar_FileNumber(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return tar->filenum;
}


int Tar_GetChar(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	if (tar->nowSeek >= tar->nowSize) {
		return EOF;
	}
	tar->nowSeek++;
	return fgetc(tar->fp);
}


long Tar_Size(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return tar->nowSize;
}


long Tar_Tell(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	return tar->nowSeek;
}


int Tar_EOF(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	if (tar->nowSeek >= tar->nowSize) {
		return 1;
	} else {
		return 0;
	}
}


int Tar_Close(SDL_Archive *archive)
{
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	tar->openNumber = -1;
	tar->seekInArchive = 0L;
	tar->nowSize = 0L;
	tar->nowSeek = 0L;
	
	return ARCHIVE_SUCCESS;
}


int Tar_Read(SDL_Archive *archive, void *mem, const int size, const int maxnum)
{
	int count;
	long position;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	count = 0;
	position = 0L;
	while (count < maxnum && tar->nowSeek + position < tar->nowSize) {
		fread(((char*)mem) + position, size, 1, tar->fp);
		position += size;
		count++;
	}
	
	tar->nowSeek += size * count;
	
	return count;
}


int Tar_Seek(SDL_Archive *archive, const long offset, const int whence)
{
	long position;
	TAR_DATA *tar;
	
	tar = (TAR_DATA*)archive->data;
	
	switch (whence) {
	case SEEK_SET:
		position = offset;
		break;
	case SEEK_CUR:
		position = tar->nowSeek + offset;
		break;
	case SEEK_END:
		position = tar->nowSize + offset;
		break;
	default:
		return ARCHIVE_ERROR_WHENCE;
	}
	
	if (position > tar->nowSize) {
		position = tar->nowSize;
	}
	if (position < 0) {
		position = 0;
	}
	
	fseek(tar->fp, tar->seekInArchive + position, SEEK_SET);
	
	tar->nowSeek = position;
	
	return ARCHIVE_SUCCESS;
}
